/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.RadialWipe", category=Categories.TRANSITION)
public class RadialWipe {

	public enum TypeOfRadialWipe { CLOCKWISE, COUNTERCLOCKWISE, BOTH }


	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property("0")
	private IAnimatableDouble startAngle;

	@Property
	private IAnimatableVec2d wipeCenter;

	@Property
	private IAnimatableEnum<TypeOfRadialWipe> typeOfWipe;

	@Property(min="0", max="500")
	private IAnimatableDouble feather;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IBlendSupport blendSupport;

	private final IShaderProgram stencilProgram1;

	private final IShaderProgram stencilProgram2;

	@Inject
	public RadialWipe(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IBlendSupport blendSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.blendSupport = blendSupport;
		stencilProgram1 = shaders.getProgram(RadialWipe.class, "STENCIL_1");
		stencilProgram2 = shaders.getProgram(RadialWipe.class, "STENCIL_2");
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion)/100;
		if (completion == 1) {
			IVideoBuffer buffer = context.createVideoBuffer(context.getPreviousBounds());
			buffer.clear();
			return buffer;
		}

		IVideoBuffer inputBuffer = context.doPreviousEffect();

		if (completion == 0) {
			return inputBuffer;
		}

		VideoBounds inputBounds = inputBuffer.getBounds();
		if (inputBounds.isEmpty()) {
			return inputBuffer;
		}


		double startAngle = context.value(this.startAngle);
		Vec2d wipeCenter = context.value(this.wipeCenter);
		TypeOfRadialWipe typeOfWipe = context.value(this.typeOfWipe);
		double feather = context.value(this.feather);

		Resolution resolution = context.getVideoResolution();
		wipeCenter = resolution.scale(wipeCenter);
		feather = resolution.scale(feather);

		// 時計回り／半時計回りどちらの場合でも startAngle < endAngle となるようにする。
		double endAngle;
		switch (typeOfWipe) {
			case COUNTERCLOCKWISE:
				startAngle -= 360*completion;
				break;
			case BOTH:
				startAngle -= 180*completion;
				break;
		}
		startAngle %= 360;
		if (startAngle < 0) startAngle += 360;
		endAngle = startAngle + 360*completion;


		IVideoBuffer stencilBuffer = null;
		try {
			stencilBuffer = createStencilBuffer(inputBounds, wipeCenter, feather, startAngle, endAngle);
			return blendSupport.blend(stencilBuffer, inputBuffer, null, BlendMode.STENCIL_ALPHA, 1.0);

		} finally {
			if (stencilBuffer != null) stencilBuffer.dispose();
			if (inputBuffer != null) inputBuffer.dispose();
		}
	}

	private IVideoBuffer createStencilBuffer(
			VideoBounds inputBounds, Vec2d wipeCenter, double feather,
			double startAngle, double endAngle) {

		int stencilExpand = (int)Math.ceil(feather);
		VideoBounds stencilBounds = new VideoBounds(
				inputBounds.x - stencilExpand, inputBounds.y - stencilExpand,
				inputBounds.width + stencilExpand*2, inputBounds.height + stencilExpand*2);

		wipeCenter = new Vec2d(wipeCenter.x - stencilBounds.x, wipeCenter.y - stencilBounds.y);


		double angleDiff = endAngle - startAngle;
		double halflineAngle = (startAngle + angleDiff * 0.5 + 90) % 360;

		endAngle %= 360;

		double a1 = (startAngle > 90 && startAngle <= 270) ? 1 : -1;
		double a2 = (endAngle > 90 && endAngle <= 270) ? -1 : 1;
		double a3 = (halflineAngle > 90 && halflineAngle <= 270) ? -1 : 1;
		double t1 = Math.tan(Math.toRadians(startAngle));
		double t2 = Math.tan(Math.toRadians(endAngle));
		double t3 = Math.tan(Math.toRadians(halflineAngle));

		a1 /= Math.sqrt(a1*a1*(1+t1*t1));
		a2 /= Math.sqrt(a2*a2*(1+t2*t2));
		a3 /= Math.sqrt(a3*a3*(1+t3*t3));

		float[] line1 = new float[] { (float)a1, (float)(a1*t1), (float)(a1*(-wipeCenter.x-wipeCenter.y*t1)) };
		float[] line2 = new float[] { (float)a2, (float)(a2*t2), (float)(a2*(-wipeCenter.x-wipeCenter.y*t2)) };
		float[] line3 = new float[] { (float)a3, (float)(a3*t3), (float)(a3*(-wipeCenter.x-wipeCenter.y*t3)) };

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("line1", 3, FloatBuffer.wrap(line1)));
		uniforms.add(new GLUniformData("line2", 3, FloatBuffer.wrap(line2)));
		uniforms.add(new GLUniformData("line3", 3, FloatBuffer.wrap(line3)));

		IVideoBuffer buffer = context.createVideoBuffer(stencilBounds);
		try {
			IShaderProgram program = (angleDiff <= 180) ? stencilProgram1 : stencilProgram2;
			support.useShaderProgram(program, uniforms, buffer);

			if (feather == 0) {
				IVideoBuffer result = buffer;
				buffer = null;
				return result;
			} else {
				return blurSupport.gaussianBlur(buffer, feather, BlurDimensions.BOTH, true, true);
			}
		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	@ShaderSource
	public static final String[] STENCIL_1 = {
		"uniform vec3 line1;",
		"uniform vec3 line2;",
		"uniform vec3 line3;",
		"",
		"void main(void)",
		"{",
		"	vec3 pt = vec3(gl_FragCoord.xy, 1.0);",
		"	if (dot(line3, pt) > 0.5) {",
		"		gl_FragColor = vec4(1.0);",
		"	} else {",
		"		float d1 = clamp(dot(line1, pt) + 0.5, 0.0, 1.0);",
		"		float d2 = clamp(dot(line2, pt) + 0.5, 0.0, 1.0);",
		"		gl_FragColor = vec4(max(d1, d2));",
		"	}",
		"}"
	};

	@ShaderSource
	public static final String[] STENCIL_2 = {
		"uniform vec3 line1;",
		"uniform vec3 line2;",
		"uniform vec3 line3;",
		"",
		"void main(void)",
		"{",
		"	vec3 pt = vec3(gl_FragCoord.xy, 1.0);",
		"	if (dot(line3, pt) < -0.5) {",
		"		gl_FragColor = vec4(0.0);",
		"	} else {",
		"		float d1 = clamp(dot(line1, pt) + 0.5, 0.0, 1.0);",
		"		float d2 = clamp(dot(line2, pt) + 0.5, 0.0, 1.0);",
		"		gl_FragColor = vec4(min(d1, d2));",
		"	}",
		"}"
	};

}
