/*
 * @(#)ReadConfig.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.conf;


import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.NotExistException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.io.FileAccessException;
import org.maachang.commons.io.IOCom;
import org.maachang.commons.io.ReadEnterIndex;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.read.ReadElement;
import org.maachang.commons.util.read.ReadIni;


/**
 * コンフィグ情報読み込み.
 * <BR><BR>
 * 指定されたコンフィグ情報(INIファイル)を読み込みます.
 *
 * @version 1.00, 2004/06/06
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class ReadConfig implements CommonConfig
{
    
    /**
     * プロパティ情報格納オブジェクト.
     */
    private ReadIni m_conf = null ;
    
    /**
     * 読み込み対象ファイル名.
     */
    private String m_filename = null ;
    
    /**
     * 読み込みキャラクタセット.
     */
    private String m_chrset = null ;
    
    
    /**
     * 同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public ReadConfig()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param name 読み込み対象のコンフィグファイルを設定します.<BR>
     *             設定可能なプロトコルタイプは[file],[Local]です.
     * @exception InputException 引数例外.
     * @exception ReadSyntaxException 構文例外.
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void create( String name )
        throws InputException,ReadSyntaxException,FileAccessException
    {
        
        try{
            
            this.clear() ;
            m_sync.create() ;
            
            this.newInit( name ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        try{
            m_conf.clear() ;
        }catch( Exception t ){
        }
        
        m_conf = null ;
        m_chrset = null ;
        m_filename = null ;
        m_sync.clear() ;
    }
    
    /**
     * 情報の取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param section 取得対象のセクション名を指定します.
     * @param key 取得対象のキー名を設定します.
     * @param no 取得対象の項番を設定します.
     * @return ReadElement 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public final ReadElement get( String section,String key,int no )
        throws InputException
    {
        ReadElement ret = null ;
        
        if( section == null || key == null || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_conf.getElements( section,key ) > no ){
                    ret = new ReadElement() ;
                    ret.add( m_conf.get( section,key,no ) ) ;
                }
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報の取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param out 取得された内容が返されます.
     * @param section 取得対象のセクション名を指定します.
     * @param key 取得対象のキー名を設定します.
     * @param no 取得対象の項番を設定します.
     * @exception InputException 入力例外.
     */
    public final void get( ReadElement out,String section,String key,int no )
        throws InputException
    {
        if(
            out == null || section == null ||
            key == null || no < 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            out.clear() ;
            
            synchronized( m_sync.get() ){
                
                if( m_conf.getElements( section,key ) > no ){
                    out.add( m_conf.get( section,key,no ) ) ;
                }
            }
            
        }catch( NullPointerException nul ){
            out.clear() ;
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            out.clear() ;
        }
        
    }
    
    /**
     * 情報の取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param section 取得対象のセクション名を指定します.
     * @param key 取得対象のキー名を設定します.
     * @return ReadElement 取得された内容が返されます.
     * @exception InputException 入力例外.
     */
    public final ReadElement get( String section,String key )
        throws InputException
    {
        int i ;
        int len ;
        ReadElement ret = null ;
        
        if( section == null || key == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                
                len = m_conf.getElements( section,key ) ;
                ret = new ReadElement() ;
                
                for( i = 0 ; i < len ; i ++ ){
                    ret.add( m_conf.get( section,key,i ) ) ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報の取得.
     * <BR><BR>
     * 対象の情報を取得します.
     * <BR>
     * @param out 取得された内容が返されます.
     * @param section 取得対象のセクション名を指定します.
     * @param key 取得対象のキー名を設定します.
     * @exception InputException 入力例外.
     */
    public final void get( ReadElement out,String section,String key )
        throws InputException
    {
        int i ;
        int len ;
        
        if( out == null || section == null || key == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            out.clear() ;
            
            synchronized( m_sync.get() ){
                
                len = m_conf.getElements( section,key ) ;
                
                for( i = 0 ; i < len ; i ++ ){
                    out.add( m_conf.get( section,key,i ) ) ;
                }
                
            }
            
        }catch( NullPointerException nul ){
            out.clear() ;
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            out.clear() ;
        }
        
    }
    
    /**
     * 対象キー名に対する情報数を取得.
     * <BR><BR>
     * 対象キー名に対する情報数を取得します.
     * <BR>
     * @param section 対象のセクション名を指定します.
     * @param key 対象のキー名を指定します.
     * @return int 格納情報数が返されます.
     * @exception InputException 入力例外.
     */
    public final int getElement( String section,String key )
        throws InputException
    {
        int ret ;
        
        if( section == null || key == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf.getElements( section,key ) ;
            }
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( NotExistException ne ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名に対するキー名群を取得.
     * <BR><BR>
     * 対象セクション名に対するキー名群を取得します.
     * <BR>
     * @param section 対象のセクション名を指定します.
     * @return String[] 対象のキー名群が返されます.<BR>
     *                  情報が存在しない場合[null]が返されます.
     * @exception InputException 入力例外.
     */
    public final String[] getNames( String section )
        throws InputException
    {
        String[] ret = null ;
        
        if( section == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_conf.getKeywords( section ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = null ;
        }catch( NotExistException ne ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名に対するキー数を取得.
     * <BR><BR>
     * 対象セクション名に対するキー数を取得します.
     * <BR>
     * @param section 対象のセクション名を指定します.
     * @return int 対象のキー名群が返されます.
     * @exception InputException 入力例外.
     */
    public final int getElement( String section )
        throws InputException
    {
        
        int ret ;
        
        if( section == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_conf.getKeywordElements( section ) ;
            }
            
        }catch( NullPointerException nul ){
            ret = 0 ;
        }catch( NotExistException ne ){
            ret = 0 ;
        }
        
        return ret ;
        
    }
    
    /**
     * セクション名群の取得.
     * <BR><BR>
     * セクション名群を取得します.
     * <BR>
     * @return String[] セクション名群が返されます.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getNames()
    {
        
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf.getSections() ;
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * セクション数の取得.
     * <BR><BR>
     * セクション名群を取得します.
     * <BR>
     * @return int セクション名数が返されます.<BR>
     *             情報が存在しない場合[0]が返されます.
     */
    public final int getElement()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf.getSectionElements() ;
            }
        }catch( NullPointerException nul ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 読み込みキャラクタセットを取得.
     * <BR><BR>
     * 読み込み対象のキャラクタセット情報が返されます.
     * <BR>
     * @return String 読み込みキャラクタセットが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final String getCharset()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_chrset ;
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 読み込みファイル名を取得.
     * <BR><BR>
     * 読み込まれた対象ファイル名が返されます.
     * <BR>
     * @return String ファイル名が返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public final String getFilename()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_filename ;
            }
        }catch( NullPointerException nul ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象セクション名存在チェック.
     * <BR><BR>
     * 対象セクション名存在チェックを行います.
     * <BR>
     * @param name チェック対象のセクション名を指定します.
     * @return boolean 情報存在結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isSection( String name )
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf.isSection( name ) ;
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 対象キー名存在チェック.
     * <BR><BR>
     * 対象キー名存在チェックを行います.
     * <BR>
     * @param section 対象のセクション名を指定します.
     * @param name チェック対象のキー名を指定します.
     * @return boolean 情報存在結果が返されます.<BR>
     *                 [true]が返された場合情報は存在します.<BR>
     *                 [false]が返された場合情報は存在しません.
     */
    public final boolean isKey( String section,String name )
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_conf.isKeyword( section,name ) ;
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 指定ファイルから情報を取得.
     */
    private final void newInit( String file )
        throws InputException,ReadSyntaxException,FileAccessException
    {
        
        String chrset = null ;
        ReadEnterIndex index = null ;
        
        if( file == null || file.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            file = UtilCom.trimPlus( file ) ;
            
            synchronized( m_sync.get() ){
                
                index = new ReadEnterIndex() ;
                chrset = IOCom.getValue( file,BaseDef.INI_CHARSET ) ;
                
                if( chrset == null || chrset.length() <= 0 ){
                    
                    index.open( file ) ;
                    m_chrset = BaseDef.THIS_CHARSET ;
                    
                }
                else{
                    
                    index.open( file,chrset ) ;
                    m_chrset = chrset ;
                    
                }
                
                if( m_conf != null ){
                    
                    try{
                        m_conf.clear() ;
                    }catch( Exception t ){
                    }
                    
                }
                
                m_conf = new ReadIni() ;
                m_conf.create( index ) ;
                m_filename = file ;
                
            }
            
        }catch( NullPointerException nul ){
            this.clear() ;
        }catch( InputException in ){
            this.clear() ;
            throw new ReadSyntaxException( in ) ;
        }catch( FileAccessException fa ){
            this.clear() ;
            throw fa ;
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }finally{
            
            try{
                index.close() ;
            }catch( Exception t ){
            }
            
            index = null ;
            chrset = null ;
            
        }
        
    }
    
}
