/*
 * @(#)StackTraceCore.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.exception;

import org.maachang.commons.util.array.ObjectArray;

/**
 * スタックトレース情報格納処理.
 *  
 * @version 1.0.0 2003/11/05
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class StackTraceCore implements StackTrace
{
    
    /**
     * スタックトレース格納データ.
     */
    class InTrace
    {
        
        public String m_package = null ;
        public String m_object = null ;
        public String m_method = null ;
        public String m_line = null ;
        
        public final void clear()
        {
            m_package = null ;
            m_object = null ;
            m_method = null ;
            m_line = null ;
        }
        
    }
    
    /**
     * 格納データ.
     */
    private final ObjectArray m_list = new ObjectArray() ;
    
    /**
     * コンストラクタ.
     */
    public StackTraceCore()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 対象の情報をクリアします.
     */
    public final void clear()
    {
        int i ;
        int len ;
        
        InTrace trace = null ;
        ObjectArray list = null ;
        
        list = m_list ;
        len = m_list.size() ;
        
        for( i = 0 ; i < len ; i ++ ){
            
            trace = ( InTrace )list.get( i ) ;
            trace.clear() ;
            trace = null ;
            
        }
        
        list.clear() ;
        list = null ;
    }
    
    /**
     * スタックトレース情報追加.
     * <BR><BR>
     * 対象のスタックトレース情報を追加します.
     * <BR>
     * @param pack 対象のパッケージ情報を設定します.
     * @param obj 対象のオブジェクト情報を設定します.
     * @param met 対象のメソッド情報を設定します.
     * @param lin 対象のライン情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void add( String pack,String obj,String met,String lin )
        throws InputException
    {
        
        InTrace trace = null ;
        
        if( pack == null || obj == null || met == null || lin == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        trace = new InTrace() ;
        trace.m_package = pack ;
        trace.m_object = obj ;
        trace.m_method = met ;
        trace.m_line = lin ;
        
        m_list.add( trace ) ;
        trace = null ;
        
    }
    
    /**
     * 格納パッケージ情報の取得.
     * <BR><BR>
     * 格納されているパッケージ情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 格納されている情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String getPackage( int no )
        throws InputException
    {
        
        String ret = null ;
        
        try{
            
            ret = ( ( InTrace )m_list.get( no ) ).m_package ;
            
        }catch( Exception t ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 格納オブジェクト情報の取得.
     * <BR><BR>
     * 格納されているオブジェクト情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 格納されている情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String getObject( int no )
        throws InputException
    {
        
        String ret = null ;
        
        try{
            
            ret = ( ( InTrace )m_list.get( no ) ).m_object ;
            
        }catch( Exception t ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 格納メソッド情報の取得.
     * <BR><BR>
     * 格納されているメソッド情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 格納されている情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String getMethod( int no )
        throws InputException
    {
        
        String ret = null ;
        
        try{
            
            ret = ( ( InTrace )m_list.get( no ) ).m_method ;
            
        }catch( Exception t ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 格納ライン情報の取得.
     * <BR><BR>
     * 格納されているライン情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 格納されている情報が返されます.
     * @exception InputException 入力例外.
     */
    public final String getLine( int no )
        throws InputException
    {
        
        String ret = null ;
        
        try{
            
            ret = ( ( InTrace )m_list.get( no ) ).m_line ;
            
        }catch( Exception t ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * 格納データ数を取得.
     * <BR><BR>
     * 格納されているデータ数を取得します.
     * <BR>
     * @return int 格納されているデータ数が返されます.
     */
    public final int size()
    {
        return m_list.size() ;
    }
    
}
