/*
 * @(#)ByteBufferWriter.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.io;

import java.io.BufferedOutputStream;
import java.io.IOException;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.io.env.EnvFileOutputStream;
import org.maachang.commons.thread.Synchronized;



/**
 * 指定情報を出力します.
 * <BR><BR>
 * 指定された情報に対して、出力します.
 *
 * @version 1.0.0 2003/11/03
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class ByteBufferWriter implements BaseWriter
{
    
    /**
     * ファイルポインタ.
     */
    private BufferedOutputStream m_fp = null ;
    
    /**
     * オープンファイル名.
     */
    private String m_fileName = null ;
    
    /**
     * キャラクターセット.
     */
    private String m_charset = null ;
    
    /**
     * 書き込みデータ長.
     */
    private long m_length = -1L ;
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ByteBufferWriter()
    {
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 出力対象ファイルオープン.
     * <BR><BR>
     * 出力対象のファイルをオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param newFlg 新規に書き込みを行うか設定します。<BR>
     *               [true]を指定した場合、指定ファイル内の情報をクリアします。<BR>
     *               [false]を指定した場合、すでにファイル情報が存在する場合追加書き込みきます。
     * @exception FileAccessException ファイルIO例外
     * @exception InputException 入力例外
     */
    public final void open( String fileName,boolean newFlg )
        throws FileAccessException,InputException
    {
        
        try{
            this.open( fileName,null,newFlg ) ;
        }catch( InputException in ){
            throw in ;
        }catch( FileAccessException fa ){
            throw fa ;
        }
        
    }
    
    /**
     * 出力対象ファイルオープン.
     * <BR><BR>
     * 出力対象のファイルをオープンします.
     * <BR>
     * @param fileName オープン対象のファイル名を指定します.
     * @param charset 出力対象のエンコード名を設定します.
     * @param newFlg 新規に書き込みを行うか設定します。<BR>
     *               [true]を指定した場合、指定ファイル内の情報をクリアします。<BR>
     *               [false]を指定した場合、すでにファイル情報が存在する場合追加書き込みきます。
     * @exception FileAccessException ファイルIO例外
     * @exception InputException 入力例外
     */
    public final void open( String fileName,String charset,boolean newFlg )
        throws FileAccessException,InputException
    {
        if( fileName == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
        try{
            
            // flip-newFlg.
            newFlg = ( newFlg == true ) ? false : true ;
            
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                
                m_fp = new BufferedOutputStream(
                    new EnvFileOutputStream( fileName,newFlg )
                ) ;
                
                m_fileName = fileName ;
                m_charset = ( charset == null || charset.length() <= 0 ) ?
                    BaseDef.THIS_CHARSET : charset ;
                m_length = ( newFlg == true ) ? 0L : IOCom.getFileLength( fileName ) ;
                
            }
            
        }catch( IOException io ){
            this.close() ;
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
            this.close() ;
        }
        
    }
    
    /**
     * ファイルクローズ.
     * <BR><BR>
     * オープンしているファイル情報をクローズします.
     * また、バッファに対して情報が存在する場合、その情報を
     * 書き込んでクローズします.
     */
    public final void close()
    {
        
        try{
            synchronized( m_sync.get() ){
                m_fp.close() ;
            }
        }catch( Exception e ){
        }finally{
            try{m_fp.close();}catch( Exception ee ){}
            m_fp = null ;
            m_fileName = null ;
            m_charset = null ;
            m_length = -1L ;
            m_sync.clear() ;
        }
        
        
    }
    
    /**
     * バッファ情報を出力.
     * <BR><BR>
     * バッファ情報をファイルに出力します。
     * <BR>
     * @exception FileAccessException ファイルIO例外
     */
    public final void flush() throws FileAccessException
    {
        
        try{
            synchronized( m_sync.get() ){
                m_fp.flush() ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 文字情報を出力.
     * <BR><BR>
     * 文字情報を出力します。
     * <BR>
     * @param stream 文字情報を設定します。
     * @exception InputException 入力例外
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void print( String stream )
        throws InputException,FileAccessException
    {
        byte[] b = null ;
        
        if( stream == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                b = stream.getBytes( m_charset ) ;
                m_length += b.length ;
                m_fp.write( b ) ;
            }
        }catch( IOException io ){
            throw new FileAccessException( io ) ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * １行の文字情報を出力.
     * <BR><BR>
     * １行の文字情報を出力します。
     * <BR>
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void println()
        throws FileAccessException
    {
        try{
            this.print( BaseDef.ENTER_SEPARATOR ) ;
        }catch( FileAccessException fa ){
            throw fa ;
        }catch( Exception e ){
        }
    }
    
    /**
     * １行の文字情報を出力.
     * <BR><BR>
     * １行の文字情報を出力します。
     * <BR>
     * @param stream 文字情報を設定します。
     * @exception InputException 入力例外
     * @exception FileAccessException ファイルアクセス例外.
     */
    public final void println( String stream )
        throws InputException,FileAccessException
    {
        
        StringBuffer buf = null ;
        
        if( stream == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer( stream.length() + 2 ) ;
        buf.append( stream ) ;
        buf.append( BaseDef.ENTER_SEPARATOR ) ;
        
        try{
            this.print( buf.toString() ) ;
        }catch( FileAccessException fa ){
            throw fa ;
        }finally{
            buf = null ;
        }
        
    }
    
    /**
     * 書き込み容量の取得.
     * <BR><BR>
     * 現在のファイル書き込み容量を取得します。
     * また、現在バッファに格納されている情報数も数に含まれます.
     * <BR>
     * @return long ファイル書き込み容量を取得します。<BR>
     *              ファイルがオープンされていない場合[-1]が返されます.
     */
    public final long length()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fileName == null ) ? -1L : m_length ;
            }
        }catch( Exception t ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * オープンファイル名の取得.
     * <BR><BR>
     * 現在オープンされているファイル名の取得を行います。
     * <BR>
     * @return String 現在オープンされているファイル名を取得します。<BR>
     *   オープンされていない場合[null]が格納されます。
     */
    public final String getFileName()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_fileName ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在状況を取得します。
     * <BR><BR>
     * 現在のファイルオープン状況を取得します。
     * <BR>
     * @return boolean 現在のファイルオープン状況が格納されています。<BR>
     *                 [true]が返された場合、ファイルはオープンされています。<BR>
     *                 [false]が返された場合、ファイルはオープンされていません。
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_fileName == null ) ? false : true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
}
