/*
 * @(#)BaseUdpProtocol.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;


/**
 * UDPプロトコルインターフェイス.
 * <BR><BR>
 * UDPプロトコルで処理するインターフェイスです.
 *
 * @version 1.00, 2004/10/19
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public interface BaseUdpProtocol
{
    /**
     * 受信バッファ長.
     * <BR><BR>
     * 受信時のバッファ長です.
     */
    public static final int BUF_LENGTH = 65535 ;
    
    /**
     * バッファモード設定 : デフォルト値設定.
     */
    public static final int NOT_MODE = 0x00000000 ;
    
    /**
     * バッファモード設定 : 送信側のみバッファ設定.
     */
    public static final int SND_MODE = 0x00000001 ;
    
    /**
     * バッファモード設定 : 受信側のみバッファ設定.
     */
    public static final int RCV_MODE = 0x00000002 ;
    
    /**
     * バッファモード設定 : 両方バッファ設定.
     */
    public static final int ALL_MODE = SND_MODE | RCV_MODE ;
    
    
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @exception NotBindException バインド失敗.
     */
    public void open( int bufLen )
        throws NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象のポート番号をオープンします.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void open( int bufLen,int port )
        throws InputException,NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象の自IPアドレス上にポート番号でオープンします.<BR>
     * この処理の場合、対象のアドレスはDHCPから割り振られたアドレスで無い事が
     * 条件となります.
     * <BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @param addr オープン対象の自IPアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void open( int bufLen,int port,InetAddress addr )
        throws InputException,NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * ポート番号をデフォルトポート番号で指定して、オープンします.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @exception NotBindException バインド失敗.
     */
    public void openTo( int bufMode,int bufLen )
        throws NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象のポート番号をオープンします.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void openTo( int bufMode,int bufLen,int port )
        throws InputException,NotBindException ;
    
    /**
     * オープン処理.
     * <BR><BR>
     * 対象の自IPアドレス上にポート番号でオープンします.<BR>
     * この処理の場合、対象のアドレスはDHCPから割り振られたアドレスで無い事が
     * 条件となります.
     * <BR>
     * @param bufMode バッファモードを設定します.<BR>
     *                [UdpProtocol#NOT_MODE]を指定した場合、デフォルト値の
     *                バッファ長を設定します.<BR>
     *                [UdpProtocol#SND_MODE]を指定した場合、送信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#RCV_MODE]を指定した場合、受信側バッファのみ
     *                バッファ値を設定します.<BR>
     *                [UdpProtocol#ALL_MODE]を指定した場合、送信/受信側バッファに
     *                バッファ値を設定します.<BR>
     * @param bufLen 送受信バッファ長を設定します.
     * @param port オープン対象のポート番号を設定します.
     * @param addr オープン対象の自IPアドレスを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド失敗.
     */
    public void openTo( int bufMode,int bufLen,int port,InetAddress addr )
        throws InputException,NotBindException ;
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を行います.
     */
    public void close() ;
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param addr 送信対象のIPアドレス/ポート番号を指定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] binary,ConnectAddress addr )
        throws InputException,UndefineBindException ;
    
    /**
     * データ送信.
     * <BR><BR>
     * 対象のデータを送信します.
     * <BR>
     * @param binary 送信対象のデータを設定します.
     * @param addr 送信対象のIPアドレスを指定します.
     * @param port 送信対象のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public void send( byte[] binary,InetAddress addr,int port )
        throws InputException,UndefineBindException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public byte[] receive( ConnectAddress addr )
        throws UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.<BR>
     * データが存在しない場合[null]が返されます.
     * <BR>
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.<BR>
     *                受信対象の情報が存在しない場合[null]が返されます.
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public byte[] receive( ConnectAddress addr,int timeout )
        throws UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public int receive( byte[] out,ConnectAddress addr )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * データ受信.
     * <BR><BR>
     * データを受信します.
     * <BR>
     * @param out 受信されたバイナリ情報が設定されます.
     * @param addr 受信先のIPアドレスとポート番号が格納された
     *             内容が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @return int 受信されたバイナリ情報長が返されます.
     * @exception InputException 入力例外
     * @exception UndefineBindException バインド未定義例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public int receive( byte[] out,ConnectAddress addr,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public void getLocal( ConnectAddress addr ) ;
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @return ConnectAddress 対象のローカルアドレスが返されます.
     */
    public ConnectAddress getLocal() ;
    
    /**
     * ローカルアドレス情報を取得.
     * <BR><BR>
     * ローカルアドレス情報を取得します.
     * <BR>
     * @return InetAddress ローカルアドレス情報が返されます.
     */
    public InetAddress getLocalAddress() ;
    
    /**
     * ローカルポート番号を取得.
     * <BR><BR>
     * ローカルポート番号を取得します.
     * <BR>
     * @return int ローカルポート番号が返されます.
     */
    public int getLocalPort() ;
    
    /**
     * 設定バッファ長を取得.
     * <BR><BR>
     * 設定されているバッファ長を取得します.
     * <BR>
     * @return int 設定バッファ長が返されます.
     */
    public int getBuffer() ;
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープン状態が返されます.<BR>
     *                 [true]が返された場合、ソケットはオープンされています.<BR>
     *                 [false]が返された場合、ソケットはオープンされていません.
     */
    public boolean isOpen() ;
    
}

