/*
 * @(#)MultiCastUtil.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;


/**
 * マルチキャストユーティリティ.
 *  
 * @version 1.0.0 2005/02/05
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class MultiCastUtil
{
    
    /**
     * コンストラクタ.
     */
    private MultiCastUtil()
    {
    }
    
    /**
     * マルチキャストIPV4 : 予約.
     */
    public static final String RESERVATION_ADDR_V4 = "224.0.0.0" ;
    
    /**
     * マルチキャストIPV4範囲.
     */
    public static final int MULTICAST_V4 = 0x0fffffff ;
    
    /**
     * マルチキャストIPV4予約範囲.
     */
    public static final int MULTICAST_RES_V4 = 0x000000ff ;
    
    /**
     * マルチキャストIPV4開始範囲.
     */
    public static final int MULTICAST_START_V4 = 1 ;
    
    /**
     * マルチキャストIPV4終了範囲.
     */
    //public static final int MULTICAST_END_V4 = MULTICAST_V4 - MULTICAST_RES_V4 ;
    public static final int MULTICAST_END_V4 = 0x0effffff ;
    
    /**
     * マルチキャストTTL(TimeToLive) : 最小値.
     */
    public static final int MIN_TTL = 1 ;
    
    /**
     * マルチキャストTTL(TimeToLive) : 最大値.
     */
    public static final int MAX_TTL = 128 ;
    
    
    
    /**
     * マルチキャストIPV4接頭語.
     */
    private static final int MIN_MULTICAST_HEADER_V4 = 224 ;
    
    /**
     * マルチキャストIPV4接頭語範囲最大値.
     */
    private static final int MAX_MULTICAST_HEADER_V4 = 239 ;
    
    
    
    /**
     * アドレス区切り.
     */
    private static final String MULTICAST_ADDR_V4 = "." ;
    
    /**
     * 対象IDから有効マルチキャストアドレス(IPV4)を生成.
     * <BR><BR>
     * 対象IDから有効マルチキャスト(IPV4)を生成します.
     * <BR>
     * @param id 対象のID情報を設定します.<BR>
     *           設定可能な範囲は[1]から[251658239]です.
     * @return InetAddress 変換されたマルチキャストアドレス(IPV4)が返されます.
     * @exception InputException 入力例外.
     */
    public static final InetAddress getIDByMulitCast_V4( int id )
        throws InputException
    {
        StringBuffer buf = null ;
        InetAddress ret = null ;
        
        if(
            id < MultiCastUtil.MULTICAST_START_V4 ||
            id > MultiCastUtil.MULTICAST_END_V4
        )
        {
            throw new InputException( "対象マルチキャストは範囲外です" ) ;
        }
        
        try{
            
            id += MULTICAST_RES_V4 ;
            
            buf = new StringBuffer() ;
            buf.append(
                (
                    MultiCastUtil.MIN_MULTICAST_HEADER_V4 +
                    ( ( ( id & 0x0f000000 ) >> 24 ) & 0x000000ff )
                )
            ) ;
            buf.append( MultiCastUtil.MULTICAST_ADDR_V4 ) ;
            buf.append( ( ( ( id & 0x00ff0000 ) >> 16 ) & 0x000000ff ) ) ;
            buf.append( MultiCastUtil.MULTICAST_ADDR_V4 ) ;
            buf.append( ( ( ( id & 0x0000ff00 ) >> 8 ) & 0x000000ff ) ) ;
            buf.append( MultiCastUtil.MULTICAST_ADDR_V4 ) ;
            buf.append( ( id & 0x000000ff ) ) ;
            
            ret = InetAddress.getByName( buf.toString() ) ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象マルチキャストアドレス(IPV4)から、ID情報を取得.
     * <BR><BR>
     * 対象マルチキャストアドレス(IPV4)から、ID情報を取得します.
     * <BR>
     * @param addr 対象のアドレス(IPV4)を設定します.
     * @return int 対象のID情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final int getMultiCastByID_V4( InetAddress addr )
        throws InputException
    {
        int hd ;
        int ret ;
        byte[] tmp = null ;
        
        if( addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            tmp = addr.getAddress() ;
            hd = ( int )( tmp[ 0 ] & 0x000000ff ) ;
            
            if(
                hd < MultiCastUtil.MIN_MULTICAST_HEADER_V4 ||
                hd > MultiCastUtil.MAX_MULTICAST_HEADER_V4
            )
            {
                throw new InputException(
                    "対象のアドレス(" +
                    addr.getHostAddress() +
                    ")はマルチキャストではありません"
                ) ;
            }
            
            ret = ( int )( ( hd - MultiCastUtil.MIN_MULTICAST_HEADER_V4 ) << 24 ) ;
            ret |= ( int )( ( tmp[ 1 ] & 0x000000ff ) << 16 ) ;
            ret |= ( int )( ( tmp[ 2 ] & 0x000000ff ) << 8 ) ;
            ret |= ( int )( tmp[ 3 ] & 0x000000ff ) ;
            
            ret -= MULTICAST_RES_V4 ;
            
        }catch( InputException in ){
            throw in ;
        }catch( Exception e ){
            throw new InputException( e ) ;
        }finally{
            tmp = null ;
        }
        
        return ret ;
    }
    
}

