/*
 * @(#)NetAdapterBase.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.sys.cmd.NativeCmd;
import org.maachang.commons.sys.cmd.NoExitProcessException;
import org.maachang.commons.util.ArrayTable;
import org.maachang.commons.util.CharTable;


/**
 * ネットワークアダプタ情報取得基本.
 * <BR><BR>
 * 基本となるネットワークアダプタ情報取得処理をサポートします.
 *  
 * @version 1.0.0 2004/10/13
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
abstract class NetAdapterBase implements NetAdapter
{
    
    /**
     * イーサネットアドレス変換媒体名.
     */
    public static final String LOCAL_ETHER_NAME = "eth" ;
    
    /**
     * その他変換媒体名.
     */
    public static final String ETC_ETHER_NAME = "wan" ;
    
    /**
     * 内部アドレス媒体名.
     */
    public static final String DEFAULT_ETHER_NAME = "lo" ;
    
    /**
     * 内部サブネットマスク.
     */
    public static final String LO_HOST_SUB = "255.0.0.0" ;
    
    
    
    /**
     * アドレス格納情報.
     */
    protected final CharTable m_table = new CharTable() ;
    
    /**
     * コンストラクタ.
     */
    public NetAdapterBase()
    {
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear()
    {
        m_table.clear() ;
    }
    
    /**
     * ネットワーク情報取得処理.
     * <BR><BR>
     * 対象のネットワーク情報を取得します.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public abstract void flush() throws AccessException ;
    
    /**
     * 対象カテゴリ情報を追加.
     * <BR><BR>
     * 対象のカテゴリ情報を追加します.
     * <BR>
     * @param name 対象の媒体名(LANの場合[eth0]など.)を設定します.
     * @param category 対象のカテゴリ名を設定します.<BR>
     *                 [NetAdapter.CATEGORY_IPADDR]を指定した場合、IPアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_SUBNET]を指定した場合、サブネットアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_MACADDR]を指定した場合、MACアドレス追加となります.
     * @param addr 対象のIPアドレス情報を設定します.
     * @exception InputException 入力例外.
     */
    protected final void addCategory( String name,String category,String addr )
        throws InputException
    {
        ArrayTable tbl = null ;
        
        if( name == null || category == null || addr == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            if( ( tbl = ( ArrayTable )m_table.get( name ) ) == null ){
                tbl = new ArrayTable() ;
                m_table.add( name,tbl ) ;
            }
            
            tbl.add( category,addr ) ;
            
        }catch( Exception t ){
            throw new InputException( t ) ;
        }finally{
            tbl = null ;
        }
    }
    
    /**
     * 対象カテゴリ情報数を取得.
     * <BR><BR>
     * 対象カテゴリの格納情報数を取得します.
     * <BR>
     * @param name 対象の媒体名(LANの場合[eth0]など.)を設定します.
     * @param category 対象のカテゴリ名を設定します.<BR>
     *                 [NetAdapter.CATEGORY_IPADDR]を指定した場合、IPアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_SUBNET]を指定した場合、サブネットアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_MACADDR]を指定した場合、MACアドレス追加となります.
     * @param no 対象の項番を設定します.
     * @return String カテゴリ内容が返されます.
     */
    public final String getCategory( String name,String category,int no )
    {
        String ret = null ;
        ArrayTable tbl = null ;
        
        if( name == null || category == null || no < 0 ){
            return null ;
        }
        
        try{
            
            if( ( tbl = ( ArrayTable )m_table.get( name ) ) == null ){
                ret = null ;
            }
            else{
                ret = ( String )tbl.get( category,no ) ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            tbl = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象カテゴリ情報数を取得.
     * <BR><BR>
     * 対象カテゴリの格納情報数を取得します.
     * <BR>
     * @param name 対象の媒体名(LANの場合[eth0]など.)を設定します.
     * @param category 対象のカテゴリ名を設定します.<BR>
     *                 [NetAdapter.CATEGORY_IPADDR]を指定した場合、IPアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_SUBNET]を指定した場合、サブネットアドレス追加となります.<BR>
     *                 [NetAdapter.CATEGORY_MACADDR]を指定した場合、MACアドレス追加となります.
     * @return int 情報長が返されます.
     */
    public final int getElement( String name,String category )
    {
        int ret ;
        
        ArrayTable tbl = null ;
        
        if( name == null || category == null ){
            return 0 ;
        }
        
        try{
            
            if( ( tbl = ( ArrayTable )m_table.get( name ) ) == null ){
                ret = 0 ;
            }
            else{
                ret = tbl.getElements( category ) ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }finally{
            tbl = null ;
        }
        
        return ret ;
    }
    
    /**
     * 媒体名群の取得.
     * <BR><BR>
     * 対象媒体名群を取得します.
     * <BR>
     * @return String[] 格納されている媒体名群を取得します.
     */
    public final String[] getNames()
    {
        String[] ret = null ;
        
        try{
            ret = m_table.getNames() ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトを文字列に変換.
     * <BR><BR>
     * オブジェクト情報を文字列に変換します.
     * <BR>
     * @return String 変換された文字列が返されます.
     */
    public final String toString()
    {
        int i,j ;
        int len ;
        int lenJ ;
        
        String[] code = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            
            code = this.getNames() ;
            len = code.length ;
            
            buf = new StringBuffer() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                buf.append( "Adapter :" ) ;
                buf.append( code[ i ] ) ;
                buf.append( BaseDef.ENTER_SEPARATOR ) ;
                
                if( ( lenJ = this.getElement( code[ i ],NetAdapter.CATEGORY_IPADDR ) ) != 0 ){
                    
                    if( lenJ == 1 ){
                        buf.append( "  IP Address     :" ) ;
                        buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_IPADDR,0 ) ) ;
                        buf.append( BaseDef.ENTER_SEPARATOR ) ;
                    }
                    else{
                        for( j = 0 ; j < lenJ ; j ++ ){
                            
                            buf.append( "  IP Address[ " ) ;
                            buf.append( (j+1) ) ;
                            buf.append( " ]     :" ) ;
                            buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_IPADDR,j ) ) ;
                            buf.append( BaseDef.ENTER_SEPARATOR ) ;
                            
                        }
                    }
                    
                }
                
                if( ( lenJ = this.getElement( code[ i ],NetAdapter.CATEGORY_SUBNET ) ) != 0 ){
                    
                    if( lenJ == 1 ){
                        buf.append( "  Sub NetAddress :" ) ;
                        buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_SUBNET,0 ) ) ;
                        buf.append( BaseDef.ENTER_SEPARATOR ) ;
                    }
                    else{
                        for( j = 0 ; j < lenJ ; j ++ ){
                            
                            buf.append( "  Sub NetAddress[ " ) ;
                            buf.append( (j+1) ) ;
                            buf.append( " ] :" ) ;
                            buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_SUBNET,j ) ) ;
                            buf.append( BaseDef.ENTER_SEPARATOR ) ;
                            
                        }
                    }
                    
                }
                
                if( ( lenJ = this.getElement( code[ i ],NetAdapter.CATEGORY_MACADDR ) ) != 0 ){
                    
                    if( lenJ == 1 ){
                        buf.append( "  Mac Address    :" ) ;
                        buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_MACADDR,0 ) ) ;
                        buf.append( BaseDef.ENTER_SEPARATOR ) ;
                    }
                    else{
                        for( j = 0 ; j < lenJ ; j ++ ){
                            
                            buf.append( "  Mac Address[ " ) ;
                            buf.append( (j+1) ) ;
                            buf.append( " ]    :" ) ;
                            buf.append( this.getCategory( code[ i ],NetAdapter.CATEGORY_MACADDR,j ) ) ;
                            buf.append( BaseDef.ENTER_SEPARATOR ) ;
                            
                        }
                    }
                    
                }
                
                code[ i ] = null ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception t ){
            ret = "" ;
        }finally{
            code = null ;
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド実行.
     */
    protected final String[] runExec( String cmd ) throws AccessException
    {
        String[] ret = null ;
        NativeCmd execCmd = null ;
        
        try{
            execCmd = new NativeCmd() ;
            ret = execCmd.execution( true,cmd ) ;
        }catch( NoExitProcessException ne ){
            throw ne ;
        }finally{
            if( execCmd != null ){
                execCmd.clear() ;
            }
        }
        
        return ret ;
    }
    
}

