/*
 * @(#)TcpProtocol.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.SocketException;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;


/**
 * TCP/IPプロトコル.
 * <BR><BR>
 * 対象のTCP/IP処理を行います.
 *
 * @version 1.00, 2004/10/05
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class TcpProtocol implements BaseTcpProtocol
{
    
    /**
     * 受信バッファサイズ.
     */
    private static final int RECV_BUFLEN = 4096 ;
    
    /**
     * バッファ長.
     */
    private int m_bufLen = 0 ;
    
    /**
     * 最終処理時間.
     */
    private long m_lastTime = 0L ;
    
    /**
     * TCP/IP用.
     */
    private Socket m_net = null ;
    
    /**
     * 送信用.
     */
    private BufferedOutputStream m_send = null ;
    
    /**
     * 受信用.
     */
    private BufferedInputStream m_recv = null ;
    
    
    /**
     * 同期オブジェクト.
     */
    protected final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public TcpProtocol()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ソケットオブジェクトを設定します.
     * <BR>
     * @param conn 接続が確立されたソケットオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public TcpProtocol( Socket conn )
        throws InputException
    {
        this( conn,NetDef.DEF_BUFLEN ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ソケットオブジェクトを設定します.
     * <BR>
     * @param conn 接続が確立されたソケットオブジェクトを設定します.
     * @param bufLen 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     */
    public TcpProtocol( Socket conn,int bufLen )
        throws InputException
    {
        if( conn == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.disconnect() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                m_net = conn ;
                this.connectTo( ( bufLen <= NetDef.DEF_BUFLEN ) ? NetDef.DEF_BUFLEN : bufLen ) ;
            }
            
        }catch( Exception t ){
            this.disconnect() ;
            throw new InputException( t ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.disconnect() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public final void connect( InetAddress addr,int port )
        throws InputException,NotConnectException
    {
        this.connect( addr,port,NetDef.DEF_BUFLEN ) ;
    }
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param loAddr バインド先のローカルアドレスを設定します.
     * @param loPort バインド先のローカルポートを設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public final void connect( InetAddress addr,int port,InetAddress loAddr,int loPort )
        throws InputException,NotBindException,NotConnectException
    {
        this.connect( addr,port,loAddr,loPort,NetDef.DEF_BUFLEN ) ;
    }
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param bufLen 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public final void connect( InetAddress addr,int port,int bufLen )
        throws InputException,NotConnectException
    {
        if( addr == null || port < 0 || port > 65535 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.disconnect() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                m_net = new Socket( addr,port ) ;
                this.connectTo( ( bufLen <= NetDef.DEF_BUFLEN ) ? NetDef.DEF_BUFLEN : bufLen ) ;
            }
            
        }catch( NullPointerException nul ){
            this.disconnect() ;
        }catch( IOException io ){
            this.disconnect() ;
            throw new NotConnectException( io ) ;
        }
        
    }
    
    /**
     * 接続処理.
     * <BR><BR>
     * 接続先を設定します.
     * <BR>
     * @param addr 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @param loAddr バインド先のローカルアドレスを設定します.
     * @param loPort バインド先のローカルポートを設定します.
     * @param bufLen 送受信バッファ長を設定します.
     * @exception InputException 入力例外.
     * @exception NotBindException バインド例外.
     * @exception NotConnectException 未コネクション例外.
     */
    public final void connect( InetAddress addr,int port,InetAddress loAddr,int loPort,int bufLen )
        throws InputException,NotBindException,NotConnectException
    {
        if(
            addr == null || port < 0 || port > 65535 ||
            loAddr == null || loPort < 0 || loPort > 65535
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.disconnect() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                m_net = new Socket( addr,port,loAddr,loPort ) ;
                this.connectTo( ( bufLen <= NetDef.DEF_BUFLEN ) ? NetDef.DEF_BUFLEN : bufLen ) ;
            }
            
        }catch( NullPointerException nul ){
            this.disconnect() ;
        }catch( SocketException so ){
            this.disconnect() ;
            throw new NotConnectException( so ) ;
        }catch( IOException io ){
            this.disconnect() ;
            throw new NotBindException( io ) ;
        }
        
    }
    
    /**
     * コネクション破棄.
     * <BR><BR>
     * コネクションを破棄します.
     */
    public final void disconnect()
    {
        m_sync.clear() ;
        
        try{
            m_send.close() ;
        }catch( Exception t ){
        }
        try{
            m_recv.close() ;
        }catch( Exception t ){
        }
        try{
            m_net.close() ;
        }catch( Exception t ){
        }
        
        m_send = null ;
        m_recv = null ;
        m_net = null ;
        m_bufLen = 0 ;
        m_lastTime = 0L ;
        
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] message )
        throws InputException,UndefineBindException
    {
        this.send( message,0,0 ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @param length 対象メッセージのデータ長を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] message,int length )
        throws InputException,UndefineBindException
    {
        this.send( message,0,length ) ;
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を設定します.
     * <BR>
     * @param message 送信対象のメッセージを設定します.
     * @param offset 対象メッセージのオフセット値を設定します.
     * @param length 対象メッセージのデータ長を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     */
    public final void send( byte[] message,int offset,int length )
        throws InputException,UndefineBindException
    {
        
        if( message == null || message.length <= 0 ){
            if( message == null ){
                throw new InputException( "引数は不正です" ) ;
            }
            else{
                return ;
            }
        }
        if( length == 0 ){
            length = message.length ;
        }
        
        try{
            
            m_send.write( message,offset,length ) ;
            m_send.flush() ;
            
            synchronized( m_sync.get() ){
                m_lastTime = System.currentTimeMillis() ;
            }
        }catch( NullPointerException nul ){
            this.disconnect() ;
            throw new UndefineBindException( "コネクション処理は行われていません" ) ;
        }catch( IOException io ){
            this.disconnect() ;
            throw new UndefineBindException( io ) ;
        }
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param timeout 受信タイムアウト値を設定します.
     * @return byte[] 受信されたバイナリ情報が返されます.
     * @exception UndefineBindException 未バインド例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final byte[] receive( int timeout )
        throws UndefineBindException,ConnectTimeoutException
    {
        byte[] ret = null ;
        ByteArrayOutputStream out = null ;
        
        out = new ByteArrayOutputStream() ;
        
        try{
            
            this.receive( out,timeout ) ;
            ret = out.toByteArray() ;
            
        }catch( InputException in ){
            this.disconnect() ;
            ret = null ;
        }catch( UndefineBindException ub ){
            this.disconnect() ;
            throw ub ;
        }catch( ConnectTimeoutException ct ){
            throw ct ;
        }finally{
            try{
                out.close() ;
            }catch( Exception t ){
            }
            out = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信されたバイナリ情報が返されます.
     * @param timeout 受信タイムアウト値を設定します.
     * @exception InputException 入力例外.
     * @exception UndefineBindException 未バインド例外.
     * @exception ConnectTimeoutException タイムアウト例外.
     */
    public final void receive( ByteArrayOutputStream out,int timeout )
        throws InputException,UndefineBindException,ConnectTimeoutException
    {
        int len ;
        
        byte[] buf = null ;
        BufferedInputStream in = null ;
        BufferedOutputStream outBuf = null ;
        
        if( out == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        outBuf = new BufferedOutputStream( out ) ;
        
        timeout = ( timeout <= 0 ) ? 0 : timeout ;
        buf = new byte[ TcpProtocol.RECV_BUFLEN ] ;
        
        try{
            
            synchronized( m_sync.get() ){
                in = m_recv ;
                m_net.setSoTimeout( timeout ) ;
            }
            
            for( ;; ){
                
                len = in.read( buf ) ;
                
                if( in.available() <= 0 ){
                    
                    if( len > 0 ){
                        outBuf.write( buf,0,len ) ;
                    }
                    
                    break ;
                    
                }
                
                outBuf.write( buf,0,len ) ;
                
            }
            
            outBuf.flush() ;
            
        }catch( InterruptedIOException ii ){
            
            try{
                out.close() ;
            }catch( Exception tt ){
            }
            throw new ConnectTimeoutException( ii ) ;
            
        }catch( Exception t ){
            
            
            try{
                out.close() ;
            }catch( Exception tt ){
            }
            this.disconnect() ;
            throw new UndefineBindException( t ) ;
            
        }finally{
            
            try{
                outBuf.close() ;
            }catch( Exception tt ){
            }
            
            try{
                synchronized( m_sync.get() ){
                    m_net.setSoTimeout( 0 ) ;
                    m_lastTime = System.currentTimeMillis() ;
                }
            }catch( Exception tt ){
            }
            
            buf = null ;
            in = null ;
            outBuf = null ;
        }
        
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @param addr 対象のローカルアドレスが返されます.
     */
    public final void getLocal( ConnectAddress addr )
    {
        try{
            if( addr != null ){
                synchronized( m_sync.get() ){
                    addr.create(
                        m_net.getLocalAddress(),
                        m_net.getLocalPort()
                    ) ;
                }
            }
        }catch( Exception t ){
            if( addr != null ){
                
                try{
                    addr.clear() ;
                    addr.create( NetDef.NOT_ADDR,NetDef.PORT_MIN ) ;
                }catch( Exception tt ){
                }
                
            }
        }
    }
    
    /**
     * 接続先アドレスを取得.
     * <BR><BR>
     * 対象の接続先アドレスを取得します.
     * <BR>
     * @param addr 対象の接続アドレスが返されます.
     */
    public final void getConnect( ConnectAddress addr )
    {
        try{
            if( addr != null ){
                synchronized( m_sync.get() ){
                    addr.create(
                        m_net.getInetAddress(),
                        m_net.getPort()
                    ) ;
                }
            }
        }catch( Exception t ){
            if( addr != null ){
                
                try{
                    addr.clear() ;
                    addr.create( NetDef.NOT_ADDR,NetDef.PORT_MIN ) ;
                }catch( Exception tt ){
                }
                
            }
        }
    }
    
    /**
     * ローカルアドレスを取得.
     * <BR><BR>
     * 対象のローカルアドレスを取得します.
     * <BR>
     * @return ConnectAddress 対象のローカルアドレスが返されます.
     */
    public final ConnectAddress getLocal()
    {
        ConnectAddress ret = new ConnectAddress() ;
        
        try{
            synchronized( m_sync.get() ){
                ret = new ConnectAddress(
                    m_net.getInetAddress(),
                    m_net.getPort()
                ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 送受信バッファ長を取得.
     * <BR><BR>
     * 設定されている送受信バッファ長が返されます.
     * <BR>
     * @return int 対象の送受信バッファ長が返されます.
     */
    public final int getBuffer()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_bufLen ;
            }
        }catch( Exception t ){
            this.disconnect() ;
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 最終処理時間を取得.
     * <BR><BR>
     * 最終処理時間を取得します.
     * <BR>
     * @return long 最終処理時間が返されます.
     */
    public final long getLastTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_lastTime ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 接続チェック.
     * <BR><BR>
     * 接続されているかチェックします.
     * <BR>
     * @return boolean 接続状況が返されます.<BR>
     *                 [true]が返された場合、接続されています.
     *                 [false]が返された場合、接続されていません.
     */
    public final boolean isConnect()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                
                // エラーチェック.
                m_net.getSoTimeout() ;
                
                // エラーが検知されない場合[true]を設定.
                ret = true ;
                
            }
        }catch( Exception t ){
            this.disconnect() ;
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * コネクション共通.
     */
    private final void connectTo( int bufLen )
        throws IOException
    {
        
        try{
            
            m_net.setKeepAlive( true ) ;
            m_net.setSoLinger( false,0 ) ;
            m_net.setTcpNoDelay( true ) ;
            
            m_net.setReceiveBufferSize( bufLen ) ;
            m_net.setSendBufferSize( bufLen ) ;
            m_net.setSoTimeout( 0 ) ;
            
            m_send = new BufferedOutputStream( m_net.getOutputStream() ) ;
            m_recv = new BufferedInputStream( m_net.getInputStream() ) ;
            m_bufLen = bufLen ;
            
            m_lastTime = System.currentTimeMillis() ;
            
        }catch( IOException io ){
            this.disconnect() ;
            throw io ;
        }
        
    }
    
}

