/*
 * @(#)FTPClient.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.ftp;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.ArrayList;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.NotLoginException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.DateTimeFormat;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;


/**
 * FTPClient.
 * <BR><BR>
 * FTPクライアント処理をサポートします.
 *  
 * @version 1.0.0 2004/11/27
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class FTPClient
{
    
    /**
     * 読み込み/書き込み用バッファサイズ.
     */
    private static final int BO_BUFFER_SIZE = 0x00080000 ;
    
    /**
     * 接続ソケット.
     */
    private Socket m_socket = null ;
    
    /**
     * 書き込み処理.
     */
    private PrintWriter m_writer = null ;
    
    /**
     * 読み込み処理.
     */
    private BufferedReader m_reader = null ;
    
    /**
     * システム名.
     */
    private String m_system = null ;
    
    /**
     * ユーザ名.
     */
    private String m_userName = null ;
    
    /**
     * パスワード.
     */
    private String m_passwd = null ;
    
    /**
     * コンソールキャラクターセット.
     */
    private String m_consoleCharset = null ;
    
    /**
     * コマンドキャラクターセット.
     */
    private String m_commandCharset = null ;
    
    /**
     * 前回のディレクトリ位置.
     */
    private String m_beforeDir = null ;
    
    /**
     * タイムアウト値.
     */
    private int m_timeout = 0 ;
    
    /**
     * 転送モード.
     */
    private boolean m_mode = false ;
    
    /**
     * PASVモード.
     */
    private boolean m_pasv = false ;
    
    /**
     * デバッグ出力モード.
     */
    private boolean m_isDebug = false ;
    
    /**
     * ログイン時のリスト取得有無.
     */
    private boolean m_isGetLoginList = true ;
    
    /**
     * デバッグ出力先.
     * デフォルトではコンソール.
     */
    private PrintStream m_debugWriter = System.out ;
    
    /**
     * 現在リスト情報格納テーブル.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * 読み込みデータバッファ.
     */
    private final ObjectArray m_readBuf = new ObjectArray() ;
    
    /**
     * データテンポラリ.
     */
    private final ObjectArray m_tmp = new ObjectArray() ;
    
    /**
     * ファイル日付解析用.
     */
    private final DateTimeFormat m_format = new DateTimeFormat( "YYYYMMDDhhmmss" ) ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * 読み込みバッファ用同期.
     */
    private final Synchronized m_readSync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public FTPClient()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * デバッグモードを設定します.
     * <BR>
     * @param mode デバッグモードを設定します.<BR>
     *             [true]を設定した場合デバッグモードはONとなります.<BR>
     *             [false]を設定した場合デバッグモードはOFFとなります.
     */
    public FTPClient( boolean mode )
    {
        this.setDebugMode( mode,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * デバッグモードを設定します.
     * <BR>
     * @param mode デバッグモードを設定します.<BR>
     *             [true]を設定した場合デバッグモードはONとなります.<BR>
     *             [false]を設定した場合デバッグモードはOFFとなります.
     * @param stream 出力先のストリーム条件を設定します.<BR>
     *               [null]を指定した場合[this.setDebugMode( mode )]と同じ処理に
     *               なります.
     */
    public FTPClient( boolean mode,PrintStream stream )
    {
        this.setDebugMode( mode,stream ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host,int port )
        throws InputException,AccessException
    {
        this.open( false,host,port,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,user,passwd,null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host,int port,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( false,host,port,user,passwd,null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host,int port )
        throws InputException,AccessException
    {
        this.open( pasv,host,port,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,user,passwd,null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host,int port,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( pasv,host,port,user,passwd,null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,user,passwd,cons,cmd ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( String host,int port,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( false,host,port,user,passwd,cons,cmd ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,user,passwd,cons,cmd ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public FTPClient( boolean pasv,String host,int port,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( pasv,host,port,user,passwd,cons,cmd ) ;
    }
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.close() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host,int port )
        throws InputException,AccessException
    {
        this.open( false,host,port,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,user,passwd,null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host,int port,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( false,host,port,user,passwd,null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host,int port )
        throws InputException,AccessException
    {
        this.open( pasv,host,port,FTPDef.ANONYMOUS,"",null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,user,passwd,null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host,int port,String user,String passwd )
        throws InputException,AccessException
    {
        this.open( pasv,host,port,user,passwd,null,null ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( false,host,FTPDef.DEF_PORT,user,passwd,cons,cmd ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( String host,int port,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( false,host,port,user,passwd,cons,cmd ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        this.open( pasv,host,FTPDef.DEF_PORT,user,passwd,cons,cmd ) ;
    }
    
    /**
     * 接続先を指定してオープン.
     * <BR><BR>
     * 接続先を指定してオープンします.
     * <BR>
     * @param pasv PASVモードの有効/無効を設定します.<BR>
     *             [true]を設定した場合PASVモードは有効となります.
     *             [false]を設定した場合PASVモードは無効となります.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @param user 接続先のユーザ名を設定します.
     * @param passwd 接続先のパスワードを設定します.
     * @param cons コンソールに対するキャラクターセットを設定します.
     * @param cmd コマンドに対するキャラクターセットを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void open( boolean pasv,String host,int port,String user,String passwd,String cons,String cmd )
        throws InputException,AccessException
    {
        
        if(
            host == null || port < 0 ||
            port > 65535 || user == null ||
            user.length() <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.close() ;
        m_sync.create() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                // PASVモードを設定.
                m_pasv = pasv ;
                
                // ソケット生成.
                this.createConnect( host,port,cmd,cons ) ;
                
                // ログイン.
                this.doLogin( user,passwd ) ;
                
                // モード設定( デフォルトはBinary. ).
                this.setMode( false ) ;
                
            }
            
        }catch( InputException in ){
            this.close() ;
            throw in ;
        }catch( AccessException ac ){
            this.close() ;
            throw ac ;
        }
        
    }
    
    /**
     * クローズ処理.
     * <BR><BR>
     * クローズ処理を実施します.
     */
    public final void close()
    {
        
        try{
            synchronized( m_sync.get() ){
                this.doQuit() ;
                this.clearConnect() ;
            }
        }catch( Exception t ){
        }
        
        m_sync.clear() ;
        
        try{
            this.doQuit() ;
        }catch( Exception t ){
        }
        try{
            this.clearConnect() ;
        }catch( Exception t ){
        }
        
        m_table.clear() ;
        m_readBuf.clear() ;
        m_tmp.clear() ;
        
        m_system = null ;
        m_userName = null ;
        m_passwd = null ;
        m_consoleCharset = null ;
        m_commandCharset = null ;
        m_beforeDir = null ;
        
        m_timeout = 0 ;
        m_mode = false ;
        m_pasv = false ;
        //m_isDebug = false ;
        //m_debugWriter = BaseDef.getInstance().getOut() ;
        
    }
    
    /**
     * 送信処理.
     * <BR><BR>
     * 送信処理を実施します.
     * <BR>
     * @param name 転送先のファイル名を設定します.
     * @param reader 送信対象のデータを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void send( String name,InputStream reader )
        throws InputException,AccessException
    {
        String file = null ;
        String tmp = null ;
        
        if( name == null || reader == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                name = UtilCom.trimPlus( name ) ;
                
                // 現在位置に対して送信する場合.
                if( name.indexOf( "/" ) == -1 ){
                    
                    // 送信対象のファイルが既に存在する場合.
                    if( m_table.isData( name ) == true ){
                        
                        tmp = this.doPwd() + name ;
                        
                        throw new AccessException(
                            "送信対象のファイル(" + tmp +
                            ")は既に存在します"
                        ) ;
                        
                    }
                    // 送信が可能な場合.
                    else{
                        
                        this.doPut( reader,name ) ;
                        this.getList( null ) ;
                        
                    }
                    
                }
                // 送信対象名がディレクトリの場合.
                else if( ( file = this.changeDirectory( name ) ) == null ){
                    
                    tmp = this.doPwd() ;
                    
                    throw new AccessException(
                        "送信対象(" + tmp +
                        ")はディレクトリ名です"
                    ) ;
                    
                }
                // 送信対象のファイルが既に存在する場合.
                else if( m_table.isData( file ) == true ){
                    
                    tmp = this.doPwd() + file ;
                    
                    throw new AccessException(
                        "送信対象のファイル(" + tmp +
                        ")は既に存在します"
                    ) ;
                    
                }
                // 送信対象のファイルが存在しない場合.
                else{
                    
                    this.doPut( reader,file ) ;
                    
                }
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            
            try{
                // 移動ディレクトリを元に戻す.
                this.beforeDirectory() ;
            }catch( Exception tt ){
            }
            
            file = null ;
            tmp = null ;
            
        }
        
    }
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を実施します.
     * <BR>
     * @param name 転送元のファイル名を設定します.
     * @param writer 転送先のデータを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final void receive( String name,OutputStream writer )
        throws InputException,AccessException
    {
        String file = null ;
        String tmp = null ;
        
        if( name == null || writer == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                name = UtilCom.trimPlus( name ) ;
                
                // 現在位置に対して受信する場合.
                if( name.indexOf( "/" ) == -1 ){
                    
                    // 送信対象のファイルが存在しない場合.
                    if( m_table.isData( name ) == false ){
                        
                        tmp = this.doPwd() + name ;
                        
                        throw new AccessException(
                            "受信対象のファイル(" + tmp +
                            ")は存在しません"
                        ) ;
                        
                    }
                    // 受信が可能な場合.
                    else{
                        
                        this.doGet( writer,name ) ;
                        
                    }
                    
                }
                // 受信対象名がディレクトリの場合.
                else if( ( file = this.changeDirectory( name ) ) == null ){
                    
                    tmp = this.doPwd() ;
                    
                    throw new AccessException(
                        "受信対象(" + tmp +
                        ")はディレクトリ名です"
                    ) ;
                    
                }
                // 受信対象のファイルが存在しない場合.
                else if( m_table.isData( file ) == false ){
                    
                    tmp = this.doPwd() + file ;
                    
                    throw new AccessException(
                        "受信対象のファイル(" + tmp +
                        ")は存在しません"
                    ) ;
                    
                }
                // 受信対象のファイルが存在する場合..
                else{
                    
                    this.doGet( writer,file ) ;
                    
                }
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            
            try{
                // 移動ディレクトリを元に戻す.
                this.beforeDirectory() ;
            }catch( Exception tt ){
            }
            
            file = null ;
            tmp = null ;
            
        }
        
    }
    
    /**
     * リストを上位に移動.
     * <BR><BR>
     * リストを指定位置に移動します.
     * <BR>
     * @param name 移動先の位置を設定します.
     * @return boolean 移動結果が返されます.<BR>
     *                 [true]が返された場合、移動されました.<BR>
     *                 [false]が返された場合、移動されませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean moveList( String name )
        throws InputException
    {
        boolean ret ;
        
        if( name == null || name.length() < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            this.doCd( name ) ;
            this.getList( null ) ;
            ret = true ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * リストを下位に移動.
     * <BR><BR>
     * リストを下位に移動します.
     * <BR>
     * @return boolean 移動結果が返されます.<BR>
     *                 [true]が返された場合、下位に移動されました.<BR>
     *                 [false]が返された場合、移動されませんでした.
     */
    public final boolean downList()
    {
        boolean ret ;
        
        try{
            this.doCd( ".." ) ;
            this.getList( null ) ;
            ret = true ;
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 転送モードを変更.
     * <BR><BR>
     * 転送モードを変更します.
     * <BR>
     * @param mode 転送モードを設定します.<BR>
     *             [true]を設定した場合ASCIIモードになります.<BR>
     *             [false]を設定した場合BINARYモードになります.
     */
    public final void setMode( boolean mode )
    {
        try{
            synchronized( m_sync.get() ){
                this.doMode( mode ) ;
                m_mode = mode ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * ファイル/ディレクトリ名を変更.
     * <BR><BR>
     * ファイル/ディレクトリ名を変更します.
     * <BR>
     * @param src 変更元の名前を設定します.
     * @param dest 変更先の名前を設定します.
     * @return boolean 結果情報が返されます.
     */
    public final boolean changeName( String src,String dest )
    {
        boolean ret ;
        
        ret = false ;
        
        if(
            src != null && dest != null &&
            src.indexOf( "/" ) == -1 &&
            dest.indexOf( "/" ) == -1
        )
        {
            
            try{
                synchronized( m_sync.get() ){
                    
                    this.doRename( src,dest ) ;
                    ret = true ;
                    
                }
            }catch( Exception t ){
                ret = false ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * ディレクトリを生成.
     * <BR><BR>
     * 対象リスト位置にディレクトリ情報を生成します.
     * <BR>
     * @param name 生成対象のディレクトリ名を設定します.
     * @return boolean 削除結果が返されます.<BR>
     *                 [true]が返された場合生成されました.<BR>
     *                 [false]が返された場合生成されませんでした.
     */
    public final boolean mkdir( String name )
    {
        boolean ret ;
        
        ret = false ;
        
        if( name != null ){
            try{
                this.doMkDir( name ) ;
                ret = true ;
            }catch( Exception t ){
                ret = false ;
            }
        }
        
        return ret ;
        
    }
    
    /**
     * 指定情報を削除.
     * <BR><BR>
     * 指定した情報を削除します.
     * またディレクトリは空で無い場合削除できません.
     * <BR>
     * @param name 削除対象の名前を指定します.
     * @return boolean 削除結果が返されます.<BR>
     *                 [true]が返された場合削除できました.<BR>
     *                 [false]が返された場合削除できませんでした.
     */
    public final boolean delete( String name )
    {
        boolean ret ;
        
        String file = null ;
        FTPPause pause = null ;
        
        ret = false ;
        
        // 指定名が存在する場合.
        if( name != null ){
            
            try{
                synchronized( m_sync.get() ){
                    
                    // パス指定が無い場合.
                    if( name.indexOf( "/" ) == -1 ){
                        
                        name = UtilCom.trimPlus( name ) ;
                        
                        // 対象名を取得.
                        if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                            
                            // 削除対象がディレクトリの場合.
                            if( pause.getMode() == true ){
                                
                                this.doRmDir( name ) ;
                                this.getList( null ) ;
                                
                                ret = true ;
                                
                            }
                            // 削除対象がファイルの場合.
                            else{
                                
                                this.doDel( name ) ;
                                this.getList( null ) ;
                                
                                ret = true ;
                                
                            }
                            
                        }
                        
                    }
                    // 削除対象がディレクトリの場合.
                    else if( ( file = this.changeDirectory( name ) ) == null ){
                        
                        if( m_table.size() == 0 ){
                            
                            if( ( name = this.getLastName( name ) ) != null ){
                                
                                this.doCd( ".." ) ;
                                this.doRmDir( name ) ;
                                ret = true ;
                                
                            }
                            
                        }
                        
                    }
                    // 削除対象がファイルの場合.
                    else{
                        
                        this.doDel( file ) ;
                        ret = true ;
                        
                    }
                    
                }
                
            }catch( Exception t ){
                ret = false ;
            }finally{
                
                try{
                    // 移動ディレクトリを元に戻す.
                    this.beforeDirectory() ;
                }catch( Exception tt ){
                }
                
                file = null ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 権限情報を設定.
     * <BR><BR>
     * 指定名の権限情報を設定します.
     * <BR>
     * @param name 設定対象の名前を設定します.
     * @param own 権限情報を設定します.<BR>
     *             ユーザ権限は[ ( ret & 0x000000ff ) << 16 ]で設定します.<BR>
     *             グループ権限は[ ( ret & 0x000000ff ) << 8 ]で設定します.<BR>
     *             その他の権限は[ ret & 0x000000ff ]で設定します.
     * @return boolean 移動結果が返されます.<BR>
     *                 [true]が返された場合、指定の権限が設定されました.<BR>
     *                 [false]が返された場合、指定の権限が設定されませんでした.
     */
    public final boolean setOwner( String name,int own )
    {
        boolean ret ;
        
        ret = false ;
        
        if( name != null && name.length() > 0 ){
            
            try{
                this.doOwner( name,own ) ;
                this.getList( null ) ;
                ret = true ;
            }catch( Exception t ){
                ret = false ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 受信タイムアウト値を設定.
     * <BR><BR>
     * 受信タイムアウト値を設定します.
     * <BR>
     * @param time 受信タイムアウト値を設定します.
     */
    public final void setTimeout( int time )
    {
        try{
            synchronized( m_sync.get() ){
                if( time > 0L ){
                    m_timeout = time ;
                }
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * ログイン時のリスト取得処理有無を設定.
     * <BR><BR>
     * ログイン時のリスト取得処理有無を設定します.
     * <BR>
     * @param flg 設定対象の有無条件を設定します.<BR>
     *            [true]を設定した場合ログイン時にリスト取得を行います.
     *            [false]を設定した場合ログイン時にリスト取得を行いません.
     */
    public final void setLoginList( boolean flg )
    {
        try{
            synchronized( m_sync.get() ){
                m_isGetLoginList = flg ;
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * デバッグモードを設定.
     * <BR><BR>
     * デバッグモードを設定します.
     * <BR>
     * @param mode デバッグモードを設定します.<BR>
     *             [true]を設定した場合デバッグモードはONとなります.<BR>
     *             [false]を設定した場合デバッグモードはOFFとなります.
     */
    public final void setDebugMode( boolean mode )
    {
        this.setDebugMode( mode,null ) ;
    }
    
    /**
     * デバッグモードを設定.
     * <BR><BR>
     * デバッグモードを設定します.
     * <BR>
     * @param mode デバッグモードを設定します.<BR>
     *             [true]を設定した場合デバッグモードはONとなります.<BR>
     *             [false]を設定した場合デバッグモードはOFFとなります.
     * @param stream 出力先のストリーム条件を設定します.<BR>
     *               [null]を指定した場合[this.setDebugMode( mode )]と同じ処理に
     *               なります.
     */
    public final void setDebugMode( boolean mode,PrintStream stream )
    {
        try{
            synchronized( m_sync.get() ){
                
                m_isDebug = mode ;
                
                if( stream != null ){
                    m_debugWriter = stream ;
                }
                
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 現在のリスト位置を取得.
     * <BR><BR>
     * 現在のリスト位置を取得します.
     * <BR>
     * @return String 現在のリスト位置が返されます.
     */
    public final String getPwd()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.doPwd() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在位置のリスト内のディレクトリ名を取得.
     * <BR><BR>
     * 現在位置のリスト内のディレクトリ名を取得します.
     * <BR>
     * @return String[] 現在のリスト内のディレクトリ名が返されます.
     */
    public final String[] getListDirs()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getNowList( true ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在リスト内のファイル名を取得.
     * <BR><BR>
     * 現在のリスト内のファイル名を取得します.
     * <BR>
     * @return String[] 現在のリスト内のファイル名が返されます.
     */
    public final String[] getListFiles()
    {
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getNowList( false ) ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 転送モードを取得.
     * <BR><BR>
     * 転送モードを取得します.
     * <BR>
     * @return boolean 転送モードを設定します.<BR>
     *                 [true]を設定した場合ASCIIモードになります.<BR>
     *                 [false]を設定した場合BINARYモードになります.
     */
    public final boolean getMode()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_mode ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 権限情報を取得.
     * <BR><BR>
     * 指定名の権限情報を取得します.
     * <BR>
     * @param name 取得対象の名前を設定します.
     * @return int 権限情報が返されます.<BR>
     *             ユーザ権限は[ ( ret & 0x00ff0000 ) >> 16 ]で取得します.<BR>
     *             グループ権限は[ ( ret & 0x0000ff00 ) >> 8 ]で取得します.<BR>
     *             その他の権限は[ ret & 0x000000ff ]で取得します.
     */
    public final int getOwner( String name )
    {
        int ret ;
        
        String file = null ;
        FTPPause pause = null ;
        
        ret = 0 ;
        
        if( name != null ){
            
            try{
                synchronized( m_sync.get() ){
                    
                    name = UtilCom.trimPlus( name ) ;
                    
                    // 現在位置に対して取得する場合.
                    if( name.indexOf( "/" ) == -1 ){
                        
                        
                        // 対象の名前が既に存在する場合.
                        if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                            ret = pause.getOwner() ;
                        }
                        // ログイン時にリストを取得していない場合.
                        else if( m_isGetLoginList == false ){
                            
                            this.getList( null ) ;
                            
                            if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                                ret = pause.getOwner() ;
                            }
                            
                        }
                        
                    }
                    // 送信対象名がディレクトリの場合.
                    else if( ( file = this.changeDirectory( name ) ) == null ){
                        
                        this.doCd( ".." ) ;
                        this.getList( null ) ;
                        file = this.getLastName( name ) ;
                        
                        if( ( pause = ( FTPPause )m_table.get( file ) ) != null ){
                            ret = pause.getOwner() ;
                        }
                        
                    }
                    // 送信対象のファイルが既に存在する場合.
                    else if( ( pause = ( FTPPause )m_table.get( file ) ) != null ){
                        
                        ret = pause.getOwner() ;
                        
                    }
                    
                }
                
            }catch( Exception t ){
                ret = 0 ;
            }finally{
                
                try{
                    // 移動ディレクトリを元に戻す.
                    this.beforeDirectory() ;
                }catch( Exception tt ){
                }
                
                file = null ;
                pause = null ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * ファイルサイズを取得.
     * <BR><BR>
     * 対象のファイルサイズを取得します.
     * <BR>
     * @param name 取得対象の名前を設定します.
     * @return long 対象のファイルサイズが返されます.<BR>
     *              ファイル名で無い場合[-1L]が返されます.
     */
    public final long getSize( String name )
    {
        long ret ;
        
        FTPPause pause = null ;
        
        ret = 0L ;
        
        if( name != null ){
            
            try{
                synchronized( m_sync.get() ){
                    
                    name = UtilCom.trimPlus( name ) ;
                    
                    // 現在位置に対して取得する場合.
                    if( name.indexOf( "/" ) == -1 ){
                        
                        // 対象の名前が既に存在する場合.
                        if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                            ret = pause.getSize() ;
                        }
                        // ログイン時にリストを取得していない場合.
                        else if( m_isGetLoginList == false ){
                            ret = this.doFileSize( name ) ;
                        }
                        
                    }
                    // 絶対/相対パス指定の場合.
                    else{
                        ret = this.doFileSize( name ) ;
                    }
                    
                }
                
            }catch( Exception t ){
                ret = 0L ;
            }finally{
                pause = null ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 最新日付を取得.
     * <BR><BR>
     * 最新の日付情報を取得します.
     * <BR>
     * @param name 取得対象の名前を設定します.
     * @return long 最新の日付が返されます.
     */
    public final long getDate( String name )
    {
        long ret ;
        
        FTPPause pause = null ;
        
        ret = 0L ;
        
        if( name != null ){
            
            try{
                synchronized( m_sync.get() ){
                    
                    name = UtilCom.trimPlus( name ) ;
                    
                    // 現在位置に対して取得する場合.
                    if( name.indexOf( "/" ) == -1 ){
                        
                        // 対象の名前が既に存在する場合.
                        if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                            ret = pause.getDate() ;
                        }
                        // ログイン時にリストを取得していない場合.
                        else if( m_isGetLoginList == false ){
                            ret = this.doDateTime( name ) ;
                        }
                        
                    }
                    // 絶対/相対パス指定の場合.
                    else{
                        ret = this.doDateTime( name ) ;
                    }
                    
                }
                
            }catch( Exception t ){
                ret = 0L ;
            }finally{
                pause = null ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 接続システム名を取得.
     * <BR><BR>
     * 接続されているシステム名を取得します.
     * <BR>
     * @return String 接続されているシステム名が返されます.
     */
    public final String getSystem()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                if( m_system == null ){
                    m_system = this.doSystem() ;
                }
                ret = m_system ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 受信タイムアウト値を取得.
     * <BR><BR>
     * 受信タイムアウト値を取得します.
     */
    public final int getTimeout()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_timeout ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド用キャラクターセットを取得.
     * <BR><BR>
     * コマンド用キャラクターセットを取得します.
     * <BR>
     * @return String 設定されているキャラクタセットが返されます.
     */
    public final String getCommandCharset()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_commandCharset == null ) ?
                    BaseDef.THIS_CHARSET : m_commandCharset ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * コンソール用キャラクターセットを取得.
     * <BR><BR>
     * コンソール用キャラクターセットを取得します.
     * <BR>
     * @return String 設定されているキャラクタセットが返されます.
     */
    public final String getConsoleCharset()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_consoleCharset == null ) ?
                    BaseDef.THIS_CHARSET : m_consoleCharset ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * デバッグ用ストリームを取得.
     * <BR><BR>
     * デバッグ用ストリームを取得します.
     * <BR>
     * @return PrintStream 設定されているデバッグ用ストリームが返されます.
     */
    public final PrintStream getDebugStream()
    {
        PrintStream ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_debugWriter ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * ファイル/ディレクトリチェック.
     * <BR><BR>
     * 指定名がファイルかディレクトリであるかチェックします.
     * <BR>
     * @param name チェック対象名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合ディレクトリです.
     *                 [false]が返された場合ファイルです.
     */
    public final boolean isDir( String name )
    {
        boolean ret ;
        
        FTPPause pause = null ;
        
        ret = false ;
        
        if( name != null ){
            
            try{
                synchronized( m_sync.get() ){
                    
                    name = UtilCom.trimPlus( name ) ;
                    
                    // 現在位置に対して取得する場合.
                    if( name.indexOf( "/" ) == -1 ){
                        
                        // 対象の名前が既に存在する場合.
                        if( ( pause = ( FTPPause )m_table.get( name ) ) != null ){
                            ret = pause.getMode() ;
                        }
                        
                    }
                    // 送信対象名がディレクトリの場合.
                    else if( this.changeDirectory( name ) == null ){
                        
                        ret = true ;
                        
                    }
                    
                }
                
            }catch( Exception t ){
                ret = false ;
            }finally{
                
                try{
                    // 移動ディレクトリを元に戻す.
                    this.beforeDirectory() ;
                }catch( Exception tt ){
                }
                
                pause = null ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * ログイン時にリストを取得するか否かを取得.
     * <BR><BR>
     * ログイン時にリストを取得するか否かを取得します.
     * <BR>
     * @return boolean ログイン時にリストを取得するか否かが返されます.<BR>
     *                 [true]が返された場合ログイン時にリストを取得します.<BR>
     *                 [false]が返された場合ログイン時にリストを取得しません.
     */
    public final boolean isLoginList()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_isGetLoginList ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * PASVモードであるかチェック.
     * <BR><BR>
     * PASVモードであるかチェックします.
     * <BR>
     * @return boolean 設定されたPASVモードの有無が返されます.
     *                 [true]が返された場合PASVモードです.
     *                 [false]が返された場合PASVモードではありません.
     */
    public final boolean isPasv()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_pasv ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }

    
    /**
     * デバッグモードを取得.
     * <BR><BR>
     * デバッグモードを取得します.
     * <BR>
     * @return boolean 現在のデバッグモードが返されます.<BR>
     *                 [true]が返された場合デバッグモードは有効です.<BR>
     *                 [false]が返された場合デバッグモードは無効です.
     */
    public final boolean isDebug()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_isDebug ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * オープンチェック.
     * <BR><BR>
     * 現在のオブジェクトがオープン中であるかチェックします.
     * <BR>
     * @return boolean オープンチェックが返されます.<BR>
     *                 [true]が返された場合オープン中です.<BR>
     *                 [false]が返された場合クローズされています.
     */
    public final boolean isOpen()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 生成処理を実施.
     */
    private final void createConnect( String host,int port,String cmd,String cons )
        throws InputException,AccessException
    {
        Thread th = null ;
        
        if( host == null || host.length() <= 0 || port < 0 || port > 65535 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clearConnect() ;
        m_readSync.create() ;
        
        try{
            
            m_socket = new Socket( host,port ) ;
            m_socket.setSoLinger( false,0 ) ;
            m_socket.setSoTimeout( FTPDef.COMMAND_TIMEOUT ) ;
            
            
            if( cmd == null ){
                
                m_writer = new PrintWriter(
                    new BufferedWriter(
                        new OutputStreamWriter( m_socket.getOutputStream() )
                    ),true
                ) ;
                
                m_reader = new BufferedReader(
                    new InputStreamReader( m_socket.getInputStream() )
                ) ;
                
            }
            else{
                
                m_writer = new PrintWriter(
                    new BufferedWriter(
                        new OutputStreamWriter( m_socket.getOutputStream(),cmd )
                    ),true
                ) ;
                
                m_reader = new BufferedReader(
                    new InputStreamReader( m_socket.getInputStream(),cmd )
                ) ;
                
            }
            
            th = new Thread( new ReaderBuffer( m_reader,m_readBuf,m_readSync ) ) ;
            th.setDaemon( true ) ;
            th.start() ;
            
            m_consoleCharset = cons ;
            m_commandCharset = cmd ;
            
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            th = null ;
        }
        
    }
    
    /**
     * クリア処理を実施.
     */
    private final void clearConnect()
    {
        try{
            m_writer.close() ;
        }catch( Exception t ){
        }
        try{
            m_reader.close() ;
        }catch( Exception t ){
        }
        try{
            m_socket.close() ;
        }catch( Exception t ){
        }
        
        m_socket = null ;
        m_writer = null ;
        m_reader = null ;
        
        m_readSync.clear() ;
        
    }
    
    /**
     * ログイン処理.
     */
    private final void doLogin( String user,String passwd )
        throws InputException,NotLoginException
    {
        StringBuffer buf = null ;
        
        if(
            user == null || passwd == null ||
            user.length() <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            // 戻り値を取得.
            this.waitReturn() ;
            
            buf = new StringBuffer() ;
            buf.append( "USER " ) ;
            buf.append( user ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            buf = new StringBuffer() ;
            buf.append( "PASS " ) ;
            buf.append( passwd ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            
            // 戻り値を取得.
            //this.waitReturn() ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            // ログイン時にリスト内容を取得する場合.
            if( m_isGetLoginList == true ){
                // ログイン後のリスト内容を取得.
                this.getList( null ) ;
            }
            
            // ログイン/パスワードを設定.
            m_userName = user ;
            m_passwd = passwd ;
            
        }catch( Exception t ){
            throw new NotLoginException( t ) ;
        }finally{
            buf = null ;
        }
        
    }
    
    /**
     * 接続切断処理.
     */
    private final void doQuit()
        throws AccessException
    {
        String quitCmd = "QUIT " ;
        
        try{
            
            // コマンド実行.
            //this.execCmd( "QUIT " ) ;
            
            // 結果処理をチェック.
            //this.checkTelegram() ;
            
            this.outDebug( false,quitCmd ) ;
            
            // コマンド実行.
            m_writer.println( quitCmd ) ;
            m_writer.flush() ;
            
        }catch( Exception t ){
        }
    }
    
    /**
     * システム名を取得.
     */
    private final String doSystem()
        throws AccessException
    {
        String ret ;
        
        try{
            
            // ASCIIに変更.
            this.doMode( true ) ;
            
            // コマンド実行.
            ret = this.execCmd( "SYST " ) ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            ret = UtilCom.trimPlus( ret ) ;
            
            // 元に戻す.
            this.doMode( m_mode ) ;
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在位置を取得.
     */
    private final String doPwd()
        throws AccessException
    {
        int i ;
        int len ;
        int pnt ;
        char cd ;
        
        StringBuffer buf = null ;
        String pwd = null ;
        String ret = null ;
        
        // コマンド実行.
        pwd = this.execCmd( "PWD " ) ;
        
        if( pwd == null ){
            throw new AccessException( "[PWD]コマンドの実施に失敗しました" ) ;
        }
        
        // 結果処理をチェック.
        this.checkTelegram() ;
        
        // 取得データを解析.
        if( pwd != null ){
            
            len = pwd.length() ;
            for( i = 0,pnt = -1 ; i < len ; i ++ ){
                if( pwd.charAt( i ) == '\"' ){
                    if( pnt == -1 ){
                        pnt = i+1 ;
                    }
                    else{
                        
                        ret = pwd.substring( pnt,i ) ;
                        len = ret.length() ;
                        
                        if( ret.charAt( len-1 ) != '/' ){
                            
                            buf = new StringBuffer() ;
                            buf.append( ret ) ;
                            buf.append( "/" ) ;
                            ret = buf.toString() ;
                            buf = null ;
                            
                        }
                        
                        break ;
                    }
                }
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * アスキー/バイナリモード設定.
     */
    private final void doMode( boolean mode )
        throws AccessException
    {
        
        // ASCIIモード.
        if( mode == true ){
            // コマンド実行.
            this.execCmd( "TYPE A " ) ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
        }
        // BINARYモード.
        else{
            // コマンド実行.
            this.execCmd( "TYPE I " ) ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
        }
        
    }
    
    /**
     * カレントディレクトリを移動.
     */
    private final void doCd( String name )
        throws AccessException
    {
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "CWD " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }
    }
    
    /**
     * ディレクトリを作成.
     */
    private final void doMkDir( String name )
        throws AccessException
    {
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "MKD " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }
    }
    
    /**
     * ディレクトリを削除.
     */
    private final void doRmDir( String name )
        throws AccessException
    {
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "RMD " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }
    }
    
    /**
     * ファイルを削除.
     */
    private final void doDel( String name )
        throws AccessException
    {
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "DELE " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }
    }
    
    /**
     * ファイル/ディレクトリ名を変更.
     */
    private final void doRename( String src,String dest )
        throws AccessException
    {
        StringBuffer buf = null ;
        
        if( src != null && dest != null ){
            
            buf = new StringBuffer() ;
            buf.append( "RNFR " ) ;
            buf.append( src ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            buf = new StringBuffer() ;
            buf.append( "RNTO " ) ;
            buf.append( dest ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
        }
        
    }
    
    /**
     * ファイル/ディレクトリから日付を取得.
     * ※[MDTMでは]正確な情報が帰ってこない可能性あり...
     */
    private final long doDateTime( String name )
        throws AccessException
    {
        long ret ;
        String tmp = null ;
        DateTimeFormat dt = null ;
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "MDTM " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            tmp = this.execCmd( buf.toString() ) ;
            buf = null ;
            
            if( tmp == null ){
                throw new AccessException( "[MDTM]コマンドの実施に失敗しました" ) ;
            }
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            dt = m_format ;
            try{
                
                // 日付を解析.
                dt.set( tmp ) ;
                ret = dt.getTime() ;
                
            }catch( Exception t ){
                throw new AccessException( t ) ;
            }
            
        }
        else{
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * ファイルサイズを取得.
     */
    private final long doFileSize( String name )
        throws AccessException
    {
        long ret ;
        String tmp = null ;
        StringBuffer buf = null ;
        
        if( name != null ){
            
            buf = new StringBuffer() ;
            buf.append( "SIZE " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            tmp = this.execCmd( buf.toString() ) ;
            buf = null ;
            
            if( tmp == null ){
                throw new AccessException( "[SIZE]コマンドの実施に失敗しました" ) ;
            }
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
            try{
                
                ret = Long.parseLong( tmp ) ;
                
            }catch( Exception t ){
                throw new AccessException( t ) ;
            }
            
        }
        else{
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 権限変更.
     */
    private final void doOwner( String name,int own )
        throws AccessException
    {
        
        StringBuffer buf = null ;
        
        if( name != null && own > 0 ){
            
            buf = new StringBuffer() ;
            buf.append( "SITE CHMOD " ) ;
            buf.append( Integer.toHexString( own ) ) ;
            buf.append( " " ) ;
            buf.append( name ) ;
            
            // コマンド実行.
            this.execCmd( buf.toString() ) ;
            buf = null ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }
        
    }
    
    /**
     * コマンドリストを取得.
     */
    private final ArrayList doLs( String cmd )
        throws AccessException
    {
        int len ;
        
        Socket sock = null ;
        StringBuffer buf = null ;
        BufferedInputStream reader = null ;
        ArrayList ret = null ;
        
        try{
            
            // 取得モードをASCIIモードに変更.
            this.execCmd( "TYPE A " ) ;
            this.checkTelegram() ;
            
            buf = new StringBuffer() ;
            buf.append( "NLST " ) ;
            
            if( cmd != null ){
                
                buf.append( cmd ) ;
                buf.append( " " ) ;
                
            }
            
            buf.append( "-alL" ) ;
            
            cmd = buf.toString() ;
            buf = null ;
            
            // PASVモードである場合.
            if( m_pasv == true ){
                
                sock = this.pasvCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( sock.getInputStream() ) ;
                ret = this.recvTelegram( reader,m_consoleCharset ) ;
                
            }
            // 通常モードである場合.
            else{
                
                sock = this.defCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( sock.getInputStream() ) ;
                ret = this.recvTelegram( reader,m_consoleCharset ) ;
                
            }
            
            // 戻り値を取得.
            this.waitReturn() ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }catch( InputException in ){
            if( ret != null ){
                ret.clear() ;
            }
            ret = null ;
            throw new AccessException( in ) ;
        }catch( AccessException ac ){
            if( ret != null ){
                ret.clear() ;
            }
            ret = null ;
            throw ac ;
        }catch( Exception t ){
            if( ret != null ){
                ret.clear() ;
            }
            ret = null ;
            throw new AccessException( t ) ;
        }finally{
            
            try{
                // 設定モードを元に戻す.
                this.doMode( m_mode ) ;
            }catch( Exception tt ){
            }
            try{
                reader.close() ;
            }catch( Exception tt ){
            }
            try{
                sock.close() ;
            }catch( Exception tt ){
            }
            
            sock = null ;
            reader = null ;
            buf = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象のファイル情報を取得.
     */
    private final boolean doGet( OutputStream out,String name )
        throws InputException,AccessException
    {
        int len ;
        boolean ret ;
        
        Socket sock = null ;
        String cmd = null ;
        StringBuffer buf = null ;
        BufferedInputStream reader = null ;
        BufferedOutputStream bOut = null ;
        
        if( out == null || name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        try{
            
            // 設定モードを再設定.
            this.doMode( m_mode ) ;
            
            buf = new StringBuffer() ;
            buf.append( "RETR " ) ;
            buf.append( name ) ;
            
            cmd = buf.toString() ;
            buf = null ;
            
            // PASVモードである場合.
            if( m_pasv == true ){
                
                sock = this.pasvCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( sock.getInputStream() ) ;
                bOut = new BufferedOutputStream( out,BO_BUFFER_SIZE ) ;
                
                if( this.ioTelegram( reader,bOut ) != -1 ){
                    ret = true ;
                }
                
                bOut.flush() ;
                
            }
            // 通常モードである場合.
            else{
                
                sock = this.defCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( sock.getInputStream() ) ;
                bOut = new BufferedOutputStream( out,BO_BUFFER_SIZE ) ;
                
                if( this.ioTelegram( reader,out ) != -1 ){
                    ret = true ;
                }
                
                bOut.flush() ;
                
            }
            
            // 戻り値を取得.
            this.waitReturn() ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }catch( InputException in ){
            throw new AccessException( in ) ;
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            
            try{
                reader.close() ;
            }catch( Exception tt ){
            }
            try{
                sock.close() ;
            }catch( Exception tt ){
            }
            try{
                bOut.close() ;
            }catch( Exception tt ){
            }
            
            sock = null ;
            reader = null ;
            bOut = null ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * 対象のファイル情報に出力.
     */
    private final boolean doPut( InputStream input,String name )
        throws InputException,AccessException
    {
        int len ;
        boolean ret ;
        
        Socket sock = null ;
        String cmd = null ;
        StringBuffer buf = null ;
        BufferedInputStream reader = null ;
        BufferedOutputStream bOut = null ;
        OutputStream writer = null ;
        
        if( input == null || name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        try{
            
            // 設定モードを再設定.
            this.doMode( m_mode ) ;
            
            buf = new StringBuffer() ;
            buf.append( "STOR " ) ;
            buf.append( name ) ;
            
            cmd = buf.toString() ;
            buf = null ;
            
            // PASVモードである場合.
            if( m_pasv == true ){
                
                sock = this.pasvCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( input ) ;
                writer = sock.getOutputStream() ;
                bOut = new BufferedOutputStream( writer ) ;
                
                if( this.ioTelegram( reader,bOut ) != -1 ){
                    ret = true ;
                }
                
                bOut.flush() ;
                
            }
            // 通常モードである場合.
            else{
                
                sock = this.defCommand( cmd,m_timeout ) ;
                
                reader = new BufferedInputStream( input ) ;
                writer = sock.getOutputStream() ;
                bOut = new BufferedOutputStream( writer ) ;
                
                if( this.ioTelegram( reader,bOut ) != -1 ){
                    ret = true ;
                }
                
                bOut.flush() ;
                
            }
            
            // 戻り値を取得.
            this.waitReturn() ;
            
            // 結果処理をチェック.
            this.checkTelegram() ;
            
        }catch( InputException in ){
            throw new AccessException( in ) ;
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            
            try{
                reader.close() ;
            }catch( Exception tt ){
            }
            try{
                writer.close() ;
            }catch( Exception tt ){
            }
            try{
                sock.close() ;
            }catch( Exception tt ){
            }
            try{
                bOut.close() ;
            }catch( Exception tt ){
            }
            
            sock = null ;
            reader = null ;
            writer = null ;
            bOut = null ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
    
    
    /**
     * デフォルトコマンド送信.
     */
    private final Socket defCommand( String cmd,int timeout )
        throws InputException,AccessException
    {
        
        int i ;
        int len ;
        int port ;
        
        byte[] addr = null ;
        StringBuffer buf = null ;
        ServerSocket tmp = null ;
        PrintWriter writer = null ;
        Socket ret = null ;
        
        if( cmd == null || cmd.length() <= 0 || timeout < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        writer = m_writer ;
        
        try{
            
            addr = InetAddress.getLocalHost().getAddress() ;
            tmp = new ServerSocket( 0,1 ) ;
            tmp.setSoTimeout( timeout ) ;
            
            len = 4 ;
            buf = new StringBuffer() ;
            buf.append( "PORT " ) ;
            
            for( i = 0 ; i < len ; i ++ ){
                buf.append( ( addr[ i ] & 0x000000ff ) ) ;
                buf.append( "," ) ;
            }
            
            port = tmp.getLocalPort() ;
            buf.append( ( ( ( port & 0x0000ff00 ) >> 8 ) & 0x000000ff ) ) ;
            buf.append( "," ) ;
            buf.append( ( port & 0x000000ff ) ) ;
            
            // PORTコマンド送信.
            this.execCmd( buf.toString() ) ;
            
            // 実行コマンド設定.
            this.execCmd( cmd ) ;
            
            ret = tmp.accept() ;
            ret.setSoTimeout( timeout ) ;
            
        }catch( Exception t ){
            
            try{
                ret.close() ;
            }catch( Exception tt ){
            }
            
            ret = null ;
            throw new AccessException( t ) ;
            
        }finally{
            
            try{
                tmp.close() ;
            }catch( Exception tt ){
            }
            
            addr = null ;
            buf = null ;
            tmp = null ;
            writer = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * PASVコマンド送信.
     */
    private final Socket pasvCommand( String cmd,int timeout )
        throws InputException,AccessException
    {
        int i ;
        int len ;
        
        int pnt ;
        int pauseLen ;
        int state ;
        int port ;
        char cd ;
        
        String host = null ;
        String data = null ;
        String tmp = null ;
        StringBuffer buf = null ;
        PrintWriter writer = null ;
        Socket ret = null ;
        
        if( cmd == null || cmd.length() <= 0 || timeout < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        writer = m_writer ;
        
        try{
            
            // PASVコマンド実行.
            data = this.execCmd( "PASV " ) ;
            if( data == null ){
                throw new AccessException( "[PASV]コマンドの実施に失敗しました" ) ;
            }
            
            // コマンド解析.
            for( ;; ){
            
                len = data.length() ;
                
                // 解析してホストとポート番号を取得.
                for( i = 0,state = -1,pnt = -1,pauseLen = 0,host = null,port = 0 ; i < len ; i ++ ){
                    
                    cd = data.charAt( i ) ;
                    
                    if( state == -1 && cd == '(' ){
                        
                        state = 0 ;
                        pnt = i + 1 ;
                        buf = new StringBuffer() ;
                        
                    }
                    else if( state == 0 && cd == ',' ){
                        
                        pauseLen ++ ;
                        buf.append( data.substring( pnt,i ) ) ;
                        
                        if( pauseLen >= 4 ){
                            
                            host = buf.toString() ;
                            buf = null ;
                            pauseLen = 0 ;
                            state = 1 ;
                            
                        }
                        else{
                            buf.append( "." ) ;
                        }
                        
                        pnt = i + 1 ;
                        
                    }
                    else if( state == 1 && cd == ',' ){
                        
                        port = (
                            ( Integer.parseInt(
                                data.substring( pnt,i )
                                ) & 0x000000ff
                            ) << 8
                        ) ;
                        
                        pnt = i + 1 ;
                        state = 2 ;
                        
                    }
                    else if( state == 2 && cd == ')' ){
                        
                        port |= (
                            Integer.parseInt(
                                data.substring( pnt,i )
                                ) & 0x000000ff
                            
                        ) ;
                        
                        break ;
                    }
                    
                }
                
                // 解析完了の場合.
                if( host != null && port != -1 ){
                    
                    // デバッグ出力!!.
                    this.outDebug( false,cmd ) ;
                    
                    // 実行コマンド設定.
                    m_writer.println( cmd ) ;
                    m_writer.flush() ;
                    
                    // PASV用コネクション処理を生成.
                    ret = new Socket( host,port ) ;
                    ret.setSoTimeout( timeout ) ;
                    
                    break ;
                    
                }
                else{
                    
                    tmp = data ;
                    
                    // 情報が存在しない場合.
                    if( ( data = this.execReturn() ) == null ){
                        
                        throw new AccessException(
                            "戻りデータ(" + tmp + ")の解析に失敗しました"
                        ) ;
                        
                    }
                    
                }
                
            }
            
        }catch( AccessException ac ){
            
            try{
                ret.close() ;
            }catch( Exception tt ){
            }
            ret = null ;
            throw ac ;
            
        }catch( Exception t ){
            
            try{
                ret.close() ;
            }catch( Exception tt ){
            }
            ret = null ;
            throw new AccessException( t ) ;
            
        }finally{
            host = null ;
            data = null ;
            tmp = null ;
            buf = null ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド実行.
     */
    private final String execCmd( String cmd ) throws AccessException
    {
        int i ;
        int len ;
        int rCd ;
        boolean flg ;
        
        String ret = null ;
        PrintWriter writer = null ;
        
        try{
            
            // 以前の結果処理をチェック.
            this.checkTelegram() ;
            
            synchronized( m_readSync.get() ){
                len = m_readBuf.size() ;
            }
            
            writer = m_writer ;
            
            // デバッグ出力!!.
            this.outDebug( false,cmd ) ;
            
            // コマンド送信.
            writer.println( cmd ) ;
            writer.flush() ;
            
            // コマンド結果が取得されたかチェック.
            for( i = 0,flg = false ; i < FTPDef.GET_MAXRETRY ; i ++ ){
                
                // 短期間待機.
                UtilCom.cpuCreate() ;
                
                synchronized( m_readSync.get() ){
                    if( len < m_readBuf.size() ){
                        flg = true ;
                        break ;
                    }
                }
                
            }
            
            // コマンド実施結果が取得されない場合.
            if( flg == false ){
                throw new AccessException(
                    "コマンド(" + cmd +
                    ")結果の取得がタイムアウトしました"
                ) ;
            }
            
            // 受信バッファを解析.
            ret = this.execReturn() ;
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            writer = null ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド結果受信.
     */
    private final String execReturn()
        throws AccessException
    {
        int rCd ;
        String ret = null ;
        
        try{
            
            // タイムラグをセット.
            UtilCom.cpuCreate() ;
            
            // 受信バッファから受信コードを取得.
            rCd = this.getReceiveCode() ;
            
            // 受信コードが存在する.
            if( rCd != -1 ){
                
                ret = ( String )m_tmp.remove( 0 ) ;
                
                if( FTPDef.getErrorLevel( rCd ) >= FTPDef.LEVEL_ERROR ){
                    throw new AccessException( "ERROR : CD:" + rCd + " MSG:" + ret ) ;
                }
                
            }
            else{
                ret = null ;
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド結果待ち.
     */
    private final String waitReturn()
        throws AccessException
    {
        int i ;
        int len ;
        int rCd ;
        
        String ret = null ;
        
        try{
            
            // データ長を取得.
            synchronized( m_readSync.get() ){
                len = m_readBuf.size() ;
            }
            
            // コマンド結果が取得されたかチェック.
            for( i = 0 ; i < FTPDef.GET_RETRY ; i ++ ){
                
                // 短期間待機.
                UtilCom.cpuCreate() ;
                
                synchronized( m_readSync.get() ){
                    
                    if( len < m_readBuf.size() ){
                        break ;
                    }
                    
                }
                
            }
            
            // タイムラグをセット.
            UtilCom.cpuCreate() ;
            
            // 受信バッファから受信コードを取得.
            rCd = this.getReceiveCode() ;
            
            // 受信コードが存在する.
            if( rCd != -1 ){
                
                ret = ( String )m_tmp.remove( 0 ) ;
                
                if( FTPDef.getErrorLevel( rCd ) >= FTPDef.LEVEL_ERROR ){
                    throw new AccessException( "ERROR : CD:" + rCd + " MSG:" + ret ) ;
                }
                
            }
            else{
                ret = null ;
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }
        
        return ret ;
    }
    
    /**
     * コマンド要求を空に設定.
     */
    private final void checkTelegram()
        throws AccessException
    {
        AccessException out = null ;
        
        // 結果処理をチェック.
        for( ;; ){
            try{
                if( this.execReturn() == null ){
                    break ;
                }
            }catch( AccessException ac ){
                out = ac ;
            }
        }
        
        if( out != null ){
            throw out ;
        }
        
    }
    
    /**
     * 受信電文からコードを取得.
     * 取得されなかった場合は[-1].
     */
    private final int getReceiveCode()
    {
        int i ;
        int len ;
        int pnt ;
        int ret ;
        char code ;
        
        String data = null ;
        
        try{
            
            synchronized( m_readSync.get() ){
                data = ( String )m_readBuf.remove( 0 ) ;
            }
            
            // 受信バッファが存在しない場合.
            if( data == null ){
                ret = -1 ;
            }
            else{
                
                // デバッグ出力!!.
                this.outDebug( true,data ) ;
                
                data = UtilCom.trimPlus( data ) ;
                len = data.length() ;
                
                for( i = 0,pnt = -1 ; i < len ; i ++ ){
                    
                    code = data.charAt( i ) ;
                    if( code >= '0' && code <= '9' ){
                        
                        pnt = i ;
                        
                    }
                    else{
                        break ;
                    }
                    
                }
                
                if( pnt != -1 ){
                    pnt += 1 ;
                    ret = Integer.parseInt( data.substring( 0,pnt ) ) ;
                    m_tmp.add( UtilCom.trimPlus( data.substring( pnt ) ) ) ;
                }
                else{
                    ret = -1 ;
                }
                
            }
            
        }catch( Exception t ){
            ret = -1 ;
        }finally{
            data = null ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 電文受信処理.
     */
    private final ArrayList recvTelegram( InputStream in,String charset )
    {
        
        ArrayList ret = null ;
        ByteArrayOutputStream bufAry = null ;
        
        try{
            
            bufAry = new ByteArrayOutputStream() ;
            
            if( this.ioTelegram( in,bufAry ) > 0 ){
                if( charset == null ){
                    ret = UtilCom.pauseEnter( new String( bufAry.toByteArray() ) ) ;
                }
                else{
                    ret = UtilCom.pauseEnter( new String( bufAry.toByteArray(),charset ) ) ;
                }
            }
            else{
                ret = null ;
            }
            
        }catch( Exception t ){
            
            if( ret != null ){
                ret.clear() ;
            }
            ret = null ;
            
        }finally{
            
            try{
                in.close() ;
            }catch( Exception tt ){
            }
            try{
                bufAry.close() ;
            }catch( Exception tt ){
            }
            
            bufAry = null ;
        }
        
        return ret ;
    }
    
    /**
     * 電文I/O処理.
     */
    private final long ioTelegram( InputStream in,OutputStream out )
    {
        int len ;
        long ret ;
        
        byte[] tmp = null ;
        
        if( in == null || out == null ){
            return -1L ;
        }
        
        try{
            
            tmp = new byte[ FTPDef.BUF_LENGTH ] ;
            
            ret = 0 ;
            while( ( len = in.read( tmp ) ) >= 0 ){
                ret += len ;
                out.write( tmp,0,len ) ;
            }
            
        }catch( Exception t ){
            ret = -1L ;
        }finally{
            
            try{
                in.close() ;
            }catch( Exception tt ){
            }
            try{
                out.close() ;
            }catch( Exception tt ){
            }
            
            tmp = null ;
            out = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * リスト情報を取得し、その情報を解析.
     */
    private final void getList( String cmd )
        throws AccessException
    {
        int i ;
        int len ;
        int lenJ ;
        int own ;
        long time ;
        long size ;
        char dirChr ;
        boolean dirFlg ;
        
        String name = null ;
        String tmp = null ;
        ArrayList pause = null ;
        ArrayList lst = null ;
        CharTable tbl = null ;
        FTPPause listPause = null ;
        
        try{
            
            tbl = m_table ;
            tbl.clear() ;
            
            lst = this.doLs( cmd ) ;
            len = lst.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                tmp = ( String )lst.get( i ) ;
                pause = UtilCom.pauseString( tmp," \t" ) ;
                
                // 情報を区分け.
                if( pause != null ){
                    
                    own = 0 ;
                    dirFlg = false ;
                    
                    lenJ = pause.size() ;
                    name = ( String )pause.get( lenJ-1 ) ;
                    tmp = ( String )pause.get( 0 ) ;
                    
                    pause.clear() ;
                    pause = null ;
                    
                    // 通常のファイル名である場合.
                    if( name.equals( "." ) == false && name.equals( ".." ) == false ){
                        
                        // 対象権限を取得.
                        lenJ = tmp.length() ;
                        if( lenJ == 10 ){
                            
                            dirChr = tmp.charAt( 0 ) ;
                            if( dirChr != '-' ){
                                dirFlg = true ;
                            }
                            own = ( int )(
                                ( ( tmp.charAt( 1 ) != '-' ) ? 0x00000400 : 0x00000000 ) |
                                ( ( tmp.charAt( 2 ) != '-' ) ? 0x00000200 : 0x00000000 ) |
                                ( ( tmp.charAt( 3 ) != '-' ) ? 0x00000100 : 0x00000000 ) |
                                ( ( tmp.charAt( 4 ) != '-' ) ? 0x00000040 : 0x00000000 ) |
                                ( ( tmp.charAt( 5 ) != '-' ) ? 0x00000020 : 0x00000000 ) |
                                ( ( tmp.charAt( 6 ) != '-' ) ? 0x00000010 : 0x00000000 ) |
                                ( ( tmp.charAt( 7 ) != '-' ) ? 0x00000004 : 0x00000000 ) |
                                ( ( tmp.charAt( 8 ) != '-' ) ? 0x00000002 : 0x00000000 ) |
                                ( ( tmp.charAt( 9 ) != '-' ) ? 0x00000001 : 0x00000000 )
                            ) ;
                            
                        }
                        
                        // 取得対象がファイルの場合.
                        if( dirFlg == false ){
                            
                            // 対象ファイル時間を取得.
                            time = this.doDateTime( name ) ;
                            
                            // 対象ファイル長を取得.
                            size = this.doFileSize( name ) ;
                            
                            // リスト要素を追加.
                            listPause = new FTPPause( dirFlg,own,size,time ) ;
                            
                        }
                        // 取得対象がディレクトリの場合.
                        else{
                            
                            try{
                                // 対象ファイル時間を取得.
                                time = this.doDateTime( name ) ;
                            }catch( Exception t ){
                                time = 0L ;
                            }
                            
                            // 対象ファイル長なし.
                            size = 0L ;
                            
                            // リスト要素を追加.
                            listPause = new FTPPause( dirFlg,own,size,time ) ;
                            
                        }
                        
                        // リスト条件を追加.
                        tbl.add( name,listPause ) ;
                        listPause = null ;
                        
                    }
                    
                }
            }
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
            throw new AccessException( t ) ;
        }finally{
            
            if( pause != null ){
                pause.clear() ;
            }
            if( lst != null ){
                lst.clear() ;
            }
            
            pause = null ;
            lst = null ;
            name = null ;
            tmp = null ;
            tbl = null ;
            listPause = null ;
            
        }
        
    }
    
    /**
     * 指定されたディレクトリ位置に変更.
     */
    private final String changeDirectory( String change )
        throws AccessException
    {
        int i ;
        int len ;
        
        String ret = null ;
        StringBuffer buf = null ;
        FTPPause pause = null ;
        ArrayList lst = null ;
        
        if( change == null ){
            return null ;
        }
        
        try{
            
            // 設定条件を解析.
            change = UtilCom.trimPlus( change ) ;
            lst = UtilCom.pauseString( change,"/" ) ;
            
            // 解析結果が存在する場合.
            if( lst != null ){
                
                // 現在のディレクトリ情報を取得.
                m_beforeDir = this.doPwd() ;
                len = lst.size() - 1 ;
                buf = new StringBuffer() ;
                
                // 絶対パスの場合.
                if( change.charAt( 0 ) == '/' ){
                    
                    buf.append( "/" ) ;
                    
                }
                // 相対パスの場合.
                else{
                    
                    // 現在の位置を設定.
                    buf.append( m_beforeDir ) ;
                    
                }
                
                // 解析データをセット.
                for( i = 0 ; i < len ; i ++ ){
                    
                    buf.append( ( String )lst.get( i ) ) ;
                    buf.append( "/" ) ;
                    
                }
                
                // 最後の設定名を取得.
                ret = ( String )lst.get( len ) ;
                
                // 設定された値から最後の設定名を引いた値でリストを移動.
                this.doCd( buf.toString() ) ;
                this.getList( null ) ;
                
                // 最後の設定名がリスト内に存在するかチェック.
                if( ( pause = ( FTPPause )m_table.get( ret ) ) != null ){
                    
                    // 最後の設定名がディレクトリの場合.
                    if( pause.getMode() == true ){
                        
                        // 最後の設定名に移動.
                        buf.append( ret ) ;
                        this.doCd( buf.toString() ) ;
                        this.getList( null ) ;
                        buf.append( "/" ) ;
                        
                        ret = null ;
                        
                    }
                    
                }
                
            }
            // ROOT位置設定の場合.
            else if( change.charAt( 0 ) == '/' ){
                
                // 現在のディレクトリ情報を取得.
                m_beforeDir = this.doPwd() ;
                
                // ROOT位置に移動.
                this.doCd( "/" ) ;
                this.getList( null ) ;
                ret = null ;
                
            }
            // 情報が不正な場合.
            else{
                throw new AccessException(
                    "指定されたリスト(" + change +
                    ")は不正です"
                ) ;
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
        }finally{
            
            if( lst != null ){
                lst.clear() ;
            }
            buf = null ;
            pause = null ;
            lst = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * this.changeDirectory()移動前の条件に移動.
     */
    private final void beforeDirectory()
        throws AccessException
    {
        if( m_beforeDir != null ){
            
            this.doCd( m_beforeDir ) ;
            this.getList( null ) ;
            
            m_beforeDir = null ;
            
        }
    }
    
    /**
     * 指定パス名の最後の名前を取得.
     */
    private final String getLastName( String name )
    {
        String ret = null ;
        ArrayList lst = null ;
        
        if( name != null ){
            
            try{
                
                name = UtilCom.trimPlus( name ) ;
                lst = UtilCom.pauseString( name,"/" ) ;
                
                if( lst.size() > 0 ){
                    ret = ( String )lst.get( lst.size()-1 ) ;
                }
                
            }catch( Exception t ){
                ret = null ;
            }finally{
                
                if( lst != null ){
                    lst.clear() ;
                }
                
                lst = null ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 現在のリスト条件を取得.
     */
    private final String[] getNowList( boolean mode )
    {
        int i ;
        int len ;
        
        String[] tmp = null ;
        CharTable tbl = null ;
        FTPPause pause = null ;
        ObjectArray ary = null ;
        String[] ret = null ;
        
        
        try{
            
            tbl = m_table ;
            
            tmp = tbl.getNames() ;
            len = tmp.length ;
            ary = new ObjectArray() ;
            
            for( i = 0 ; i < len ; i ++ ){
                pause = ( FTPPause )tbl.get( tmp[ i ] ) ;
                if( pause.getMode() == mode ){
                    ary.add( tmp[ i ] ) ;
                }
                tmp[ i ] = null ;
                pause = null ;
            }
            
            if( ( len = ary.size() ) > 0 ){
                ret = new String[ len ] ;
                System.arraycopy( ary.getObjects(),0,ret,0,len ) ;
            }
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            if( ary != null ){
                ary.clear() ;
            }
            tmp = null ;
            tbl = null ;
            pause = null ;
            ary = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * デバッグ情報を出力.
     */
    private final void outDebug( boolean mode,String cmd )
    {
        // デバッグ出力!!.
        if( m_isDebug == true ){
            StringBuffer buf = null ;
            buf = new StringBuffer() ;
            buf.append( FTPClient.getNowDate() ) ;
            if( mode == true ){
                buf.append( " FTP-OUT(" ) ;
            }
            else{
                buf.append( " FTP-IN (" ) ;
            }
            buf.append( m_socket.getInetAddress() ) ;
            buf.append( ":" ) ;
            buf.append( m_socket.getPort() ) ;
            buf.append( ") [" ) ;
            buf.append( cmd ) ;
            buf.append( "] " ) ;
            
            m_debugWriter.println( buf.toString() ) ;
            buf = null ;
        }
    }
    
    /**
     * 出力用日付を生成.
     */
    private static final String getNowDate()
    {
        int i ;
        int len ;
        
        String[] days = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        len = UtilCom.TIMESTAMP_STRING_LENGTH ;
        days = new String[ len ] ;
        UtilCom.getNowTimestampByString( days ) ;
        
        buf = new StringBuffer() ;
        
        buf.append( days[ 0 ] ) ;
        buf.append( "/" ) ;
        
        buf.append( "00".substring(  days[ 1 ].length()) ) ;
        buf.append( days[ 1 ] ) ;
        buf.append( "/" ) ;
        
        buf.append( "00".substring(  days[ 2 ].length()) ) ;
        buf.append( days[ 2 ] ) ;
        buf.append( " " ) ;
        
        buf.append( "00".substring(  days[ 3 ].length()) ) ;
        buf.append( days[ 3 ] ) ;
        buf.append( ":" ) ;
        
        buf.append( "00".substring(  days[ 4 ].length()) ) ;
        buf.append( days[ 4 ] ) ;
        buf.append( ":" ) ;
        
        buf.append( "00".substring(  days[ 5 ].length()) ) ;
        buf.append( days[ 5 ] ) ;
        buf.append( "." ) ;
        
        buf.append( "000".substring(  days[ 6 ].length()) ) ;
        buf.append( days[ 6 ] ) ;
        
        ret = buf.toString() ;
        
        for( i = 0 ; i < len ; i ++ ){ days[ i ] = null ; }
        days = null ;
        buf = null ;
        
        return ret ;
    }
    
}

/**
 * 受信データバッファ.
 */
class ReaderBuffer implements Runnable
{
    
    BufferedReader m_reader = null ;
    ObjectArray m_array = null ;
    Synchronized m_read2Sync = null ;
    
    public ReaderBuffer( BufferedReader in,ObjectArray ary,Synchronized sync )
    {
        m_reader = in ;
        m_array = ary ;
        m_read2Sync = sync ;
    }
    
    public void run(){
        
        String tmp = null ;
        
        // 情報取得ループ.
        while(true){
            
            try{
                
                tmp = null ;
                tmp = m_reader.readLine() ;
                
                synchronized( m_read2Sync.get() ){
                    if( tmp != null && tmp.length() > 0 ){
                        m_array.add( tmp ) ;
                    }
                }
                
            }catch( InterruptedIOException ii ){
                // タイムアウトの場合リトライ.
            }catch( IOException io ){
                // ソケットクローズ等のエラー.
                break ;
            }catch( Exception t ){
                // その他のエラー.
                break ;
            }
            
        }
        
        m_reader = null ;
        m_array = null ;
        m_read2Sync = null ;
        
    }
    
}

