/*
 * @(#)MgcServerValue.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net.mgc ;

import java.io.Serializable;
import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.serialize.SerializeUtil;


/**
 * サーバ用マルチキャストグループ要素.
 * <BR><BR>
 * サーバ用マルチキャストグループ要素です.
 *  
 * @version 1.0.0 2005/07/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class MgcServerValue implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            MgcServerValue.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * サーバ名.
     */
    private String m_serverName = null ;
    
    /**
     * サーバID.
     */
    private int m_serverID = -1 ;
    
    /**
     * プロトコルタイプを設定.
     */
    private int m_protocolType = -1 ;
    
    /**
     * バインドアドレス.
     */
    private InetAddress m_addr = null ;
    
    /**
     * バインドポート.
     */
    private int m_port = -1 ;
    
    /**
     * 現在のコネクションカウント数を取得.
     */
    private int m_nowConnectCnt = -1 ;
    
    /**
     * MgcServer標準時間.
     */
    private long m_serverTime = -1L ;
    
    
    
    
    /**
     * コンタストラクタ.
     */
    public MgcServerValue()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param id 対象のサーバIDを設定します.
     * @param ptype 対象のプロトコルタイプを設定します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 通知対象のバインドアドレスを設定します.<BR>
     *             バインドアドレスを設定しない場合は[null]を設定してください.
     * @param port 通知対象のバインドポート番号を設定します.
     * @param connCnt 現在のコネクションカウント数を取得します.
     * @param time 対象のサーバ時間を設定します.
     * @exception InputException 入力例外.
     */
    public MgcServerValue( String name,int id,int ptype,InetAddress addr,int port,int connCnt,long time )
        throws InputException
    {
        try{
            this.create( name,id,ptype,addr.getHostAddress(),port,connCnt,time ) ;
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param id 対象のサーバIDを設定します.
     * @param ptype 対象のプロトコルタイプを設定します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 通知対象のバインドアドレスを設定します.<BR>
     *             バインドアドレスを設定しない場合は[null]を設定してください.
     * @param port 通知対象のバインドポート番号を設定します.
     * @param connCnt 現在のコネクションカウント数を取得します.
     * @param time 対象のサーバ時間を設定します.
     * @exception InputException 入力例外.
     */
    public MgcServerValue( String name,int id,int ptype,String addr,int port,int connCnt,long time )
        throws InputException
    {
        this.create( name,id,ptype,addr,port,connCnt,time ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param id 対象のサーバIDを設定します.
     * @param ptype 対象のプロトコルタイプを設定します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 通知対象のバインドアドレスを設定します.<BR>
     *             バインドアドレスを設定しない場合は[null]を設定してください.
     * @param port 通知対象のバインドポート番号を設定します.
     * @param connCnt 現在のコネクションカウント数を取得します.
     * @param time 対象のサーバ時間を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String name,int id,int ptype,InetAddress addr,int port,int connCnt,long time )
        throws InputException
    {
        try{
            this.create( name,id,ptype,addr.getHostAddress(),port,connCnt,time ) ;
        }catch( Exception e ){
            throw new InputException( e ) ;
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 条件を設定してオブジェクトを生成します.
     * <BR>
     * @param name 対象のサーバ名を設定します.
     * @param id 対象のサーバIDを設定します.
     * @param ptype 対象のプロトコルタイプを設定します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_TCP]を設定した場合、TCP/IPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_UDP]を設定した場合、UDPを示します.<BR>
     *              [MgcValue.PROTOCOL_TYPE_MCAST]を設定した場合、Multicastを示します.
     * @param addr 通知対象のバインドアドレスを設定します.<BR>
     *             バインドアドレスを設定しない場合は[null]を設定してください.
     * @param port 通知対象のバインドポート番号を設定します.
     * @param connCnt 現在のコネクションカウント数を取得します.
     * @param time 対象のサーバ時間を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( String name,int id,int ptype,String addr,int port,int connCnt,long time )
        throws InputException
    {
        if(
            name == null || name.length() <= 0 ||
            id < 0 || port <= 0 || port >= 65535 ||
            (
                ptype != MgcValue.PROTOCOL_TYPE_TCP &&
                ptype != MgcValue.PROTOCOL_TYPE_UDP &&
                ptype != MgcValue.PROTOCOL_TYPE_MCAST
            )
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            m_serverName = name ;
            m_serverID = id ;
            m_protocolType = ptype ;
            m_addr = InetAddress.getByName( addr ) ;
            m_port = port ;
            m_nowConnectCnt = connCnt ;
            m_serverTime = time ;
        }catch( Exception e ){
            this.clear() ;
            throw new InputException( e ) ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_serverName = null ;
        m_serverID = -1 ;
        m_protocolType = -1 ;
        m_addr = null ;
        m_port = -1 ;
        m_nowConnectCnt = -1 ;
        m_serverTime = -1L ;
    }
    
    /**
     * 設定されているサーバ名を取得.
     * <BR><BR>
     * 設定されているサーバ名を取得します.
     * <BR>
     * @return String 設定されているサーバ名が返されます.<BR>
     *                [null]が返された場合、このオブジェクトは無効です.
     */
    public final String getServerName()
    {
        return m_serverName ;
    }
    
    /**
     * 設定されているサーバIDを取得.
     * <BR><BR>
     * 設定されているサーバIDを取得します.
     * <BR>
     * @return int 設定されているサーバIDが返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getServerID()
    {
        return m_serverID ;
    }
    
    /**
     * 設定されているプロトコルタイプを取得.
     * <BR><BR>
     * 設定されているプロトコルタイプを取得します.
     * <BR>
     * @return int 設定されているプロトコルタイプが返されます.<BR>
     *             [MgcServerValue.PROTOCOL_TYPE_TCP]が返された場合、TCP/IPを示します.<BR>
     *             [MgcServerValue.PROTOCOL_TYPE_UDP]が返された場合、UDPを示します.<BR>
     *             [MgcServerValue.PROTOCOL_TYPE_MCAST]が返された場合、Multicastを示します.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getProtocolType()
    {
        return m_protocolType ;
    }
    
    /**
     * 設定されているバインドアドレスを取得.
     * <BR><BR>
     * 設定されているバインドアドレスを取得します.
     * <BR>
     * @return InetAddress 設定されているバインドアドレスが返されます.<BR>
     *                     [null]が返された場合、このオブジェクトは無効です.
     */
    public final InetAddress getBindAddress()
    {
        return m_addr ;
    }
    
    /**
     * 設定されているバインドポートを取得.
     * <BR><BR>
     * 設定されているバインドポート番号を取得します.
     * <BR>
     * @return int 設定されているバインドポート番号が返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getBindPort()
    {
        return m_port ;
    }
    
    /**
     * 設定されているバインド情報を取得.
     * <BR><BR>
     * 設定されているバインド情報を取得します.
     * <BR>
     * @param out 取得先のオブジェクトを設定します.
     * @return boolean 取得結果が返されます.<BR>
     *                 [true]が返された場合、正常に取得できました.<BR>
     *                 [false]が返された場合、取得出来ませんでした.
     */
    public final boolean getBind( ConnectAddress out )
    {
        boolean ret ;
        
        ret = false ;
        if( out != null ){
            
            try{
                if( m_serverName != null ){
                    out.create( m_addr,m_port ) ;
                    ret = true ;
                }
                else{
                    out.clear() ;
                }
            }catch( Exception e ){
                out.clear() ;
                ret = false ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 現在のコネクションカウントを取得.
     * <BR><BR>
     * 現在のコネクションカウントを取得します.
     * <BR>
     * @return int 現在のコネクションカウントが返されます.<BR>
     *             [-1]が返された場合、このオブジェクトは無効です.
     */
    public final int getConnectCount()
    {
        return m_nowConnectCnt ;
    }
    
    /**
     * 対象のマシン時間を取得.
     * <BR><BR>
     * 対象のマシン時間を取得します.
     * <BR>
     * @return long 対象のマシン時間が返されます.<BR>
     *              [-1L]が返された場合、このオブジェクトは無効です.
     */
    public final long getTime()
    {
        return m_serverTime ;
    }
    
    /**
     * このオブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public final boolean isUse()
    {
        if( m_serverName != null ){
            return true ;
        }
        else{
            return false ;
        }
    }
    
    
}
