/*
 * @(#)BinResourceRoll.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource ;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.Roll;


/**
 * バイナリリソースロール.
 * <BR><BR>
 * バイナリリソース用のロール管理を行うオブジェクトです.
 *
 * @version     1.00, 2005/07/27
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BinResourceRoll implements Roll
{
    /**
     * デフォルト管理数.
     */
    private static final int DEF_SIZE = 32 ;
    
    
    
    /**
     * 送信ロール配列.
     */
    private BinResource[] m_roll = null ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType m_resType = null ;
    
    /**
     * ロール管理最大数.
     */
    private int m_max = 0 ;
    
    /**
     * 現在位置.
     */
    private int m_now = 0 ;
    
    /**
     * 格納数.
     */
    private int m_nowLen = 0 ;
    
    /**
     * 追加予約フラグ.
     */
    private int m_addResCnt = 0 ;
    
    /**
     * コンストラクタ.
     */
    private BinResourceRoll(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロール管理数を設定して生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param max ロール管理数を設定します.<BR>
     *            設定値は[32]以下を設定した場合、その値となります.
     * @exception InputException 入力例外.
     */
    public BinResourceRoll( ResourceType resType,int max )
        throws InputException
    {
        int i ;
        BinResource[] roll = null ;
        
        if( resType == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        max = ( max <= DEF_SIZE ) ? DEF_SIZE : max ;
        
        roll = new BinResource[ max ] ;
        
        for( i = 0 ; i < max ; i ++ ){
            roll[ i ] = null ;
        }
        
        m_roll = roll ;
        m_resType = resType ;
        m_max = max ;
        m_now = 0 ;
        m_nowLen = 0 ;
        m_addResCnt = 0 ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        this.destroy() ;
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy()
    {
        int i ;
        int len ;
        
        BinResource[] roll = null ;
        
        len = m_max ;
        
        if( len > 0 ){
            
            roll = m_roll ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( roll[ i ] != null ){
                    roll[ i ].clear() ;
                }
                roll[ i ] = null ;
            }
            
        }
        
        m_roll = null ;
        m_resType = null ;
        m_max = 0 ;
        m_now = 0 ;
        m_nowLen = 0 ;
        m_addResCnt = 0 ;
    }
    
    /**
     * 追加予約をONにセット.
     * <BR><BR>
     * 追加予約をONにセットします.
     */
    public final void addReservationByON()
    {
        synchronized( this ){
            m_addResCnt ++ ;
        }
    }
    
    /**
     * 追加予約をOFFにセット.
     * <BR><BR>
     * 追加予約をOFFにセットします.
     */
    public final void addReservationByOFF()
    {
        synchronized( this ){
            m_addResCnt = ( m_addResCnt <= 0 ) ? 0 : m_addResCnt - 1 ;
        }
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( byte[] value )
        throws InputException
    {
        return this.addTo( false,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最後のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean add( BinResource value )
        throws InputException
    {
        return this.addTo( false,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( byte[] value )
        throws InputException
    {
        return this.addTo( true,value ) ;
    }
    
    /**
     * 情報追加.
     * <BR><BR>
     * 対象情報を追加します.<BR>
     * この処理は最初のロールに情報を追加します.
     * <BR>
     * @param value 対象の情報を設定します.
     * @return boolean 設定の合否が返されます.<BR>
     *                 [true]が返された場合、正しく設定されました.<BR>
     *                 [false]が返された場合、空き情報が存在しないことから、
     *                 正しく設定できませんでした.
     * @exception InputException 入力例外.
     */
    public final boolean addHead( BinResource value )
        throws InputException
    {
        return this.addTo( true,value ) ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報をバイナリ(byte[])で取得します.
     * <BR>
     * return byte[] バイナリ情報が返されます.<BR>
     *               [null]が返された場合、情報は存在しません.
     */
    public final byte[] getBinary()
    {
        int pnt ;
        
        BinResource[] roll = null ;
        byte[] ret = null ;
        
        synchronized( this ){
            
            if( m_nowLen <= 0 ){
                ret = null ;
            }
            else{
                
                roll = m_roll ;
                
                if( ( pnt = m_now - m_nowLen ) < 0 ){
                    pnt = ( m_max + pnt ) ;
                }
                
                ret = roll[ pnt ].getBinary() ;
                roll[ pnt ].reset() ;
                
                m_nowLen -- ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の情報をバイナリリソース(BinResource)で取得します.
     * <BR>
     * return BinResource バイナリリソース情報が返されます.<BR>
     *                    [null]が返された場合、情報は存在しません.
     */
    public final BinResource getBinResource()
    {
        int pnt ;
        
        BinResource[] roll = null ;
        BinResource ret = null ;
        
        synchronized( this ){
            
            if( m_nowLen <= 0 ){
                ret = null ;
            }
            else{
                
                roll = m_roll ;
                
                if( ( pnt = m_now - m_nowLen ) < 0 ){
                    pnt = ( m_max + pnt ) ;
                }
                
                ret = roll[ pnt ] ;
                roll[ pnt ] = null ;
                
                m_nowLen -- ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 対象の管理サイズを取得.
     * <BR><BR>
     * 対象の管理サイズが返されます.
     * <BR>
     * @return int 管理サイズが返されます.
     */
    public final int getMax()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_max ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の格納数を取得.
     * <BR><BR>
     * 現在の格納数を取得します.
     * <BR>
     * @return int 現在の格納数が返されます.
     */
    public final int getSize()
    {
        int ret ;
        
        synchronized( this ){
            ret = m_nowLen ;
        }
        
        return ret ;
    }
    
    /**
     * データが追加できるかチェック.
     * <BR><BR>
     * データが追加できるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、設定可能です.<BR>
     *                 [false]が返された場合、ロールは満杯のため設定はできません.
     */
    public final boolean isAdd()
    {
        boolean ret ;
        
        synchronized( this ){
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * 追加予約が行われているかチェック.
     * <BR><BR>
     * 追加予約が行われているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、追加予約されています.<BR>
     *                 [false]が返された場合追加予約はされていません.
     */
    public final boolean isAddReservation()
    {
        boolean ret ;
        
        synchronized( this ){
            ret = ( m_addResCnt > 0 ) ? true : false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,byte[] value )
        throws InputException
    {
        int next ;
        boolean ret ;
        
        BinResource[] roll = null ;
        
        if( value == null || value.length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        synchronized( this ){
            
            // データ格納領域が存在しない場合.
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // 前回のデータが存在しないか、前回のデータ長と
                // 今回設定するデータ長*係数より大きい場合.
                if(
                    roll[ next ] == null ||
                    roll[ next ].getAllSize() > Resource.getRenewRollValueCode( value.length )
                )
                {
                    // 前回のデータが存在する場合.
                    if( roll[ next ] != null ){
                        roll[ next ].clear() ;
                        roll[ next ] = null ;
                    }
                    // 新たに生成.
                    roll[ next ]= Resource.createBinResource( m_resType,value ) ;
                }
                // 再利用可能な場合.
                else{
                    roll[ next ].reset() ;
                    roll[ next ].setBinary( 0,value ) ;
                }
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 条件を追加.
     */
    private final boolean addTo( boolean hlMode,BinResource value )
        throws InputException
    {
        int next ;
        boolean ret ;
        
        BinResource[] roll = null ;
        
        if( value == null || value.isUse() == false || value.size() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = false ;
        
        synchronized( this ){
            
            // データ格納領域が存在しない場合.
            if( ( m_nowLen + m_addResCnt ) >= m_max ){
                ret = false ;
            }
            else{
                
                // 追加ポイントを取得.
                next = this.getAddPoint( hlMode ) ;
                
                // ロールオブジェクトを取得.
                roll = m_roll ;
                
                // 情報をセット.
                if( roll[ next ] != null ){
                    roll[ next ].clear() ;
                    roll[ next ] = null ;
                }
                roll[ next ] = value ;
                
                ret = true ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 追加位置を取得.
     */
    private final int getAddPoint( boolean hlMode )
    {
        int ret ;
        
        ret = m_now ;
        
        // 先頭に追加する場合.
        if( hlMode == true ){
            
            // データが存在する場合.
            if( m_nowLen > 0 ){
                if( ( ret = m_now - m_nowLen ) < 0 ){
                    ret = ( m_max + ret ) ;
                }
                ret -- ;
                if( ret < 0 ){
                    ret = m_max - 1 ;
                }
            }
            // データが存在しない場合.
            else{
                if( ret >= m_max ){
                    ret = 0 ;
                }
                m_now = ret + 1 ;
            }
            
            m_nowLen ++ ;
            
        }
        // 最後に追加する場合.
        else{
            
            if( ret >= m_max ){
                ret = 0 ;
            }
            
            m_nowLen ++ ;
            m_now = ret + 1 ;
            
        }
        
        return ret ;
    }
}

