/*
 * @(#)CacheIOImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource.cache;

import org.maachang.commons.exception.InputException;

/**
 * キャッシュI/Oオブジェクト.
 * <BR><BR>
 * 1つのキャッシュ情報を管理するI/Oオブジェクトです.
 *  
 * @version 2006/08/14
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class CacheIOImple implements CacheIO {
    
    /**
     * セクタI/Oオブジェクト.
     */
    private SectorIO sectorIO = null ;
    
    
    /**
     * コンストラクタ.
     */
    private CacheIOImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * セクタI/Oオブジェクトを設定して、キャッシュ管理
     * オブジェクトを生成します.
     * <BR>
     * @param sectorIO 対象のセクタI/Oオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public CacheIOImple( SectorIO sectorIO )
        throws InputException {
        
        if( sectorIO == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.sectorIO = sectorIO ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.destroy() ;
        } catch ( Exception t ) {
            
        }
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public void destroy() {
        
        if( sectorIO != null ) {
            sectorIO.destroy() ;
        }
        sectorIO = null ;
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @return Cache 対象のキャッシュオブジェクトが返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     */
    public Cache newCacheObject()
        throws NotUseSectorException {
        
        return this.newCacheObject( 0 ) ;
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @param newBytesLength 初期利用のバイト数を設定します.
     * @return Cache 対象のキャッシュオブジェクトが返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     */
    public Cache newCacheObject( int newBytesLength )
        throws NotUseSectorException {
        
        int id = -1 ;
        int sector = 0 ;
        boolean err = false ;
        
        CacheImple ret = null ;
        
        if( sectorIO != null ) {
            
            try{
                
                newBytesLength = ( newBytesLength <= 0 ) ? 1 : newBytesLength ;
                sector = ( ( ( newBytesLength & ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
                    ( ( newBytesLength & (~ELEMENT_MASK) ) >> ELEMENT_BITCOUNT ) ;
                
                if( ( id = sectorIO.reservationID() ) < 0 ) {
                    throw new NotUseSectorException( "予約可能なIDは存在しません" ) ;
                }
                
                if( ( sector = sectorIO.reservationSector( id,sector ) ) <= 0 ) {
                    throw new NotUseSectorException( "空きセクタは存在しません" ) ;
                }
                
                ret = new CacheImple( id,sectorIO ) ;
                err = false ;
                
            } catch( InputException in ) {
                err = true ;
                throw new NotUseSectorException( "空きセクタは存在しません" ) ;
            } catch( NotUseSectorException nu ) {
                err = true ;
                throw nu ;
            } finally {
                if( err == true ) {
                    if( id != -1 ) {
                        try{
                            sectorIO.releaseID( id ) ;
                        } catch( Exception ee ) {
                        }
                    }
                }
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * セクタI/Oオブジェクトを取得.
     * <BR><BR>
     * セクタI/Oオブジェクトを取得します.
     * <BR>
     * @return SectorIO 管理されているセクタI/Oオブジェクトが返されます.
     */
    public SectorIO getSectorIO() {
        
        return sectorIO ;
        
    }
    
    /**
     * 管理セクタ数を取得.
     * <BR><BR>
     * 管理セクタ数を取得します.
     * <BR>
     * @return int 管理セクタ数を取得します.
     */
    public int getMaxSector() {
        
        return sectorIO.getMaxSector() ;
        
    }
    
    /**
     * 現在利用中のセクタ数を取得.
     * <BR><BR>
     * 現在利用中のセクタ数を取得します.
     * <BR>
     * @return int 現在利用中のセクタ数が返されます.
     */
    public int getUseSector() {
        
        return sectorIO.getUseSector() ;
        
    }
    
    /**
     * ユニークIDを設定.
     * <BR><BR>
     * ユニークIDを再設定します.
     * <BR>
     * @param id 再設定対象のユニークIDを設定します.
     */
    public void setUniqueID( long id ) {
        
        if( sectorIO != null ) {
            
            sectorIO.setUniqueID( id ) ;
            
        }
        
    }
    
    /**
     * ユニークIDを取得.
     * <BR><BR>
     * このオブジェクトに対するユニークIDを取得します.
     * <BR>
     * @return long ユニークIDが返されます.
     */
    public long getUniqueID() {
        
        if( sectorIO != null ) {
            
            return sectorIO.getUniqueID() ;
            
        }
        
        return -1L ;
        
    }
    
    /**
     * このオブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public boolean isUseObject() {
        
        boolean ret ;
        
        try {
            ret = sectorIO.isUseObject() ;
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    
}

