/*
 * @(#)CacheImple.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource.cache;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;

/**
 * キャッシュオブジェクト.
 *  
 * @version 2006/08/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
class CacheImple implements Cache {
    
    /**
     * セクタID.
     */
    private int id = -1 ;
    
    /**
     * セクタI/O.
     */
    private SectorIO sectorIO = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    private CacheImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * キャッシュオブジェクトを生成します.
     * <BR>
     * @param sectorIO 対象のセクタIDを設定します.
     * @param sectorIO 対象のセクタI/Oオブジェクトを設定します.
     * @exception InputException 入力例外.
     */
    public CacheImple( int id,SectorIO sectorIO )
        throws InputException {
        
        if( id < 0 || sectorIO == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.id = id ;
        this.sectorIO = sectorIO ;
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.close() ;
        } catch ( Exception t ) {
            
        }
    }
    
    /**
     * オブジェクトクローズ.
     * <BR><BR>
     * オブジェクトをクローズします.
     */
    public synchronized void close() {
        
        if( sectorIO != null ) {
            
            try {
                sectorIO.releaseID( id ) ;
            } catch( Exception e ) {
            }
            
        }
        
        id = -1 ;
        sectorIO = null ;
        
    }
    
    /**
     * リセット.
     * <BR><BR>
     * 有効データ長を0にリセットします.
     */
    public synchronized void reset() {
        sectorIO.setUseIDByDataSize( id,0 ) ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param seek 読み込みポイント開始位置を設定します.
     * @return byte[] 読み込まれた情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized byte[] read( int seek ) throws AccessException {
        int len ;
        byte[] ret = null ;
        
        len = sectorIO.getUseIDByDataSize( id ) ;
        if( len <= 0 || len - seek <= 0 ) {
            return null ;
        }
        
        ret = new byte[ len - seek ] ;
        this.read( ret,seek,0,len ) ;
        
        return ret ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized int read( byte[] out,int seek )
        throws AccessException {
        if( out == null || out.length <= 0 ) {
            return -1 ;
        }
        return this.read( out,seek,0,out.length ) ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @param offset 読み込み格納のオフセット値を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized int read( byte[] out,int seek,int offset )
        throws AccessException {
        if( out == null || out.length <= 0 ) {
            return -1 ;
        }
        return this.read( out,seek,offset,out.length ) ;
    }
    
    /**
     * データ読み込み.
     * <BR><BR>
     * データを読み込みます.
     * <BR>
     * @param out 読み込み対象のデータを設定します.
     * @param seek 読み込みポイント開始位置を設定します.
     * @param offset 読み込み格納のオフセット値を設定します.
     * @param length 取得データ長を設定します.
     * @return int 読み込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized int read( byte[] out,int seek,int offset,int length )
        throws AccessException {
        
        int i ;
        int len ;
        int binOff ;
        int binLen ;
        int scLen ;
        int ret = -1 ;
        
        if(
            out == null || ( binLen = out.length ) <= 0 ||
            offset >= binLen
        ) {
            return -1 ;
        }
        
        // seek指定を調整.
        seek = ( seek <= 0 ) ? 0 : seek ;
        
        // 指定データ長をバイナリに合わせて調整.
        length = ( offset + length >= binLen ) ? binLen - offset : length ;
        
        // 現在のデータ長を取得.
        scLen = sectorIO.getUseIDByDataSize( id ) ;
        
        // 現在のデータ長に対して読み込み領域が超えた場合.
        if( ( length = ( seek + length >= scLen ) ? scLen - seek : length ) <= 0 ) {
            // 読み込みができない.
            return -1 ;
        }
        
        // 読み込みデータ長をセット.
        ret = length ;
        
        // 処理開始領域を取得.
        i = ( ( seek & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) ;
        len = ( ( ( ( length + seek ) & CacheIO.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( ( length + seek ) & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) ;
        
        // 処理開始時の１セクタに対する条件を取得.
        binOff = ( seek & CacheIO.ELEMENT_MASK ) ;
        binLen = ( length >= CacheIO.ELEMENT_LENGTH ) ? CacheIO.ELEMENT_LENGTH : length ;
        
        // 読み込み処理.
        for(
            ; i < len ;
            i ++,binOff = 0
        ) {
            
            binLen = sectorIO.readData( out,id,i,binOff,offset,binLen ) ;
            offset += binLen ;
            length -= binLen ;
            binLen = ( length >= CacheIO.ELEMENT_LENGTH ) ? CacheIO.ELEMENT_LENGTH : length ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public synchronized int write( byte[] in,int seek )
        throws NotUseSectorException,AccessException {
        if( in == null || in.length <= 0 ) {
            return -1 ;
        }
        return this.write( in,seek,0,in.length ) ;
    }
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @param offset 書き込み開始のオフセット値を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public synchronized int write( byte[] in,int seek,int offset )
        throws NotUseSectorException,AccessException {
        if( in == null || in.length <= 0 ) {
            return -1 ;
        }
        return this.write( in,seek,offset,in.length ) ;
    }
    
    /**
     * データ書き込み.
     * <BR><BR>
     * 対象データを書き込みます.
     * <BR>
     * @param in 書き込み対象のデータを設定します.
     * @param seek 書き込みポイント開始位置を設定します.
     * @param offset 書き込み開始のオフセット値を設定します.
     * @param length 書き込みデータ長を設定します.
     * @return int 書き込まれた情報長が返されます.
     * @exception NotUseSectorException セクタ取得失敗例外.
     * @exception AccessException アクセス例外.
     */
    public synchronized int write( byte[] in,int seek,int offset,int length )
        throws NotUseSectorException,AccessException {
        
        
        int i ;
        int len ;
        int binOff ;
        int binLen ;
        int scLen ;
        int newLen ;
        int ret = -1 ;
        
        if(
            in == null || ( binLen = in.length ) <= 0 ||
            offset >= binLen
        ) {
            return -1 ;
        }
        
        // seek指定を調整.
        seek = ( seek <= 0 ) ? 0 : seek ;
        
        // 指定データ長をバイナリに合わせて調整.
        length = ( offset + length >= binLen ) ? binLen - offset : length ;
        
        // 書き込み予定のセクタ領域を取得.
        newLen = ( ( ( ( seek + length ) & CacheIO.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( ( seek + length ) & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) ;
        
        // 書き込み予定のセクタ領域が、現在予約されているセクタ領域を越す場合.
        if( ( newLen = newLen - sectorIO.getReservationSectorSize( id ) ) > 0 ) {
            // 新しい領域を予約.
            try{
                if( sectorIO.reservationSector( id,newLen ) <= 0 ) {
                    throw new NotUseSectorException( "空きセクタは存在しません" ) ;
                }
            } catch( NotUseSectorException ne ) {
                throw ne ;
            } catch( Exception e ) {
                throw new AccessException( e ) ;
            }
        }
        
        // 現在のデータ領域を取得.
        scLen = sectorIO.getUseIDByDataSize( id ) ;
        
        // 書き込みデータ長を取得.
        ret = length ;
        
        // 処理開始領域を取得.
        i = ( ( seek & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) ;
        len = ( ( ( ( length + seek ) & CacheIO.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( ( length + seek ) & (~CacheIO.ELEMENT_MASK) ) >> CacheIO.ELEMENT_BITCOUNT ) ;
        
        // 処理開始時の１セクタに対する条件を取得.
        binOff = ( seek & CacheIO.ELEMENT_MASK ) ;
        binLen = ( length >= CacheIO.ELEMENT_LENGTH ) ? CacheIO.ELEMENT_LENGTH : length ;
        
        // 書き込み処理.
        for(
            ; i < len ;
            i ++,binOff = 0
        ) {
            
            binLen = sectorIO.writeData( in,id,i,binOff,offset,binLen ) ;
            offset += binLen ;
            length -= binLen ;
            binLen = ( length >= CacheIO.ELEMENT_LENGTH ) ? CacheIO.ELEMENT_LENGTH : length ;
            
        }
        
        // 新しい書き込みデータ長をセット.
        sectorIO.setUseIDByDataSize( id,( scLen <= ret + seek ) ? ret + seek : scLen ) ;
        
        return ret ;
        
    }
    
    /**
     * 現在の有効データ長を取得.
     * <BR><BR>
     * 現在の有効データ長を取得します.
     * <BR>
     * @return int 現在の有効データ長を取得します.
     */
    public synchronized int getLength() {
        
        return sectorIO.getUseIDByDataSize( id ) ;
        
    }
    
    /**
     * 現在確保しているセクタ数を取得.
     * <BR><BR>
     * 現在確保しているセクタ数を取得します.
     * <BR>
     * @return int 確保しているセクタ数が返されます.
     */
    public synchronized int getSector() {
        
        return sectorIO.getReservationSectorSize( id ) ;
        
    }
    
}

