/*
 * @(#)OneThread.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;

/**
 * スレッド処理実施.
 * <BR><BR>
 * 対象処理をスレッド処理で1度実施するためのオブジェクトです.
 * <BR>
 *
 * @version 1.0.0 2003/11/02
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class OneThread extends ProcessThread
{
    
    /**
     * スレッドスタートアップ名.
     */
    private static final String STARTUP_NAME = "startup" ;
    
    /**
     * スレッドシャットダウン名.
     */
    private static final String SHUTDOWN_NAME = "shutdown" ;
    
    
    /**
     * 実行処理.
     */
    private ExecutionThread m_exec = null ;
    
    /**
     * スレッド実施処理.
     */
    private final BaseThread m_thread = new BaseThread() ;
    
    /**
     * スレッドステータス.
     */
    private ThreadState m_state = null ;
    
    /**
     * スレッド実施時間.
     */
    private volatile long m_nowTime = 0L ;
    
    
    /**
     * スレッド同期用.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param exec 実施用オブジェクトを設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance( ExecutionThread exec )
        throws InputException,ExecutionException
    {
        return OneThread.createInstance( false,Thread.NORM_PRIORITY,exec,null ) ;
    }
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param exec 実施用オブジェクトを設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @param input 対象のスレッド引数を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance( ExecutionThread exec,Object input )
        throws InputException,ExecutionException
    {
        return OneThread.createInstance( false,Thread.NORM_PRIORITY,exec,input ) ;
    }
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param exec 実施用オブジェクトを設定します.
     * @param priority 対象のプライオリティを設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @param input 対象のスレッド引数を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance(
        ExecutionThread exec,int priority,Object input )
        throws InputException,ExecutionException
    {
        return OneThread.createInstance( false,priority,exec,input ) ;
    }
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec 実施用オブジェクトを設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance( boolean mode,ExecutionThread exec )
        throws InputException,ExecutionException
    {
        return OneThread.createInstance( mode,Thread.NORM_PRIORITY,exec,null ) ;
    }
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param exec 実施用オブジェクトを設定します.
     * @param input 対象のスレッド引数を設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance(
        boolean mode,ExecutionThread exec,Object input )
        throws InputException,ExecutionException
    {
        return OneThread.createInstance( mode,Thread.NORM_PRIORITY,exec,input ) ;
    }
    
    /**
     * オブジェクト情報の生成.
     * <BR><BR>
     * 対象のオブジェクト情報を生成します.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param priority 対象のプライオリティを設定します.
     * @param exec 実施用オブジェクトを設定します.
     * @param input 対象のスレッド引数を設定します.
     * @return OneThread 生成オブジェクト情報が返されます.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    public static final OneThread createInstance(
        boolean mode,int priority,ExecutionThread exec,Object input )
        throws InputException,ExecutionException
    {
        
        OneThread ret = null ;
        
        ret = new OneThread() ;
        
        try{
            ret.createThread( mode,priority,exec,input ) ;
        }catch( InputException in ){
            throw in ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
        
        return ret ;
        
    }
    
    /**
     * コンストラクタ.
     */
    private OneThread()
    {
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * スレッド削除.
     * <BR><BR>
     * 対象のスレッド情報を破棄します.
     */
    public final void clear()
    {
        
        m_sync.clear() ;
        
        try{
            
            m_thread.clear() ;
            
        }catch( Exception t ){
        }
            
        try{
            
            m_state.clear() ;
            
        }catch( Exception tt ){
        }
        
        m_exec = null ;
        m_state = null ;
        m_nowTime = 0L ;
        
    }
    
    /**
     * スレッド破棄を実施.
     * <BR><BR>
     * スレッド破棄を実施します.
     */
    public final void destroyThread()
    {
        try{
            m_thread.destroyThread() ;
        }catch( Exception t1 ){
        }
    }
    
    /**
     * スレッドステータス情報を取得.
     * <BR><BR>
     * スレッドステータス情報を取得します.
     * <BR>
     * @return int スレッドステータス情報が返されます.
     */
    public final int getState()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getThreadState() ;
            }
        }catch( Exception t ){
            ret = ExecutionThread.STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実施時間の取得.
     * <BR><BR>
     * スレッドが実施された時間を取得します.
     * <BR>
     * @return long スレッドが実施された時間が返されます.
     */
    public final long getThreadTime()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_nowTime ;
            }
        }catch( Exception t ){
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * スレッドプライオリティを取得.
     * <BR><BR>
     * 対象スレッドのスレッドプライオリティを取得します.
     * <BR>
     * @return int 設定されているスレッドプライオリティが返されます.<BR>
     *             スレッドが存在しない場合[-1]が返されます.
     */
    public final int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_thread.getPriority() ;
            }
        }catch( Exception t ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行用オブジェクトを取得.
     * <BR><BR>
     * スレッド実行用オブジェクトを取得します.
     * <BR>
     * @return ExecutionThread スレッド実行用オブジェクトが返されます.
     */
    public final ExecutionThread getExecutionThread()
    {
        ExecutionThread ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_exec ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド生成チェック.
     * <BR><BR>
     * スレッド生成チェックを行います.
     * <BR>
     * @return boolean スレッド生成チェックが返されます.<BR>
     *                 [true]が返された場合、スレッド情報は生成されています.<BR>
     *                 [false]が返された場合、スレッド情報は生成されていません.
     */
    public final boolean isCreate()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_thread.isThread() ;
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド開始チェック.
     * <BR><BR>
     * スレッド処理が開始されているかチェックします.
     * <BR>
     * @return boolean スレッド処理が開始されたか否かをチェックします.<BR>
     *                 [true]が返された場合、スレッド処理は開始しています.<BR>
     *                 [false]が返された場合、スレッド処理は開始されていません.
     */
    public final boolean isStart()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = this.getFlg( OneThread.STARTUP_NAME ) ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド終了チェック.
     * <BR><BR>
     * スレッド処理が終了されているかチェックします.
     * <BR>
     * @return boolean スレッド処理が終了されたか否かをチェックします.<BR>
     *                 [true]が返された場合、スレッド処理は終了しています.<BR>
     *                 [false]が返された場合、スレッド処理は終了していないか
     *                 スレッドが動作していません.
     */
    public final boolean isExit()
    {
        boolean ret ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = this.getFlg( OneThread.SHUTDOWN_NAME ) ;
                
            }
            
        }catch( Exception t ){
            ret = true ;
        }
        
        return ret ;
    }
    
    /**
     * スレッド実行.
     * <BR><BR>
     * スレッド実行を行う処理部を実装します.
     * @param in スレッド処理での引数情報を指定します.
     * @exception ExecutionException 実行例外.
     * @exception ThreadDeath スレッド破棄通知.
     */
    protected void processThread( Object in )
        throws ExecutionException,ThreadDeath
    {
        
        try{
            
            try{
                
                synchronized( m_sync.get() ){
                    
                    this.setFlg( OneThread.STARTUP_NAME,true ) ;
                    this.setFlg( OneThread.SHUTDOWN_NAME,false ) ;
                    
                    m_nowTime = System.currentTimeMillis() ;
                    
                }
                
                m_exec.init( null ) ;
                m_exec.execution( null ) ;
                
            }catch( ThreadDeath death ){
                throw death ;
            }catch( OutOfMemoryError mem ) {
            }catch( NullPointerException nul ){
            }catch( ExecutionException ee ){
                
                if( ee.getLevel() != ExecutionException.LEVEL_STOP ){
                    throw ee ;
                }
                
            }catch( Exception t ){
            }finally{
                
                try{
                    m_exec.exit( null ) ;
                }catch( Exception tt ){
                }
                
                try{
                    
                    synchronized( m_sync.get() ){
                        
                        this.setFlg( OneThread.STARTUP_NAME,false ) ;
                        this.setFlg( OneThread.SHUTDOWN_NAME,true ) ;
                        m_nowTime = 0L ;
                        
                    }
                    
                }catch( Exception tt ){
                    m_nowTime = 0L ;
                }
                
            }
            
        }catch( ExecutionException ex ){
            throw ex ;
        }
        
    }
    
    /**
     * スレッド破棄時の処理.
     * <BR><BR>
     * スレッド破棄時の処理を生成します.
     * <BR>
     * @param in スレッド処理で渡される引数情報です.
     * @exception Exception 処理エラーを設定します.
     */
    protected void destroyThread( Object in )
        throws Exception
    {
        m_exec.destroy( in ) ;
    }
    
    /**
     * スレッド例外発生処理実施.
     * <BR><BR>
     * スレッド処理での例外が発生した場合、実施されます.
     * <BR>
     * @param err 実行例外が返されます.
     */
    protected final void toException( ExecutionException err )
    {
        // 特になし..
    }
    
    
    /**
     * スレッド生成.
     * <BR><BR>
     * スレッドを利用可能な状態にします.
     * <BR>
     * @param mode 処理時にスレッドログに登録するか有無を指定します.<BR>
     *             [true]を設定した場合、スレッドログに登録します.<BR>
     *             [false]を設定した場合、スレッドログに登録しません.
     * @param priority 対象のプライオリティを設定します.
     * @param exec スレッド実行対象のオブジェクトを設定します.
     * @param in 対象のスレッド引数を設定します.
     * @exception InputException 入力例外.
     * @exception ExecutionException 実施例外.
     */
    private final void createThread( boolean mode,int priority,ExecutionThread exec,Object in )
        throws InputException,ExecutionException
    {
        int state ;
        
        if( exec == null ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            
            m_sync.create() ;
            
            synchronized( m_sync.get() ){
                m_exec = exec ;
            }
            
            m_thread.setPriority( priority ) ;
            m_thread.create( ( mode == true ) ? false : true,this,in,true ) ;
            
            if( mode == true ){
                m_thread.setObjectName( exec.getClass().getName() ) ;
            }
            else{
                m_thread.setObjectName( null ) ;
            }
            
            m_state = this.initState() ;
            m_thread.startThread() ;
            
            // スレッド実行待ち.
            /*for( ;; ){
                
                state = m_thread.getState() ;
                
                if(
                    state == ExecutionThread.STATE_NOT ||
                    state == ExecutionThread.STATE_EXECUTION
                )
                {
                    if( state == ExecutionThread.STATE_NOT ){
                        this.clear() ;
                        throw new ExecutionException(
                            "スレッド実行に失敗しました"
                        ) ;
                    }
                    break ;
                }
                
                UtilCom.cpuCreate() ;
                
            }*/
            
        }catch( ExecutionException ex ){
            this.clear() ;
            throw ex ;
        }catch( BaseException be ){
            this.clear() ;
            throw new ExecutionException( be ) ;
        }catch( Exception t ){
            this.clear() ;
            throw new ExecutionException( t ) ;
        }
        
    }
    
    /**
     * ステータス初期化処理.
     */
    private final ThreadState initState()
        throws AccessException
    {
        boolean[] value = null ;
        ThreadState ret = null ;
        
        try{
            
            // スレッド対象のパラメータ値を取得.
            ret = ThreadManager.
                getInstance().
                    getState( m_thread.getName() ) ;
            
            // 各パラメータを初期処理.
            value = new boolean[ 1 ] ;
            value[ 0 ] = false ;
            ret.setParameter( OneThread.STARTUP_NAME,value ) ;
            
            value = new boolean[ 1 ] ;
            value[ 0 ] = false ;
            ret.setParameter( OneThread.SHUTDOWN_NAME,value ) ;
            
        }catch( Exception t ){
            
            ret = null ;
            throw new AccessException( t ) ;
            
        }finally{
            
            value = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * フラグの設定.
     */
    private final void setFlg( String name,boolean flg )
        throws NullPointerException
    {
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            obj[ 0 ] = flg ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
    }
    
    /**
     * フラグの取得.
     */
    private final boolean getFlg( String name )
        throws NullPointerException
    {
        boolean ret ;
        boolean[] obj ;
        ThreadState state = null ;
        
        try{
            
            state = m_state ;
            obj = ( boolean[] )state.getParameter( name ) ;
            ret = obj[ 0 ] ;
            
        }catch( Exception t ){
            throw new NullPointerException( "ステータス情報は既にクリアされています" ) ;
        }finally{
            
            state = null ;
            obj = null ;
            
        }
        
        return ret ;
    }
    
}
