/*
 * @(#)ArrayCharBox.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.box;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;


/**
 * SingleChar固定配列オブジェクト.
 * <BR><BR>
 * 1つの数値[char(32bit)]の固定配列を管理するオブジェクトです.<BR>
 * また、このオブジェクトは同期されていません.
 *
 * @version     1.00, 2004/11/02
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ArrayCharBox implements BaseBox
{
	
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ArrayCharBox.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * データ管理.
     */
    private Object[] m_array = null ;
    
    /**
     * フラグ配列.
     */
    private FlagBox m_flags = null ;
    
    /**
     * 格納データ長.
     */
    private int m_length = 0 ;
    
    /**
     * コンストラクタ.
     */
    public ArrayCharBox()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 格納対象情報を設定します.
     * <BR>
     * @param size 生成対象サイズを設定します.
     * @exception InputException 入力例外.
     */
    public ArrayCharBox( int size )
        throws InputException
    {
        this.create( size ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 格納対象情報を設定します.
     * <BR>
     * @param size 生成対象サイズを設定します.
     * @exception InputException 入力例外.
     */
    public final void create( int size )
        throws InputException
    {
        if( size < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        this.clear() ;
        
        size = (
            ( ( ( size & BoxDef.ELEMENT_MASK ) != 0 ) ? 1 : 0 ) +
            ( ( size & (~BoxDef.ELEMENT_MASK) ) >> BoxDef.ELEMENT_PAUSE )
        ) * BoxDef.ELEMENT_LENGTH ;
        
        m_length = size ;
        m_array = new Object[ size ] ;
        m_flags = new FlagBox( size ) ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        if( m_flags != null ){
            m_flags.clear() ;
        }
        m_flags = null ;
        m_array = null ;
        m_length = 0 ;
    }
    
    /**
     * 領域を追加.
     * <BR><BR>
     * 領域を追加します.
     * <BR>
     * @param size 追加対象のサイズを設定します.
     * @return int 追加された後の全体サイズが返されます.
     */
    public final int addSpace( int size )
    {
        int len ;
        Object[] tmp = null ;
        
        if( size > 0 ){
            
            len = m_flags.addSpace( size ) ;
            
            tmp = new Object[ len ] ;
            System.arraycopy( m_array,0,tmp,0,m_array.length ) ;
            m_array = tmp ;
            tmp = null ;
            
            m_length = len ;
            
        }
        else{
            len = m_length ;
        }
        
        return len ;
    }
    
    /**
     * 領域を削除.
     * <BR><BR>
     * 領域を削除します.
     * <BR>
     * @param size 削除対象のサイズを設定します.
     * @return int 削除された後の全体サイズが返されます.
     */
    public final int removeSpace( int size )
    {
        int len ;
        Object[] tmp = null ;
        
        if( size > 0 ){
            
            if( ( len = m_flags.removeSpace( size ) ) != m_length ){
                
                tmp = new Object[ len ] ;
                System.arraycopy( m_array,0,tmp,0,tmp.length ) ;
                m_array = tmp ;
                tmp = null ;
                
                m_length = len ;
                
            }
            
        }
        else{
            len = m_length ;
        }
        
        return len ;
    }
    
    /**
     * 情報を追加.
     * <BR><BR>
     * 情報を追加します.
     * <BR>
     * @param val 追加対象の要素を設定します.
     * @return int 格納された項番が返されます.<BR>
     *             格納対象の空きが存在しない場合や、
     *             格納要素が無効な場合[-1]が返されます.
     */
    public final int add( char[] val )
    {
        int ret ;
        
        Object[] arrays = null ;
        FlagBox flgs = null ;
        
        arrays = m_array ;
        flgs = m_flags ;
        ret = -1 ;
        
        if(
            arrays != null &&
            ( ret = flgs.getSearch() ) != -1
        )
        {
            m_flags.add( true,ret ) ;
            m_array[ ret ] = val ;
        }
        
        return ret ;
    }
    
    /**
     * 情報を設定.
     * <BR><BR>
     * 情報を設定します.
     * <BR>
     * @param no 設定対象の項番を設定します.
     * @param val 追加対象の要素を設定します.
     */
    public final void set( int no,char[] val )
    {
        
        Object[] arrays = null ;
        FlagBox flgs = null ;
        
        arrays = m_array ;
        flgs = m_flags ;
        
        if(
            arrays == null ||
            no < 0 || no >= m_length
        )
        {
            return ;
        }
        
        arrays[ no ] = val ;
        flgs.add( true,no ) ;
        
    }
    
    /**
     * 対象要素位置を削除.
     * <BR><BR>
     * 対象の要素位置を削除します.
     * <BR>
     * @param no 削除対象の要素項番を設定します.
     * @return char[] 削除された情報が返されます.
     */
    public final char[] remove( int no )
    {
        char[] ret = null ;
        Object[] arrays = null ;
        FlagBox flgs = null ;
        
        arrays = m_array ;
        flgs = m_flags ;
        
        if(
            arrays == null ||
            no < 0 || no >= m_length ||
            flgs.get( no ) == false
        )
        {
            return null ;
        }
        
        flgs.add( false,no ) ;
        ret = ( char[] )m_array[ no ] ;
        m_array[ no ] = null ;
        
        return ret ;
    }
    
    /**
     * 要素情報を取得.
     * <BR><BR>
     * 格納されている要素情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return char[] 取得された情報が返されます.<BR>
     *                情報が不正な場合[null]が返されます.
     */
    public final char[] get( int no )
    {
        return (
            m_array == null ||
            no < 0 || no >= m_length ||
            m_flags.get( no ) == false
        ) ? null : ( char[] )m_array[ no ] ;
    }
    
    /**
     * 管理要素長を取得.
     * <BR><BR>
     * 管理要素長を取得します.
     * <BR>
     * @return int 管理要素長が返されます.
     */
    public final int getMaxSize()
    {
        return ( m_array == null ) ? 0 : m_length ;
    }
    
    /**
     * 利用中の要素長を取得.
     * <BR><BR>
     * 現在利用されている要素長を取得します.
     * <BR>
     * @return int 利用要素長が返されます.
     */
    public final int getUseSize()
    {
        return ( m_array == null ) ? -1 : m_flags.getUseSize() ;
    }
    
    /**
     * 情報存在チェック.
     * <BR><BR>
     * 対象要素位置に情報が存在しているかチェックします.
     * <BR>
     * @param no チェック対象の要素項番を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、要素情報は存在します.
     *                 [false]が返された場合、要素情報は存在しません.
     */
    public final boolean isData( int no )
    {
        return ( m_array == null || m_flags.get( no ) == false ) ? false : true ;
    }
    
    /**
     * 生成チェック.
     * <BR><BR>
     * オブジェクトが生成されているかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、生成されています.<BR>
     *                 [false]が返された場合、生成されていません.
     */
    public final boolean isCreate()
    {
        return ( m_array == null ) ? false : true ;
    }
    
}

