/*
 * @(#)ReadCsv.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.read;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.util.ReadIndex;
import org.maachang.commons.util.convert.ConvertToIndexNo;


/**
 * CSV情報を読み込み.
 * <BR><BR>
 * CSV情報読み込み処理をサポートします.
 *
 * @version 1.00, 2003/12/10
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ReadCsv
{
    
    /**
     * コーテーション : シングル.
     */
    private static final char SINGLE_COTE = '\'' ;
    
    /**
     * コーテーション : ダブル.
     */
    private static final char DOUBLE_COTE = '\"' ;
    
    /**
     * 標準区切りコード.
     */
    private static final String DEF_CAT = "," ;
    
    /**
     * 情報なし.
     */
    private static final String NO_DATA = "" ;
    
    /**
     * 区切りコード.
     */
    private String m_cat = ReadCsv.DEF_CAT ;
    
    /**
     * 読み込み用Index情報.
     */
    private ReadIndex m_index = null ;
    
    /**
     * コントラクタ.
     */
    public ReadCsv()
    {
        this( null,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * CSV読み込みのための情報生成を行います.
     * <BR>
     * @param index 読み込み用Index情報を設定します.
     */
    public ReadCsv( ReadIndex index )
    {
        this( index,null ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * CSV読み込みのための情報生成を行います.
     * <BR>
     * @param cat 区切りコードを設定します.
     * @exception InputException 入力例外.
     */
    public ReadCsv( String cat )
        throws InputException
    {
        this( null,cat ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * CSV読み込みのための情報生成を行います.
     * <BR>
     * @param index 読み込み用Index情報を設定します.
     * @param cat 区切りコードを設定します.
     */
    public ReadCsv( ReadIndex index,String cat )
    {
        
        m_cat = ( cat == null ) ? ReadCsv.DEF_CAT : cat ;
        
        try{
            
            if( index != null ){
                this.create( index ) ;
            }
            
        }catch( InputException in ){
            this.clear() ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * CSV読み込みのための情報生成を行います.
     * <BR>
     * @param index 読み込み用Index情報を設定します.
     * @exception InputException 入力例外.
     */
    public final void create( ReadIndex index )
        throws InputException
    {
        if( index == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        m_index = index ;
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_index = null ;
    }
    
    /**
     * 指定行の情報を取得.
     * <BR><BR>
     * 指定された行内容の情報を取得します.
     * <BR>
     * @param no 取得指定行位置を設定します.
     * @param out 結果内容が返されます.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final void getIndex( int no,ConvertToIndexNo out )
        throws InputException,ReadSyntaxException
    {
        String line = null ;
        
        try{
            
            line = ( String )m_index.getIndex( no ) ;
            
            ( ( ReadElement )out ).clear() ;
            if( line == null || line.length() <= 0 ){
                return ;
            }
            
            ReadCsv.analisys_CSV( line,m_cat,( ReadElement )out ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }catch( AccessException ac ){
            throw new ReadSyntaxException( ac ) ;
        }finally{
            line = null ;
        }
        
    }
    
    
    /**
     * 指定行の情報を取得.
     * <BR><BR>
     * 指定された行内容の情報を取得します.
     * <BR>
     * @param no 取得指定行位置を設定します.
     * @return ConvertToIndexNo 対象行内容の情報が返されます.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public final ConvertToIndexNo getIndex( int no )
        throws InputException,ReadSyntaxException
    {
        String line = null ;
        ConvertToIndexNo ret = null ;
        
        try{
            
            line = ( String )m_index.getIndex( no ) ;
            
            if( line == null || line.length() <= 0 ){
                return null ;
            }
            
            ret = new ReadElement() ;
            ReadCsv.analisys_CSV( line,m_cat,( ReadElement )ret ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }catch( AccessException ac ){
            throw new ReadSyntaxException( ac ) ;
        }finally{
            line = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 指定文字列をCSV形式として解析.
     * <BR><BR>
     * 指定文字列をCSV形式として解析します.
     * <BR>
     * @param code 解析対象の文字列を指定します.
     * @param cat CSV形式解析のための区切り文字を設定します.
     * @param out 結果内容が返されます.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public static final void getStringByCsv( String code,String cat,ConvertToIndexNo out )
        throws InputException,ReadSyntaxException
    {
        
        if( code == null || cat == null || out == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            ( ( ReadElement )out).clear() ;
            ReadCsv.analisys_CSV( code,cat,( ReadElement )out ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
    }
    
    /**
     * 指定文字列をCSV形式として解析.
     * <BR><BR>
     * 指定文字列をCSV形式として解析します.
     * <BR>
     * @param code 解析対象の文字列を指定します.
     * @param cat CSV形式解析のための区切り文字を設定します.
     * @return ConvertToIndexNo 結果内容が返されます.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 読み込み構文例外.
     */
    public static final ConvertToIndexNo getStringByCsv( String code,String cat )
        throws InputException,ReadSyntaxException
    {
        ConvertToIndexNo ret = null ;
        
        if( code == null || cat == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            ret = new ReadElement() ;
            ReadCsv.analisys_CSV( code,cat,( ReadElement )ret ) ;
            
        }catch( InputException in ){
            throw in ;
        }catch( ReadSyntaxException rs ){
            throw rs ;
        }
        
        return ret ;
        
    }
    
    /**
     * 全体行数の取得.
     * <BR><BR>
     * 全体の行数を取得します.
     * <BR>
     * @return int 全体の行数が返されます.
     */
    public final int getLineSize()
    {
        return m_index.size() ;
    }
    
    
    
    /**
     * 対象行数の解析処理.
     */
    private static final void analisys_CSV( String oneLine,String catCode,ReadElement out )
        throws InputException,ReadSyntaxException
    {
        
        int i ;
        int len ;
        int pointLen ;
        int catLen ;
        int catPoint ;
        int mode ;
        int snpPnt ;
        
        int sx,ex ;
        int datLen ;
        
        char cote ;
        
        int[] points = null ;
        char[] string = null ;
        char[] cat = null ;
        
        if( oneLine == null || catCode == null || out == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        len = oneLine.length() ;
        catLen = catCode.length() ;
        
        if( len <= 0 || catLen <= 0 || len <= catLen ){
            throw new InputException( "引数の情報長は不正です" ) ;
        }
        
        points = new int[ len ] ;
        string = new char[ len ] ;
        cat = new char[ catLen ] ;
        
        oneLine.getChars( 0,len,string,0 ) ;
        catCode.getChars( 0,catLen,cat,0 ) ;
        
        try{
            
            for(
                i = 0,pointLen = 0,catPoint = 0,mode = 0,snpPnt = 0,cote = 0 ;
                i < len ; i ++
            )
            {
                
                if( mode == 0 ){
                    
                    if( string[ i ] == cat[ 0 ] ){
                        
                        mode = 1 ;
                        catPoint = 1 ;
                        snpPnt = i + 1 ;
                        
                        continue ;
                        
                    }
                    else if(
                        string[ i ] == ReadCsv.DOUBLE_COTE ||
                        string[ i ] == ReadCsv.SINGLE_COTE
                    )
                    {
                        cote = string[ i ] ;
                        mode = 2 ;
                    }
                    
                }
                else if( mode == 1 ){
                    
                    if( catPoint == catLen ){
                        
                        mode = 0 ;
                        catPoint = 0 ;
                        
                        points[ pointLen ] = i - catLen ;
                        pointLen ++ ;
                        i -- ;
                        
                        continue ;
                        
                    }
                    else if( string[ i ] == cat[ catPoint ] ){
                        
                        catPoint ++ ;
                        continue ;
                        
                    }
                    else{
                        
                        mode = 0 ;
                        catPoint = 0 ;
                        i = snpPnt ;
                        
                        continue ;
                        
                    }
                    
                }
                else if( mode == 2 ){
                    if( string[ i ] == cote ){
                        mode = 0 ;
                    }
                }
                
            }
            
            if( mode == 1 ){
                if( catPoint == catLen ){
                    
                    mode = 0 ;
                    catPoint = 0 ;
                    
                    points[ pointLen ] = i - catLen ;
                    pointLen ++ ;
                }
            }
            
            points[ pointLen ] = len ;
            pointLen ++ ;
            
            if( pointLen == 1 ){
                out.add(
                    ReadCsv.catCote(
                        oneLine
                    )
                ) ;
            }
            else{
                
                int st,ed ;
                
                len = pointLen ;
                ex = points[ 0 ] - catLen ;
                
                for( i = 1,sx = 0 - ( catLen - 1 ) ; i <= len ; i ++ ){
                    
                    datLen = ( (ex+1) - sx ) + 1 ;
                    
                    if( datLen == 1 ){
                        
                        out.add( ReadCsv.NO_DATA ) ;
                        
                    }
                    else{
                        
                        st = sx + catLen - 1 ;
                        ed = st + datLen - 1 ;
                        out.add(
                            ReadCsv.catCote(
                                oneLine.substring( st,ed )
                            )
                        ) ;
                        
                    }
                    
                    if( i != len ){
                        sx = points[ i-1 ] + 1 ;
                        ex = points[ i ] - catLen ;
                     }
                    
                }
                
                
            }
            
        }catch( Exception t ){
            out.clear() ;
            throw new ReadSyntaxException( t ) ;
        }finally{
            points = null ;
            string = null ;
            cat = null ;
        }
        
    }
    
    /**
     * 指定コーテーション区切りをはずす.
     */
    private static final String catCote( String str )
    {
        int i ;
        int len ;
        
        String ret = null ;
        char[] chrs = null ;
        StringBuffer buf = null ;
        
        if( str == null ){
            ret = ReadCsv.NO_DATA ;
        }
        else if( ( len = str.length() ) <= 0 ){
            ret = ReadCsv.NO_DATA ;
        }
        else{
            
            chrs = new char[ len ] ;
            str.getChars( 0,len,chrs,0 ) ;
            buf = new StringBuffer( len ) ;
            
            for( i = 0 ; i < len ; i ++ ){
                if(
                    chrs[ i ] != ReadCsv.DOUBLE_COTE &&
                    chrs[ i ] != ReadCsv.SINGLE_COTE
                )
                {
                    buf.append( chrs[ i ] ) ;
                }
            }
            
            ret = buf.toString() ;
            buf = null ;
            
        }
        
        return ret ;
    }
    
}
