/*
 * @(#)GZIPBinary.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.zip;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipException;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.BinResourceInputStream;
import org.maachang.commons.resource.BinResourceOutputStream;


/**
 * GZIPBinaryオブジェクト.
 * <BR><BR>
 * 指定したバイナリ情報をGZIPによる圧縮／解凍します.
 *
 * @version 1.00, 2004/09/20
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class GZIPBinary
{
    
    /**
     * バッファ長.
     */
    private static final int BUF_LENGTH = 1024 ;
    
    /**
     * シングルトン.
     */
    private static final GZIPBinary SNGL = new GZIPBinary() ;
    
    /**
     * コンストラクタ.
     */
    private GZIPBinary()
    {
        
    }
    
    /**
     * オブジェクトを取得.
     * <BR><BR>
     * オブジェクトを取得します.
     * <BR>
     * @return GZIPBinary オブジェクト情報が返されます.
     */
    public static final GZIPBinary getInstance() {
        return SNGL ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByGZIP( byte[] binary )
        throws InputException,AccessException
    {
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByGZIP( binary,0,binary.length ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param length 圧縮情報長を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByGZIP( byte[] binary,int length )
        throws InputException,AccessException
    {
        if( binary == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByGZIP( binary,0,length ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param offset 圧縮情報開始位置を設定します.
     * @param length 圧縮情報長を設定します.
     * @return byte[] 圧縮されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertBinaryByGZIP( byte[] binary,int offset,int length )
        throws InputException,AccessException
    {
        int len ;
        
        ByteArrayOutputStream bin = null ;
        GZIPOutputStream zip = null ;
        BufferedOutputStream buf = null ;
        
        byte[] ret = null ;
        
        if(
            binary == null || ( len = binary.length ) <= 0 ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            zip = new GZIPOutputStream( bin ) ;
            buf = new BufferedOutputStream( zip ) ;
            
            buf.write( binary,offset,length ) ;
            buf.flush() ;
            zip.finish() ;
            
            ret = bin.toByteArray() ;
            
        }catch( IllegalArgumentException ia ){
            throw new InputException( ia ) ;
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                bin.close() ;
            }catch( Exception t ){
            }
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            bin = null ;
            buf = null ;
            zip = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param out 圧縮された情報を格納するバイナリリソースを設定します.
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @return int 圧縮されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertBinaryByGZIP( BinResource out,BinResource binary )
        throws InputException,AccessException
    {
        if( binary == null || binary.isUse() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByGZIP( out,binary,0,binary.size() ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param out 圧縮された情報を格納するバイナリリソースを設定します.
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param length 圧縮情報長を設定します.
     * @return int 圧縮されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertBinaryByGZIP( BinResource out,BinResource binary,int length )
        throws InputException,AccessException
    {
        if( binary == null || binary.isUse() == false ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertBinaryByGZIP( out,binary,0,length ) ;
    }
    
    /**
     * 指定されたバイナリ情報を圧縮.
     * <BR><BR>
     * 指定されたバイナリ情報を圧縮します.
     * <BR>
     * @param out 圧縮された情報を格納するバイナリリソースを設定します.
     * @param binary 圧縮対象のバイナリ情報を設定します.
     * @param offset 圧縮情報開始位置を設定します.
     * @param length 圧縮情報長を設定します.
     * @return int 圧縮されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertBinaryByGZIP( BinResource out,BinResource binary,int offset,int length )
        throws InputException,AccessException
    {
        int all ;
        int len ;
        int pnt ;
        int ret ;
        
        byte[] tmp = null ;
        BinResourceOutputStream bin = null ;
        GZIPOutputStream zip = null ;
        BufferedOutputStream buf = null ;
        
        if(
            out == null || out.isUse() == false ||
            binary == null || binary.isUse() == false || ( all = binary.size() ) <= 0 ||
            offset < 0 || offset > all ||
            length <= 0 || length > all ||
            ( offset + length ) > all
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            out.reset() ;
            bin = new BinResourceOutputStream( false,out ) ;
            zip = new GZIPOutputStream( bin ) ;
            buf = new BufferedOutputStream( zip ) ;
            
            tmp = new byte[ GZIPBinary.BUF_LENGTH ] ;
            
            for( pnt = offset ;; ){
                
                len = ( length <= GZIPBinary.BUF_LENGTH ) ? length : GZIPBinary.BUF_LENGTH ;
                if( len <= 0 ){
                    break ;
                }
                len = binary.getBinary( tmp,pnt,0,len ) ;
                buf.write( tmp,0,len ) ;
                
                pnt += len ;
                length -= len ;
            }
            
            buf.flush() ;
            zip.finish() ;
            
            ret = out.size() ;
            
        }catch( IllegalArgumentException ia ){
            throw new InputException( ia ) ;
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                bin.close() ;
            }catch( Exception t ){
            }
            
            buf = null ;
            zip = null ;
            bin = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertGZIPByBinary( byte[] zbin )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertGZIPByBinary( zbin,0,zbin.length ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param length 解凍情報長を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertGZIPByBinary( byte[] zbin,int length )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertGZIPByBinary( zbin,0,length ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param offset 解凍情報開始位置を設定します.
     * @param length 解凍情報長を設定します.
     * @return byte[] 解凍されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final byte[] convertGZIPByBinary( byte[] zbin,int offset,int length )
        throws InputException,AccessException
    {
        int len ;
        
        ByteArrayInputStream bIn = null ;
        ByteArrayOutputStream bOut = null ;
        GZIPInputStream zip = null ;
        BufferedInputStream buf = null ;
        byte[] tmp = null ;
        byte[] ret = null ;
        
        if(
            zbin == null || ( len = zbin.length ) <= 0 ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            bIn = new ByteArrayInputStream( zbin,offset,length ) ;
            bOut = new ByteArrayOutputStream() ;
            zip = new GZIPInputStream( bIn ) ;
            buf = new BufferedInputStream( zip ) ;
            
            tmp = new byte[ GZIPBinary.BUF_LENGTH ] ;
            while( ( len = buf.read( tmp ) ) >= 0 ){
                bOut.write( tmp,0,len ) ;
            }
            
            bOut.flush() ;
            ret = bOut.toByteArray() ;
            
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            try{
                bIn.close() ;
            }catch( Exception t ){
            }
            
            try{
                bOut.close() ;
            }catch( Exception t ){
            }
            
            bIn = null ;
            bOut = null ;
            zip = null ;
            buf = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param out 解凍された情報を格納するバイナリリソースを設定します.
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @return int 解凍されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertGZIPByBinary( BinResource out,BinResource zbin )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertGZIPByBinary( out,zbin,0,zbin.size() ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param out 解凍された情報を格納するバイナリリソースを設定します.
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param length 解凍情報長を設定します.
     * @return int 解凍されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertGZIPByBinary( BinResource out,BinResource zbin,int length )
        throws InputException,AccessException
    {
        if( zbin == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        return this.convertGZIPByBinary( out,zbin,0,length ) ;
    }
    
    /**
     * 指定された圧縮バイナリ情報を解凍.
     * <BR><BR>
     * 指定された圧縮バイナリ情報を解凍します.
     * <BR>
     * @param out 解凍された情報を格納するバイナリリソースを設定します.
     * @param zbin 解凍対象の圧縮バイナリ情報を設定します.
     * @param offset 解凍情報開始位置を設定します.
     * @param length 解凍情報長を設定します.
     * @return int 解凍されたバイナリ長が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public final int convertGZIPByBinary( BinResource out,BinResource zbin,int offset,int length )
        throws InputException,AccessException
    {
        int len ;
        int ret ;
        
        BinResourceInputStream bIn = null ;
        BinResourceOutputStream bOut = null ;
        GZIPInputStream zip = null ;
        BufferedInputStream buf = null ;
        BufferedOutputStream bufOut = null ;
        byte[] tmp = null ;
        
        if(
            zbin == null || ( len = zbin.size() ) <= 0 ||
            offset < 0 || offset > len ||
            length <= 0 || length > len ||
            ( offset + length ) > len
            
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            out.reset() ;
            bIn = new BinResourceInputStream( false,zbin,offset,length ) ;
            bOut = new BinResourceOutputStream( false,out ) ;
            zip = new GZIPInputStream( bIn ) ;
            buf = new BufferedInputStream( zip ) ;
            bufOut = new BufferedOutputStream( bOut ) ;
            
            tmp = new byte[ GZIPBinary.BUF_LENGTH ] ;
            while( ( len = buf.read( tmp ) ) >= 0 ){
                bufOut.write( tmp,0,len ) ;
            }
            
            bufOut.flush() ;
            ret = out.size() ;
            
        }catch( ZipException ze ){
            throw new AccessException( ze ) ;
        }catch( IOException io ){
            throw new AccessException( io ) ;
        }finally{
            
            try{
                zip.close() ;
            }catch( Exception t ){
            }
            
            try{
                buf.close() ;
            }catch( Exception t ){
            }
            
            try{
                bIn.close() ;
            }catch( Exception t ){
            }
            
            try{
                bOut.close() ;
            }catch( Exception t ){
            }
            
            try{
                bufOut.close() ;
            }catch( Exception t ){
            }
            
            bIn = null ;
            bOut = null ;
            zip = null ;
            buf = null ;
            bufOut = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
}
