/*
 * @(#)MaachangConnectInterface.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect ;

import java.net.InetAddress;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.resource.BinResource;

/**
 * MaachangConnectプロトコルインターフェイス.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public interface MaachangConnectInterface {
    
    /**
     * クローズ処理.
     * <BR><BR>
     * 接続状態をクローズします.
     */
    public void close() ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,byte[] binary )
        throws InputException,AccessException ;
    
    /**
     * 送信処理.
     * <BR><BR>
     * 対象電文を送信します.
     * <BR>
     * @param address 送信対象アドレスを設定します.
     * @param port 送信対象ポートを設定します.
     * @param binary 送信対象のバイナリを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public void send( InetAddress address,int port,BinResource binary )
        throws InputException,AccessException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out )
        throws AccessException ;
    
    /**
     * 受信処理.
     * <BR><BR>
     * 受信処理を行います.
     * <BR>
     * @param out 受信先のアドレスが設定されます.
     * @param timeout 受信タイムアウト値を設定します.<BR>
     *                [0]以下を設定した場合、無限タイムアウト待ちです.
     * @return BinResource 受信された情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public BinResource receive( ConnectAddress out,int timeout )
        throws AccessException ;
    
    /**
     * バインドアドレスを取得.
     * <BR><BR>
     * バインドアドレスを取得します.
     * <BR>
     * @return InetAddress バインドアドレスが返されます.
     * @exception AccessException アクセス例外.
     */
    public InetAddress getBindAddress()
        throws AccessException ;
    
    /**
     * バインドポート番号を取得.
     * <BR><BR>
     * バインドポート番号を取得します.
     * <BR>
     * @return int バインドポート番号が返されます.
     * @exception AccessException アクセス例外.
     */
    public int getBindPort() throws AccessException ;
    
    /**
     * バッファサイズを取得.
     * <BR><BR>
     * バッファサイズを取得します.
     * <BR>
     * @return int バッファサイズが返されます.
     * @exception AccessException アクセス例外.
     */
    public int getBuffer() throws AccessException ;
    
    /**
     * 暗号ワード条件を設定.
     * <BR><BR>
     * 暗号ワード条件を設定します.
     * <BR>
     * @param cb32 暗号ワード条件を追加する条件を設定します.
     */
    public void setCb32( boolean cb32 ) ;
    
    /**
     * 暗号ワードを追加.
     * <BR><BR>
     * 暗号ワードを追加します.
     * <BR>
     * @param word 追加対象の暗号ワードを設定します.
     */
    public void addWord( String word ) ;
    
    /**
     * 暗号ワードを削除.
     * <BR><BR>
     * 対象の暗号ワードを削除します.
     * <BR>
     * @param word 削除対象の暗号ワードを設定します.
     */
    public void removeWord( String word ) ;
    
    /**
     * 暗号ワード一覧を取得.
     * <BR><BR>
     * 暗号ワード一覧が返されます.
     * <BR>
     * @return String[] 暗号ワード一覧が返されます.
     */
    public String[] getWords() ;
    
    /**
     * 暗号ワード数を取得.
     * <BR><BR>
     * 設定されている暗号ワード数が返されます.
     * <BR>
     * @return int 暗号ワード数が返されます.
     */
    public int getWordSize() ;
    
    /**
     * 暗号モードを取得.
     * <BR><BR>
     * 暗号モードを取得します.
     * <BR>
     * @return boolean 暗号モードが返されます.<BR>
     *                 [true]が返された場合、暗号モードはONです.<BR>
     *                 [false]が返された場合、暗号モードはOFFです.
     */
    public boolean isCb32() ;
    
    /**
     * オープンチェック.
     * <BR><BR>
     * オープンされているかチェックします.
     * <BR>
     * @return boolean オープンされているか返されます.<BR>
     *                 [true]が返された場合、オープン中です.<BR>
     *                 [false]が返された場合、クローズされています.
     */
    public boolean isOpen() ;
}

