/*
 * @(#)UnderReceiveTable.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.table ;

import java.net.InetAddress;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.resource.ResourceType;
import org.maachang.commons.util.BigTable;
import org.maachang.commons.util.array.ObjectArray;

/**
 * 受信中のデータを管理するテーブル.
 *
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class UnderReceiveTable extends BaseTable {
    
    /**
     * デフォルトコネクションタイムアウト.
     * 25sec.
     */
    private static final long DEF_TIMEOUT = 25000L ;
    
    /**
     * 最小コネクションタイムアウト.
     * 10sec.
     */
    private static final long MIN_TIMEOUT = 10000L ;
    
    /**
     * 最大コネクションタイムアウト.
     * 90sec.
     */
    private static final long MAX_TIMEOUT = 90000L ;
    
    /**
     * デフォルト再送依頼タイムアウト値.
     * 5sec.
     */
    private static final long DEF_RESEND_TIMEOUT = 3000L ;
    
    /**
     * 最小再送依頼タイムアウト値.
     * 1.5sec.
     */
    private static final long MIN_RESEND_TIMEOUT = 1500L ;
    
    /**
     * 最大再送依頼タイムアウト値.
     * 10sec.
     */
    private static final long MAX_RESEND_TIMEOUT = 10000L ;
    
    /**
     * デフォルト再送依頼カウント.
     */
    private static final int DEF_RETRY_COUNT = 6 ;
    
    /**
     * 最小再送依頼カウント.
     */
    private static final int MIN_RETRY_COUNT = 2 ;
    
    /**
     * 最大再送依頼カウント.
     */
    private static final int MAX_RETRY_COUNT = 18 ;

    
    /**
     * インデックステーブル.
     */
    private BigTable indexTable = null ;
    
    /**
     * データタイムアウト.
     */
    private long timeout = -1L ;
    
    /**
     * 再送依頼タイムアウト.
     */
    private long resendTimeout = -1L ;
    
    /**
     * 再送依頼待ちカウント.
     */
    private int retryCount = 0 ;
    
    /**
     * リソースタイプ.
     */
    private ResourceType resType = null ;
    
    
    /**
     * コンストラクタ.
     */
    public UnderReceiveTable() {
        this( DEF_TIMEOUT,null,DEF_RESEND_TIMEOUT,DEF_RETRY_COUNT ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して、受信中テーブルを生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     */
    public UnderReceiveTable( ResourceType resType ) {
        this( DEF_TIMEOUT,resType,DEF_RESEND_TIMEOUT,DEF_RETRY_COUNT ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して、受信中テーブルを生成します.
     * <BR>
     * @param timeout コネクションタイムアウト値を設定します.<BR>
     *                 設定可能な最小値は[10000]ミリ秒です.<BR>
     *                 設定可能な最大値は[90000]ミリ秒です.
     * @param resType 対象のリソースタイプを設定します.
     */
    public UnderReceiveTable( long timeout,ResourceType resType ) {
        this( timeout,resType,DEF_RESEND_TIMEOUT,DEF_RETRY_COUNT ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプを設定して、受信中テーブルを生成します.
     * <BR>
     * @param resType 対象のリソースタイプを設定します.
     * @param resendTimeout 再送依頼タイムアウト値を設定します.<BR>
     *                      設定可能な最小値は[1500]ミリ秒です.<BR>
     *                      設定可能な最大値は[10000]ミリ秒です.
     * @param retryCount 再送リトライカウント値を設定します.<BR>
     *                    設定可能な最小値は[2]回です.<BR>
     *                    設定可能な最大値は[18]回です.
     */
    public UnderReceiveTable( ResourceType resType,long resendTimeout,int retryCount ) {
        this( DEF_TIMEOUT,resType,resendTimeout,retryCount ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * リソースタイプと、タイムアウト値を設定して、受信中テーブルを生成します.
     * <BR>
     * @param timeout コネクションタイムアウト値を設定します.<BR>
     *                 設定可能な最小値は[10000]ミリ秒です.<BR>
     *                 設定可能な最大値は[90000]ミリ秒です.
     * @param resType 対象のリソースタイプを設定します.
     * @param resendTimeout 再送依頼タイムアウト値を設定します.<BR>
     *                      設定可能な最小値は[1500]ミリ秒です.<BR>
     *                      設定可能な最大値は[10000]ミリ秒です.
     * @param retryCount 再送リトライカウント値を設定します.<BR>
     *                    設定可能な最小値は[2]回です.<BR>
     *                    設定可能な最大値は[18]回です.
     */
    public UnderReceiveTable( long timeout,ResourceType resType,
        long resendTimeout,int retryCount ) {
        
        if( timeout == -1L ) {
            timeout = DEF_TIMEOUT ;
        }
        if( resendTimeout == -1L ) {
            resendTimeout = DEF_RESEND_TIMEOUT ;
        }
        if( retryCount == -1 ) {
            retryCount = DEF_RETRY_COUNT ;
        }
        
        if( timeout <= MIN_TIMEOUT ) {
            timeout = MIN_TIMEOUT ;
        }
        else if( timeout >= MAX_TIMEOUT ) {
            timeout = MAX_TIMEOUT ;
        }
        
        if( resendTimeout <= MIN_RESEND_TIMEOUT ) {
            resendTimeout = MIN_RESEND_TIMEOUT ;
        }
        else if( resendTimeout >= MAX_RESEND_TIMEOUT ) {
            resendTimeout = MAX_RESEND_TIMEOUT ;
        }
        
        if( retryCount <= MIN_RETRY_COUNT ) {
            retryCount = MIN_RETRY_COUNT ;
        }
        else if( retryCount >= MAX_RETRY_COUNT ) {
            retryCount = MAX_RETRY_COUNT ;
        }
        
        this.table = new ObjectArray() ;
        this.indexTable = new BigTable() ;
        this.timeout = timeout ;
        this.resendTimeout = resendTimeout ;
        this.retryCount = retryCount ;
        this.resType = resType ;
    }
    
    /**
     * 終了化処理.
     * <BR><BR>
     * 終了化処理です.
     */
    protected void finalize() throws Exception {
        this.table = null ;
        this.indexTable = null ;
        this.timeout = -1L ;
        this.resendTimeout = -1L ;
        this.retryCount = -1 ;
        this.resType = null ;
    }
    
    /**
     * 新しい受信中データを生成.
     * <BR><BR>
     * 新しい受信中データを生成します.
     * <BR>
     * @param id 対象のIDを設定します.
     * @param addr 対象のアドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param size 受信データ長を設定します.
     * @exception InputException 入力例外.
     */
    public synchronized void createUnderReceive(
        long id,InetAddress addr,int port,String cb32Word,int size ) {
        if( this.getNumber( id ) == -1 ) {
            UnderReceiveBean bean = new UnderReceiveBean(
                this.resType,id,addr,port,cb32Word,size ) ;
            table.add( bean ) ;
            indexTable.add( id,bean ) ;
        }
    }
    
    /**
     * 受信中データを削除.
     * <BR><BR>
     * 受信中データをテーブルから削除します.<BR>
     * この処理は、受信データが完了した場合に呼び出されます.
     * <BR>
     * @param id 削除対象のコネクションIDを設定します.
     * @return CompletionBean 削除された受信完了Beanが返されます.<BR>
     *                        [null]が返された場合、情報は存在しません.
     */
    public synchronized CompletionBean removeUnderReceive( long id ) {
        int num = this.getNumber( id ) ;
        if( num != -1 ) {
            UnderReceiveBean bean = ( UnderReceiveBean )table.remove( num ) ;
            if( bean != null ) {
                indexTable.remove( id ) ;
                if( bean.isCompletion() == true ) {
                    return bean.getCompletionBean() ;
                }
            }
        }
        return null ;
    }
    
    /**
     * 受信中データを削除.
     * <BR><BR>
     * 受信中データをテーブルから削除します.
     * <BR>
     * @param num 削除対象の項番を設定します.
     * @return CompletionBean 削除された受信完了Beanが返されます.<BR>
     *                        [null]が返された場合、情報は存在しません.
     */
    public synchronized CompletionBean removeUnderReceive( int num ) {
        if( num < 0 || table.size() <= num ) {
            return null ;
        }
        UnderReceiveBean bean = ( UnderReceiveBean )table.remove( num ) ;
        if( bean != null ) {
            indexTable.remove( bean.getId() ) ;
            if( bean.isCompletion() == true ) {
                return bean.getCompletionBean() ;
            }
        }
        return null ;
    }
    
    /**
     * 指定条件がタイムアウトしている場合、受信中データを削除.
     * <BR><BR>
     * 指定条件がタイムアウトしている場合、受信中データを削除します.
     * <BR>
     * @param num 項番を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、タイムアウトしています.
     *                  [false]が返された場合、タイムアウトしていません.
     */
    public synchronized boolean removeUnderReceiveaByTimeout( int num ) {
        if( num < 0 || table.size() <= num ) {
            return false ;
        }
        
        UnderReceiveBean bean = ( UnderReceiveBean )table.get( num ) ;
        if( bean != null ) {
            if( ( bean.getRetryCount()+1 ) >= retryCount ||
                bean.getCreateTime() + timeout <= System.currentTimeMillis() ) {
                table.remove( num ) ;
                indexTable.remove( bean.getId() ) ;
                return true ;
            }
        }
        
        return false ;
    }
    
    /**
     * 送信待ちタイムアウト値を取得.
     * <BR><BR>
     * 指定されている送信待ちタイムアウト値を取得します.
     * <BR>
     * @return long 送信待ちタイムアウト値が返されます.
     */
    public synchronized long getTimeout() {
        return timeout ;
    }
    
    /**
     * 再送依頼タイムアウト値を取得.
     * <BR><BR>
     * 指定されている再送依頼タイムアウト値を取得します.
     * <BR>
     * @return long 再送依頼タイムアウト値が返されます.
     */
    public synchronized long getResendTimeout() {
        return resendTimeout ;
    }
    
    /**
     * 再送リトライカウント値を取得.
     * <BR><BR>
     * 再送リトライカウント値を取得します.
     * <BR>
     * @return int 再送リトライカウント値が返されます.
     */
    public synchronized int getRetryCount() {
        return retryCount ;
    }
    
    /**
     * コネクションIDを取得.
     * <BR><BR>
     * 項番を設定して、コネクションIDを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return long コネクションIDが返されます.<BR>
     *               [-1L]が返された場合、情報は存在しません.
     */
    public synchronized long getConnectId( int num ) {
        if( num < 0 || table.size() <= num ) {
            return -1L ;
        }
        
        ConnectBean bean = ( ConnectBean )table.get( num ) ;
        if( bean != null ) {
            return bean.getId() ;
        }
        
        return -1L ;
    }
    
    /**
     * 受信中データBeanを取得.
     * <BR><BR>
     * コネクションIDを設定して、受信中データBeanを取得します.
     * <BR>
     * @param id コネクションIDを設定します.
     * @return UnderReceiveBean 受信中データBeanが返されます.<BR>
     *                          [null]が返された場合、情報は存在しません.
     */
    public synchronized UnderReceiveBean getUnderReceiveBean( long id ) {
        return ( UnderReceiveBean )indexTable.get( id ) ;
        //return this.getUnderReceiveBean( this.getNumber( id ) ) ;
    }
    
    /**
     * 受信中データBeanを取得.
     * <BR><BR>
     * 項番を設定して、受信中データBeanを取得します.
     * <BR>
     * @param num 項番を設定します.
     * @return ConnectBean 受信中データBeanが返されます.<BR>
     *                      [null]が返された場合、情報は存在しません.
     */
    public synchronized UnderReceiveBean getUnderReceiveBean( int num ) {
        if( num < 0 || table.size() <= num ) {
            return null ;
        }
        
        return ( UnderReceiveBean )table.get( num ) ;
    }
    
    /**
     * 情報数を取得.
     * <BR><BR>
     * 格納されている情報数を取得します.
     * <BR>
     * @return int 情報数が返されます.
     */
    public synchronized int size() {
        return table.size() ;
    }
    
    /**
     * 格納コネクションID群を取得.
     * <BR><BR>
     * 格納コネクションID群を取得します.
     * <BR>
     * @return long[] 格納されているコネクションID群が返されます.
     */
    public synchronized long[] getConnectIds() {
        return indexTable.getNumbers() ;
    }
    
    /**
     * リソースタイプを取得.
     * <BR><BR>
     * 対象のリソースタイプを取得します.
     * <BR>
     * @return ResourceType 対象のリソースタイプが返されます.
     */
    public synchronized ResourceType getResourceType() {
        return resType ;
    }
    
    /**
     * 対象コネクションIDが存在するかチェック.
     * <BR><BR>
     * 指定したコネクションIDが存在するかチェックします.
     * <BR>
     * @param id 対象のコネクションIDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                  [true]が返された場合、情報は存在します.<BR>
     *                  [false]が返された場合、情報は存在しません.
     */
    public synchronized boolean isConnectId( long id ) {
        return indexTable.isData( id ) ;
        //int num = this.getNumber( id ) ;
        //if( num != -1 ) {
        //    return true ;
        //}
        //return false ;
    }
}

