/*
 * @(#)ProtocolDef.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread ;

import org.maachang.connect.table.Cb32Table;

import org.maachang.commons.conv.CodeBase32;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.resource.cache.CacheDef;
import org.maachang.commons.util.ConvertParam;
import org.maachang.commons.util.UtilCom;

/**
 * プロトコル定義.
 * 
 * @version 2006/12/22
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ProtocolDef {
    
    private ProtocolDef() {}
    
    /**
     * プロトコルヘッダ : コネクション送信.
     */
    public static final int HEADER_SEND_CONNECT = 0x0000ec01 ;
    
    /**
     * プロトコルヘッダ : コネクション受信.
     */
    public static final int HEADER_RECV_CONNECT = 0x0000cc01 ;
    
    /**
     * プロトコルヘッダ : パケット送信.
     */
    public static final int HEADER_SEND_PACKET = 0x0000eac0 ;
    
    /**
     * プロトコルヘッダ : パケット再送依頼.
     */
    public static final int HEADER_RETRY_PACKET = 0x0000ce17 ;
    
    /**
     * プロトコルヘッダ : パケット完了通知.
     */
    public static final int HEADER_EXIT_PACKET = 0x0000c32d ;
    
    /**
     * プロトコルヘッダ : 暗号ヘッダ.
     */
    public static final int HEADER_CB32 = 0x0000cb32 ;
    
    
    /**
     * 暗号電文データ長.
     */
    public static final int CB32_LENGTH_PLUS = 
        2                               // プロトコルヘッダ.
        + 4                             // ヘッダ確認ランダムコード.
        + 1                             // ヘッダ確認ランダムコード認証値.
        + 1                             // ステップコード.
        + 16                            // SHA1.
        + CodeBase32.CODE32_KEY_LENGTH  // 暗号表.
        + 8 ;                           // 電文生成日付.
    
    /**
     * コネクション送信データ長.
     */
    public static final int SEND_CONNECT_LENGTH = 
        2                       // プロトコルヘッダ.
        + 8                     // コネクションID.
        + 4 ;                   // 送信データ長.
    
    /**
     * コネクション受信データ長.
     */
    public static final int RECV_CONNECT_LENGTH =
        2                       // プロトコルヘッダ.
        + 8 ;                   // コネクションID.
    
    
    
    /**
     * 送信パケットヘッダ長.
     */
    public static final int SEND_PACKET_HEADER_LENGTH = 
        2                       // プロトコルヘッダ.
        + 8                     // コネクションID.
        + 2                     // パケット格納データ長.
        + 2                     // チェックコード.
        + 4 ;                   // パケット項番.
    
    /**
     * 送信パケットデータ長.
     */
    public static final int SEND_PACKET_DATA_LENGTH = CacheDef.SECTOR_LENGTH ;
    
    /**
     * 送信パケットプロトコルサイズ.
     */
    public static final int SEND_PACKET_LENGTH = 
        SEND_PACKET_HEADER_LENGTH    // パケットヘッダ長.
        + SEND_PACKET_DATA_LENGTH ;  // 1データ長.
    
    
    
    /**
     * 再送パケットヘッダ長.
     */
    public static final int RETRY_PACKET_HEADER_LENGTH =
        2                       // プロトコルヘッダ.
        + 8                     // コネクションID.
        + 4 ;                   // 再送パケット個数.
    
    /**
     * 完了パケットデータ長.
     */
    public static final int EXIT_PACKET_LENGTH = 
        2                       // プロトコルヘッダ.
        + 8 ;                   // コネクションID.
    
    
    /**
     * 対象電文から、プロトコルタイプを判別.
     * <BR><BR>
     * 対象電文から、プロトコルタイプを判別します.
     * <BR>
     * @param telegram 対象の電文を設定します.
     * @return int プロトコルタイプが返されます.<BR>
     *             [-1]が返された場合、プロトコルタイプは判別できませんでした.
     */
    public static final int getProtocolType( byte[] telegram ) {
        
        if( telegram == null || telegram.length <= 2 ) {
            return -1 ;
        }
        
        return ( int )( ConvertParam.convertShort( 0,telegram ) & 0x0000ffff ) ;
    }
    
    /**
     * データ電文チェックコードを生成.
     * <BR><BR>
     * データ電文チェックコードを生成します.
     * <BR>
     * @param binary チェックコード生成対象の条件を設定します.
     * @return int チェックコードが返されます.
     */
    public static final int checkCode( byte[] binary,int off ) {
        
        int len = binary.length ;
        
        off = off + SEND_PACKET_HEADER_LENGTH ;
        int code = len - off ;
        int bin = 0x01020304 ;
        
        for( int i = off,j = 0 ; i < len ; i ++,j ++ ) {
            bin += ( int )( ( ~( binary[ i ] & 0x000000ff ) ) & 0x0000ffff ) ;
            switch( ( bin + code ) & 0x00000003 ) {
                case 0 :
                    code = ( ( ( ~code ) & 0x0000ffff ) +
                            ( bin - j ) ) & 0x0000ffff ;
                    i++ ;
                    j++ ;
                    break ;
                case 1 :
                    code = ( ( code & 0x0000ffff ) ^
                            ( ( bin | ( ( ~bin ) & 0x000000ff ) << 8 ) +
                            j ) ) & 0x0000ffff ;
                    break ;
                case 2 :
                    code = ( ( code & 0x0000ffff ) -
                            ( bin + j ) ) & 0x0000ffff ;
                    break ;
                case 3 :
                    code = ( ( ( ~code ) & 0x0000ffff ) *
                            ( bin - j ) ) & 0x0000ffff ;
                    i ++ ;
                    j ++ ;
                    break ;
            }
        }
        
        return code ;
    }
    
    /**
     * 暗号基本電文を生成.
     * <BR><BR>
     * 暗号基本電文を生成します.
     * <BR>
     * @param length 対象の電文データを設定します.
     * @return byte[] 対象バイナリが返されます.
     */
    public static final byte[] getCb32Binary( int length ) {
        
        byte[] ret = new byte[ length + CB32_LENGTH_PLUS ] ;
        
        // プロトコルヘッダ.
        int pnt = 0 ;
        ConvertParam.convertShort( ret,pnt,( short )HEADER_CB32 ) ;
        pnt += 2 ;
        
        // ヘッダ確認コード.
        ret[ pnt ] = ( byte )UtilCom.random( 0x00000100 ) ;
        ret[ pnt+1 ] = ( byte )UtilCom.random( 0x00000100 ) ;
        ret[ pnt+2 ] = ( byte )UtilCom.random( 0x00000100 ) ;
        ret[ pnt+3 ] = ( byte )UtilCom.random( 0x00000100 ) ;
        ProtocolDef.checkCb32HeaderCode( ret,pnt ) ;
        pnt += 4 + 1 ;
        
        // ステップコード.
        ret[ pnt ] = 0 ;
        pnt += 1 ;
        
        // SHA1.
        pnt += 16 ;
        
        // 暗号表.
        System.arraycopy(
            CodeBase32.getPublicKey(),0,
            ret,pnt,CodeBase32.CODE32_KEY_LENGTH ) ;
        pnt += CodeBase32.CODE32_KEY_LENGTH ;
        
        // 電文生成日付.
        ConvertParam.convertLong( ret,pnt,System.currentTimeMillis() ) ;
        pnt += 8 ;
        
        return ret ;
        
    }
    
    /**
     * 暗号処理.
     * <BR><BR>
     * 対象電文を暗号化します.
     * <BR>
     * @param table 対象の暗号管理テーブルを設定します.
     * @param cb32Word 対象の暗号ワードを設定します.
     * @param privateKey 対象のプライベートキーを設定します.
     * @param binary 暗号化対象のバイナリを設定します.
     */
    public static final void encryption(
        Cb32Table table,String cb32Word,byte[] binary ) {
        
        // 対象暗号ワードに対する暗号条件を取得.
        byte[] sha1 = null ;
        byte[] privateKey = null ;
        if( cb32Word == null || cb32Word.length() <= 0 ) {
            sha1 = table.defaultSha1() ;
            privateKey = table.defaultPrivateKey() ;
        }
        else {
            sha1 = table.getSha1( cb32Word ) ;
            privateKey = table.getPrivateKey( cb32Word ) ;
            if( sha1 == null || privateKey == null ) {
                sha1 = table.defaultSha1() ;
                privateKey = table.defaultPrivateKey() ;
            }
        }
        
        // SHA1を設定.
        int pnt = 2 + 4 + 1 + 1 ;
        System.arraycopy( sha1,0,binary,pnt,16 ) ;
        pnt += 16 ;
        
        // プライベートKeyを設定.
        CodeBase32 cb32 = new CodeBase32() ;
        cb32.create( privateKey ) ;
        
        // 暗号化.
        byte[] key = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
        System.arraycopy( binary,pnt,key,0,CodeBase32.CODE32_KEY_LENGTH ) ;
        pnt += CodeBase32.CODE32_KEY_LENGTH ;
        int stepCode = cb32.encryption( key,binary,pnt,binary.length-pnt ) ;
        
        // ステップコードを設定.
        binary[ 2 + 4 + 1 ] = ( byte )( stepCode & 0x000000ff ) ;
        
    }
    
    /**
     * 暗号電文を解析して、元のデータとして、再構成.
     * <BR><BR>
     * 暗号電文を解析して、元のデータとして、再構成します.
     * <BR>
     * @param out 対象の暗号ワードが返されます.
     * @param table 対象の暗号管理テーブルを設定します.
     * @param binary 暗号化対象のバイナリを設定します.
     * @return byte[] 元の電文情報が返されます.
     */
    public static final byte[] analysis(
        String[] out,Cb32Table table,byte[] binary ) {
        
        // 受信側は暗号解析対象でない場合.
        if( table == null ) {
            return null ;
        }
        
        CodeBase32 cb32 = new CodeBase32() ;
        
        // SHA1から暗号キーを取得.
        int pnt = 2 + 4 + 1 + 1 ;
        byte[] privateKey = table.getPrivateKey( binary,pnt ) ;
        if( privateKey == null ) {
            return null ;
            //throw new AccessException( "暗号電文に対応する暗号ワードは存在しません" ) ;
        }
        
        // 対象SHA1に対する暗号ワードを取得.
        out[ 0 ] = table.getCb32Word( binary,pnt ) ;
        
        cb32.create( privateKey ) ;
        pnt = 2 + 4 ;
        
        // ヘッダ確認コードを取得.
        int code = ( int )( binary[ pnt ] & 0x000000ff ) ;
        ProtocolDef.checkCb32HeaderCode( binary,2 ) ;
        
        // ヘッダ確認コードと条件が一致しない場合.
        if( code != ( binary[ pnt ] & 0x000000ff ) ) {
            throw new AccessException( "ヘッダ確認コードは一致しない不正な電文です" ) ;
        }
        
        pnt += 1 ;
        
        // ステップコードを取得.
        code = ( int )( binary[ pnt ] & 0x000000ff ) ;
        pnt += 1 + 16 ;
        
        // 暗号解析.
        byte[] key = new byte[ CodeBase32.CODE32_KEY_LENGTH ] ;
        System.arraycopy( binary,pnt,key,0,CodeBase32.CODE32_KEY_LENGTH ) ;
        pnt += CodeBase32.CODE32_KEY_LENGTH ;
        cb32.analysis( key,code,binary,pnt,binary.length-pnt ) ;
        
        // 解析結果から、本来の電文だけを取得.
        pnt += 8 ;
        byte[] ret = new byte[ binary.length-pnt ] ;
        System.arraycopy( binary,pnt,ret,0,binary.length-pnt ) ;
        
        return ret ;
        
    }
    
    /**
     * 暗号ヘッダ確認情報を取得.
     */
    private static final void checkCb32HeaderCode( byte[] binary,int offset ) {
        
        int bin = ( int )(
            ( int )( binary[ offset ] & 0x000000ff ) +
            ( int )( binary[ offset+1 ] & 0x000000ff ) +
            ( int )( binary[ offset+2 ] & 0x000000ff ) +
            ( int )( binary[ offset+3 ] & 0x000000ff ) +
            ( int )( binary[ offset-2 ] & 0x000000ff ) +
            ( int )( binary[ offset-1 ] & 0x000000ff )
        ) ;
        int code = binary.length ;
        
        for( int i = 0,j = offset ; i < 4 ; i ++,j ++ ) {
            bin += ( int )( ( ~( binary[ j ] & 0x000000ff ) ) & 0x0000ffff ) ;
            switch( ( bin + code ) & 0x00000003 ) {
                case 0 :
                    code = ( ( ( ~code ) & 0x0000ffff ) +
                            ( bin - j ) ) & 0x0000ffff ;
                    break ;
                case 1 :
                    code = ( ( code & 0x0000ffff ) ^
                            ( ( bin | ( ( ~bin ) & 0x000000ff ) << 8 ) +
                            j ) ) & 0x0000ffff ;
                    break ;
                case 2 :
                    code = ( ( code & 0x0000ffff ) -
                            ( bin + j ) ) & 0x0000ffff ;
                    break ;
                case 3 :
                    code = ( ( ( ~code ) & 0x0000ffff ) *
                            ( bin - j ) ) & 0x0000ffff ;
                    break ;
            }
        }
        
        binary[ offset+4 ] = ( byte )( code & 0x000000ff ) ;
        
    }

}

