/*
 * @(#)ExecutionUnderRetry.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.connect.thread.execution ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.connect.table.CompletionBean;
import org.maachang.connect.table.MaachangConnectSync;
import org.maachang.connect.table.MaachangConnectTableFactory;
import org.maachang.connect.table.UnderReceiveBean;
import org.maachang.connect.table.UnderReceiveTable;
import org.maachang.connect.thread.MaachangConnectBaseExecution;
import org.maachang.connect.thread.ProtocolDef;

import org.maachang.commons.net.BaseUdpProtocol;
import org.maachang.commons.net.ConnectAddress;
import org.maachang.commons.util.ConvertParam;
import org.maachang.commons.util.UtilCom;

/**
 * 受信待ち再送依頼処理.
 *
 * @version 2006/12/23
 * @author  Masahito Suzuki
 * @since   MaachangConnect 1.00
 */
public class ExecutionUnderRetry implements MaachangConnectBaseExecution {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionUnderRetry.class ) ;
    
    /**
     * 完了電文送信回数.
     */
    private static final int EXIT_TELEGRAM_SEND_COUNT = 3 ;
    
    /**
     * 再送電文長.
     */
    private static final int MAX_RETRY_LENGTH = 500 ;
    
    /**
     * コンストラクタ.
     */
    public ExecutionUnderRetry() {
        
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * MaachangConnectプロトコル実行処理.
     * <BR>
     * @param threadNum スレッド項番を設定されます.
     * @param udp 対象のUDPプロトコルを設定されます.
     * @param tableFactory テーブルFactoryを設定されます.
     * @param sync 同期オブジェクトが設定されます.
     * @param connectAddress 受信処理などに利用するコネクションアドレス.
     * @param cb32Word 暗号解析結果の暗号ワードを格納する情報が設定されます.
     * @exception Exception 処理例外.
     */
    public void execution( int threadNum,BaseUdpProtocol udp,
        MaachangConnectTableFactory tableFactory,MaachangConnectSync sync,
        ConnectAddress connectAddress,String[] cb32Word ) throws Exception {
        
        try {
            
            UtilCom.idleTime() ;
            
            UnderReceiveBean bean = null ;
            UnderReceiveTable underReceiveTable = tableFactory.getUnderReceiveTable() ;
            
            int len ;
            long[] ids = underReceiveTable.getConnectIds() ;
            if( ids != null && ( len = ids.length ) > 0 ) {
                
                // 受信中管理テーブルに設定されている内容を処理.
                for( int i = 0 ; i < len ; i ++ ) {
                    
                    UtilCom.idleTime() ;
                    
                    synchronized( sync.getUnderReceiveSync().get() ) {
                        
                        // １つの受信中管理情報を取得.
                        if( ( bean = underReceiveTable.getUnderReceiveBean( ids[ i ] ) ) == null ) {
                            // 存在しない場合は処理しない.
                            continue ;
                        }
                        
                    }
                    
                    // 受信中電文が完了している場合.
                    if( bean.isCompletion() == true ) {
                        // 送信完了処理.
                        this.completion( tableFactory,underReceiveTable,bean,sync ) ;
                        continue ;
                    }
                    
                    // コネクション送信可能時間を計算.
                    long retryTime = bean.getCreateTime() + 
                        underReceiveTable.getResendTimeout() ;
                    
                    // リトライ要求を満たす条件の場合.
                    if( retryTime <= System.currentTimeMillis() ) {
                        
                        // コネクションデータ送信.
                        this.retry( tableFactory,underReceiveTable,bean,sync ) ;
                        
                    }
                    
                }
                
            }
            
        } catch( Exception e ) {
            LOG.warn( "受信中再送処理で例外",e ) ;
        }
        
    }
    
    /**
     * 受信電文完了処理.
     */
    private final void completion( MaachangConnectTableFactory tableFactory,
        UnderReceiveTable underReceiveTable,UnderReceiveBean bean,MaachangConnectSync sync ) {
        
        LOG.debug( "[ExecutionUnderRetry]完了電文送信" ) ;
        
        CompletionBean comp = null ;
        
        synchronized( sync.getUnderReceiveSync().get() ) {
            
            // 受信完了電文を、受信完了テーブルにセット.
            comp = underReceiveTable.removeUnderReceive( bean.getId() ) ;
            if( comp == null ) {
                return ;
            }
            
        }
        
        // 受信完了テーブルに設定.
        tableFactory.getCompletionTable().add( comp ) ;
        
        // 完了通知電文を送信.
        byte[] bin = null ;
        int pnt = 0 ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            bin = ProtocolDef.getCb32Binary( ProtocolDef.EXIT_PACKET_LENGTH ) ;
            pnt = ProtocolDef.CB32_LENGTH_PLUS ;
        }
        // 非暗号条件の場合.
        else {
            bin = new byte[ ProtocolDef.EXIT_PACKET_LENGTH ] ;
            pnt = 0 ;
        }
        
        // コネクション送信ヘッダ.
        ConvertParam.convertShort(
            bin,pnt,( short )( ProtocolDef.HEADER_EXIT_PACKET & 0x0000ffff ) ) ;
        pnt += 2 ;
        
        // コネクションID.
        ConvertParam.convertLong( bin,pnt,bean.getId() ) ;
        pnt += 8 ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            // 暗号処理.
            ProtocolDef.encryption(
                tableFactory.getCb32Table(),bean.getCb32Word(),bin ) ;
        }
        
        // 送信条件としてセット.
        tableFactory.getSendReceiveTable().addSend(
            bean.getAddress(),bean.getPort(),
            bin,EXIT_TELEGRAM_SEND_COUNT ) ;
    }
    
    /**
     * リトライ電文を送信.
     */
    private final void retry( MaachangConnectTableFactory tableFactory,
        UnderReceiveTable underReceiveTable,UnderReceiveBean bean,MaachangConnectSync sync ) {
        
        LOG.debug( "[ExecutionUnderRetry]リトライ電文送信" ) ;
        
        // リトライ項番を取得.
        int[] packetNos = bean.getRetryPacketIds( MAX_RETRY_LENGTH ) ;
        if( packetNos == null ) {
            return ;
        }
        
        int len = packetNos.length ;
        byte[] bin = null ;
        int pnt = 0 ;
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            bin = ProtocolDef.getCb32Binary( ProtocolDef.RETRY_PACKET_HEADER_LENGTH + ( len * 4 ) ) ;
            pnt = ProtocolDef.CB32_LENGTH_PLUS ;
        }
        // 非暗号条件の場合.
        else {
            bin = new byte[ ProtocolDef.RETRY_PACKET_HEADER_LENGTH + ( len * 4 ) ] ;
            pnt = 0 ;
        }
        
        // 再送ヘッダ.
        ConvertParam.convertShort(
            bin,pnt,( short )( ProtocolDef.HEADER_RETRY_PACKET & 0x0000ffff ) ) ;
        pnt += 2 ;
        
        // コネクションID.
        ConvertParam.convertLong( bin,pnt,bean.getId() ) ;
        pnt += 8 ;
        
        // 再送パケットサイズ.
        ConvertParam.convertInt( bin,pnt,len ) ;
        pnt += 4 ;
        
        // 再送パケットNo群を設定.
        for( int i = 0 ; i < len ; i ++ ) {
            ConvertParam.convertInt( bin,pnt,packetNos[ i ] ) ;
            pnt += 4 ;
        }
        
        // 暗号条件の場合.
        if( tableFactory.isCb32Flag() == true ) {
            // 暗号処理.
            ProtocolDef.encryption(
                tableFactory.getCb32Table(),bean.getCb32Word(),bin ) ;
        }
        
        // 送信条件としてセット.
        tableFactory.getSendReceiveTable().addSend(
            bean.getAddress(),bean.getPort(),bin ) ;
        
        // リトライカウントを更新.
        bean.retryCountIncrement() ;
        
    }
    
}

