/*
 * @(#)ExecutionAdminCache.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminCacheBean;
import org.maachang.queue.access.protocol.admin.AdminCacheProtocol;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.main.cache.MqCache;
import org.maachang.queue.main.cache.MqCacheFactory;
import org.maachang.queue.main.service.SaveAndLoadService;


/**
 * キャッシュ電文処理.
 *
 * @version 2007/01/22
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionAdminCache
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminCache() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminCacheProtocol.CATEGORY_TYPE_ADMIN_CACHE ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect キャッシュオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminCacheBean bean = new AdminCacheBean() ;
        AdminCacheProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminCacheProtocol.TYPE_ADD_CACHE :
                this.addMqCache( session,connect,bean ) ;
                return true ;
            case AdminCacheProtocol.TYPE_REMOVE_CACHE :
                this.removeMqCache( session,connect,bean ) ;
                return true ;
            case AdminCacheProtocol.TYPE_GET_CACHE :
                this.getCache( session,connect,bean ) ;
                return true ;
            case AdminCacheProtocol.TYPE_GET_CACHE_NAMES :
                this.getCacheNames( session,connect,bean ) ;
                return true ;
            case AdminCacheProtocol.TYPE_GET_CACHE_SIZE :
                this.getCacheSize( session,connect,bean ) ;
                return true ;
            case AdminCacheProtocol.TYPE_IS_CACHE :
                this.isMqCache( session,connect,bean ) ;
                return true ;
        }
        
        return false ;
        
    }
    
    /**
     * キャッシュ追加.
     */
    public final void addMqCache(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        try {
            synchronized( MqCacheFactory.getSync() ) {
                
                if( MqCacheFactory.get( bean.getName() ) != null ) {
                    throw new MaachangQException(
                        "キャッシュ(" + bean.getName() +
                        ")は既に存在します",
                        MaachangQErrorCode.ERROR_USE_CACHE ) ;
                }
                
                MqCacheFactory.create(
                    bean.getName(),bean.getMaxSize() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CACHE ) ;
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キャッシュ追加に失敗しました",
                MaachangQErrorCode.ERROR_ADD_CACHE ) ;
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キャッシュ追加" ) ;
        
    }
    
    /**
     * キャッシュ削除.
     */
    public final void removeMqCache(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( MqCacheFactory.getSync() ) {
            
            this.getCache( bean ) ;
            
            try {
                // キャッシュ削除.
                MqCacheFactory.remove( true,bean.getName() ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save( SaveAndLoadService.SAVE_TYPE_CACHE ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キャッシュの削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_CACHE ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キャッシュ削除" ) ;
        
    }
    
    /**
     * キャッシュ情報取得.
     */
    public final void getCache(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqCacheFactory.getSync() ) {
                
                MqCache cache = this.getCache( bean ) ;
                
                bean.setMaxSize( cache.getCacheSize() ) ;
                bean.setUseSize( cache.getUseCacheSize() ) ;
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キャッシュ取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_CACHE ) ;
        }
        
        this.sendMqCache( session,connect,bean ) ;
        
    }
    
    /**
     * キャッシュ名群取得.
     */
    public final void getCacheNames(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqCacheFactory.getSync() ) {
                
                String[] names = MqCacheFactory.getNames() ;
                bean.setParams( names ) ;
                
            }
            
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キャッシュ名一覧に失敗しました",
                MaachangQErrorCode.ERROR_GET_LIST_CACHE ) ;
        }
        
        this.sendMqCache( session,connect,bean ) ;
        
    }
    
    /**
     * キャッシュ数取得.
     */
    public final void getCacheSize(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqCacheFactory.getSync() ) {
                
                int len = MqCacheFactory.size() ;
                bean.setSize( len ) ;
                
            }
            
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キャッシュ数取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_SIZE_CACHE ) ;
        }
        
        this.sendMqCache( session,connect,bean ) ;
        
    }
    
    /**
     * キャッシュ存在チェック.
     */
    public final void isMqCache(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( MqCacheFactory.getSync() ) {
            
            try {
                this.getCache( bean ) ;
            } catch( MaachangQException fa ) {
                throw fa ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キャッシュ存在確認に失敗",
                    MaachangQErrorCode.ERROR_IS_CACHE ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キャッシュ存在" ) ;
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendMqCache(
        LoginSession session,ConnectObject connect,AdminCacheBean bean )
        throws Exception {
        
        byte[] binary = AdminCacheProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
    
    /**
     * キャッシュ取得.
     */
    private final MqCache getCache( AdminCacheBean bean )
        throws Exception {
        
        MqCache cache =  MqCacheFactory.get( bean.getName() ) ;
        if( cache == null ) {
            throw new MaachangQException(
                "キャッシュ(" + bean.getName() +
                ")は存在しません",
                MaachangQErrorCode.ERROR_NOT_CACHE ) ;
        }
        
        return cache ;
    }
}

