/*
 * @(#)ExecutionAdminQueue.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.connect.admin ;

import java.util.HashSet;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.admin.AdminQueueBean;
import org.maachang.queue.access.protocol.admin.AdminQueueProtocol;
import org.maachang.queue.connect.ExecutionAdmin;
import org.maachang.queue.connect.admin.login.LoginSession;
import org.maachang.queue.connect.common.Telegram;
import org.maachang.queue.main.queue.Mq;
import org.maachang.queue.main.queue.MqManager;
import org.maachang.queue.main.queue.QueueKey;
import org.maachang.queue.main.queue.SendMq;
import org.maachang.queue.main.queue.base.BaseQueueBean;
import org.maachang.queue.main.service.SaveAndLoadService;


/**
 * キュー電文処理.
 *
 * @version 2007/01/22
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionAdminQueue
    extends AbstractExecutionAdmin
    implements ExecutionAdmin {
    
    /**
     * コンストラクタ.
     */
    public ExecutionAdminQueue() {
        
    }
    
    /**
     * 管理者実行処理IDを取得.
     * <BR><BR>
     * 設定されている管理者実行処理IDを取得します.
     * <BR>
     * @return int 管理者実行処理IDが返されます.
     */
    public int getId() {
        return AdminQueueProtocol.CATEGORY_TYPE_ADMIN_QUEUE ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * 各管理者用実行処理を実装するためのメソッド.
     * <BR>
     * @param telegramType 処理対象の実行タイプを格納します.
     * @param connect コネクションオブジェクトが設定されます.
     * @param session 対象のログインセッションが設定されます.
     * @param id 対象の電文IDが設定されます.
     * @param telegram 受信された電文が設定されます.
     * @return boolean 処理結果が返されます.
     * @exception Exception 例外.
     */
    public boolean execution( int[] telegramType,LoginSession session,
        ConnectObject connect,int id,Telegram telegram )
        throws Exception {
        
        byte[] binary = telegram.getTelegram() ;
        telegram.clear() ;
        
        // 受信バイナリを解析.
        AdminQueueBean bean = new AdminQueueBean() ;
        AdminQueueProtocol.analysisProtocol( bean,binary ) ;
        telegramType[ 0 ] = bean.getType() ;
        bean.setId( id ) ;
        binary = null ;
        
        switch( bean.getType() ) {
            case AdminQueueProtocol.TYPE_ADD_QUEUE :
                this.addQueue( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_SET_QUEUE :
                this.setQueue( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_REMOVE_QUEUE :
                this.removeQueue( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_GET_QUEUE :
                this.getQueue( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_GET_QUEUE_NAMES :
                this.getQueueNames( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_GET_QUEUE_SIZE :
                this.getQueueSize( session,connect,bean ) ;
                return true ;
            case AdminQueueProtocol.TYPE_IS_QUEUE :
                this.isQueue( session,connect,bean ) ;
                return true ;
        }
        
        return false ;
        
    }
    
    /**
     * キュー追加.
     */
    public final void addQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        BaseQueueBean queueBean = new BaseQueueBean() ;
        queueBean.setType( bean.getQueueType() ) ;
        queueBean.setGzip( bean.isGzipFlag() ) ;
        queueBean.setQueueManagerName( bean.getManager() ) ;
        queueBean.setQueueName( bean.getName() ) ;
        queueBean.setWarning( ( double )bean.getWarningPersend() / 100.0f ) ;
        queueBean.setMaxMessageLength( bean.getMaxMessageLength() ) ;
        queueBean.setMaxQueue( bean.getMaxQueue() ) ;
        queueBean.setAutoCommit( bean.isAutoCommit() ) ;
        queueBean.setCacheName( bean.getCacheName() ) ;
        
        try {
            synchronized( MqManager.getSync() ) {
                
                Mq mq = MqManager.create( queueBean ) ;
                
                if( mq == null ) {
                    throw new MaachangQException(
                        "キュー追加に失敗しました",
                        MaachangQErrorCode.ERROR_ADD_QUEUE ) ;
                }
                
                if( mq instanceof SendMq ) {
                    SendMq send = ( SendMq )mq ;
                    
                    String[] channels = bean.getParams() ;
                    if( channels != null && channels.length > 0 ) {
                        int len = channels.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( channels[ i ] != null ) {
                                channels[ i ] = channels[ i ].trim().toLowerCase() ;
                                if( channels[ i ].length() > 0 ) {
                                    send.addChannel( channels[ i ] ) ;
                                }
                            }
                        }
                    }
                    
                }
                
                // 状態セーブ.
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_QUEUE ) ;
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キュー追加に失敗しました",
                MaachangQErrorCode.ERROR_ADD_QUEUE ) ;
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キュー追加" ) ;
        
    }
    
    /**
     * キュー再設定.
     */
    public final void setQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        try {
            synchronized( MqManager.getSync() ) {
                
                Mq mq = this.getMq( bean ) ;
                
                if( mq instanceof SendMq ) {
                    SendMq send = ( SendMq )mq ;
                    
                    // 削除対象の条件準備.
                    HashSet set = new HashSet() ;
                    if( send.getChannelSize() > 0 ) {
                        String[] names = send.getChannelNames() ;
                        int len = names.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            set.add( names[ i ] ) ;
                        }
                    }
                    
                    String[] channels = bean.getParams() ;
                    
                    // 対象条件が存在する場合は、削除準備から除外.
                    if( channels != null && channels.length > 0 ) {
                        int len = channels.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( channels[ i ] != null ) {
                                channels[ i ] = channels[ i ].trim().toLowerCase() ;
                                if( channels[ i ].length() > 0 ) {
                                    if( set.remove( channels[ i ] ) == true ) {
                                        channels[ i ] = null ;
                                    }
                                }
                            }
                        }
                    }
                    
                    // 削除準備に残った内容を現在のキューから削除.
                    if( set.size() > 0 ) {
                        Object[] objs = set.toArray() ;
                        int len = objs.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            send.removeChannel( ( String )objs[ i ] ) ;
                        }
                    }
                    
                    // 現在条件として存在しないものを追加.
                    if( channels != null && channels.length > 0 ) {
                        int len = channels.length ;
                        for( int i = 0 ; i < len ; i ++ ) {
                            if( channels[ i ] != null && channels[ i ].length() > 0 ) {
                                send.addChannel( channels[ i ] ) ;
                            }
                        }
                    }
                    
                }
                
                // 基本条件を再設定.
                BaseQueueBean queueBean = new BaseQueueBean() ;
                int warning = bean.getWarningPersend() ;
                int max = bean.getMaxQueue() ;
                int maxMsg = bean.getMaxMessageLength() ;
                
                if( warning <= 0 ) {
                    warning = Integer.MAX_VALUE ;
                }
                else if( warning <= MaachangQAccessDefine.MIN_WARNING_PARSEND ) {
                    warning = MaachangQAccessDefine.MIN_WARNING_PARSEND ;
                }
                else if(  warning >= MaachangQAccessDefine.MAX_WARNING_PARSEND ) {
                    warning = MaachangQAccessDefine.MAX_WARNING_PARSEND ;
                }
                
                if( max <= 0 || max <= mq.getMaxSize() ) {
                    max = Integer.MAX_VALUE ;
                }
                
                if( maxMsg <= -1 ) {
                    maxMsg = Integer.MAX_VALUE ;
                }
                
                queueBean.setWarning( warning ) ;
                queueBean.setMaxQueue( max ) ;
                queueBean.setMaxMessageLength( maxMsg ) ;
                
                mq.setQueueBean( queueBean ) ;
                
                // 状態セーブ.
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_QUEUE ) ;
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キュー設定に失敗しました",
                MaachangQErrorCode.ERROR_SET_QUEUE ) ;
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キュー設定" ) ;
        
    }
    
    /**
     * キュー削除.
     */
    public final void removeQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( true,session,bean ) ;
        
        synchronized( MqManager.getSync() ) {
            
            try {
                // キューマネージャ削除.
                MqManager.delete( bean.getManager(),
                    bean.getName(),bean.getQueueType() ) ;
                // 状態セーブ.
                SaveAndLoadService.save(
                    SaveAndLoadService.SAVE_TYPE_QUEUE ) ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューの削除に失敗",
                    MaachangQErrorCode.ERROR_REMOVE_QUEUE ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キュー削除" ) ;
        
    }
    
    /**
     * キュー情報取得.
     */
    public final void getQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqManager.getSync() ) {
                
                Mq mq = this.getMq( bean ) ;
                
                if( mq instanceof SendMq ) {
                    bean.setParams(
                        ( ( SendMq )mq ).getChannelNames() ) ;
                }
                
                BaseQueueBean queueBean = ( BaseQueueBean )mq.getQueueBean() ;
                bean.setGzipFlag( queueBean.isGzip() ) ;
                bean.setWarningPersend(
                    ( int )( queueBean.getWarning() * 100.0f ) ) ;
                bean.setMaxMessageLength( queueBean.getMaxMessageLength() ) ;
                bean.setMaxQueue( queueBean.getMaxQueue() ) ;
                bean.setAutoCommit( queueBean.isAutoCommit() ) ;
                bean.setUseQueue( mq.size() ) ;
                bean.setState( mq.getState() ) ;
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キュー取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_QUEUE ) ;
        }
        
        this.sendQueue( session,connect,bean ) ;
        
    }
    
    /**
     * キュー名群取得.
     */
    public final void getQueueNames(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqManager.getSync() ) {
                
                QueueKey[] keys = MqManager.getQueues(
                    true,bean.getManager() ) ;
                
                if( keys != null && keys.length > 0 ) {
                    int len = keys.length ;
                    int[] nums = new int[ len ] ;
                    String[] names = new String[ len ] ;
                    for( int i = 0 ; i < len ; i ++ ) {
                        nums[ i ] = keys[ i ].getQueueType() ;
                        names[ i ] = keys[ i ].getName() ;
                    }
                    bean.setNumbers( nums ) ;
                    bean.setParams( names ) ;
                }
                else {
                    bean.setNumbers( null ) ;
                    bean.setParams( null ) ;
                }
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キュー名一覧に失敗しました",
                MaachangQErrorCode.ERROR_GET_LIST_QUEUE ) ;
        }
        
        this.sendQueue( session,connect,bean ) ;
        
    }
    
    /**
     * キュー数取得.
     */
    public final void getQueueSize(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        try {
            synchronized( MqManager.getSync() ) {
                
                int len = MqManager.size(
                    bean.getManager() ) ;
                
                bean.setSize( len ) ;
                
            }
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            LOG.error( "error",e ) ;
            throw new MaachangQException(
                "キュー数取得に失敗しました",
                MaachangQErrorCode.ERROR_GET_SIZE_QUEUE ) ;
        }
        
        this.sendQueue( session,connect,bean ) ;
        
    }
    
    /**
     * キュー存在チェック.
     */
    public final void isQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        super.checkLogin( false,session,bean ) ;
        
        synchronized( MqManager.getSync() ) {
            
            try {
                this.getMq( bean ) ;
            } catch( MaachangQException fa ) {
                throw fa ;
            } catch( Exception e ) {
                LOG.error( "error",e ) ;
                throw new MaachangQException(
                    "キューマネージャ存在確認に失敗",
                    MaachangQErrorCode.ERROR_IS_QUEUE ) ;
            }
            
        }
        
        super.sendResult( connect,bean,
            MaachangQErrorCode.SUCCESS,"キュー存在" ) ;
    }
    
    /**
     * 戻り電文送信.
     */
    private final void sendQueue(
        LoginSession session,ConnectObject connect,AdminQueueBean bean )
        throws Exception {
        
        byte[] binary = AdminQueueProtocol.createProtocol(
            bean.getId(),session.isOwner(),bean ) ;
        
        connect.send( binary ) ;
        
    }
    
    /**
     * キュー取得.
     */
    private final Mq getMq( AdminQueueBean bean )
        throws Exception {
        
        return MqManager.get( true,bean.getManager(),
            bean.getName(),bean.getQueueType() ) ;
        
    }
}

