/*
 * @(#)MqCacheFactory.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.cache ;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.sys.NamingManager;
import org.maachang.commons.util.SearchPortion;
import org.maachang.queue.config.MqDefine;

/**
 * キャッシュファクトリ.
 *  
 * @version 2006/09/02
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class MqCacheFactory
{
    
    /**
     * ネーミングマネージャ登録拡張子.
     */
    public static final String NAMING_PLUS = "@maachangq.cache.factory" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public MqCacheFactory() {
        
    }
    
    /**
     * コネクション情報を全て破棄.
     * <BR><BR>
     * コネクション情報を全て破棄します.
     */
    public final void destroy() {
        
        synchronized( SYNC ) {
            
            int len ;
            String[] names = MqCacheFactory.getNames() ;
            if( names != null && ( len = names.length ) > 0 ) {
                for( int i = 0 ; i < len ; i ++ ) {
                    try {
                        MqCacheFactory.remove( false,names[ i ] ) ;
                    } catch( Exception e ) {
                    }
                }
            }
            
        }
    }
    
    /**
     * キャッシュ生成.
     * <BR><BR>
     * 条件を設定してキャッシュを生成します.
     * <BR>
     * @param bean キャッシュBeanを設定します.
     * @exception InputException 入力例外.
     * @exceptino AccessException アクセス例外.
     */
    public static final void create( MqCacheBean bean )
        throws InputException,AccessException {
        
        if( bean == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        MqCacheFactory.create(
            bean.getQueueCacheName(),bean.getCacheSize() ) ;
        
    }
    
    /**
     * キャッシュ生成.
     * <BR><BR>
     * 条件を設定してキャッシュを生成します.
     * <BR>
     * @param naming ネーミングサービスに登録するための名前を設定します.
     * @param size キャッシュセクタ数を設定します.<BR>
     *             設定可能な最小値は[MqDefine.MIN_CACHE_SECTOR]です.<BR>
     *             設定可能な最大値は[MqDefine.MAX_CACHE_SECTOR]です.
     * @exception InputException 入力例外.
     * @exceptino AccessException アクセス例外.
     */
    public static final void create( String naming,int size )
        throws InputException,AccessException {
        
        String key = null ;
        MqCacheImple imple = null ;
        
        if(
            naming == null || ( naming = naming.trim().toLowerCase() ).length() <= 0 ||
            ( key = MqCacheFactory.createNaming( naming ) ) == null ) {
            throw new InputException( "引数は不正です" ) ;
        }
        else if( NamingManager.get( key ) != null ) {
            throw new InputException( "対象のキャッシュ名[" + naming +
                "]は既に登録されています" ) ;
        }
        
        if( size <= MqDefine.MIN_CACHE_SECTOR ) {
            size = MqDefine.MIN_CACHE_SECTOR ;
        }
        else if( size >= MqDefine.MAX_CACHE_SECTOR ) {
            size = MqDefine.MAX_CACHE_SECTOR ;
        }
        
        synchronized( SYNC ) {
            
            imple = new MqCacheImple( naming,size ) ;
            NamingManager.add( key,imple ) ;
            
        }
        
    }
    
    /**
     * キャッシュ削除.
     * <BR><BR>
     * キャッシュオブジェクトを削除します.
     * <BR>
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( String naming ) {
        MqCacheFactory.remove( false,naming ) ;
    }
    
    /**
     * キャッシュ削除.
     * <BR><BR>
     * キャッシュオブジェクトを削除します.
     * <BR>
     * @param delete 削除モードを設定します.<BR>
     *               [true]を設定した場合、実際のキャッシュファイルも削除します.<BR>
     *               [false]を設定した場合、実際のキャッシュファイルは削除されません.
     * @param naming 削除対象のネーミングを設定します.
     */
    public static final void remove( boolean delete,String naming ) {
        
        Object o = null ;
        
        if( ( naming = MqCacheFactory.createNaming( naming ) ) == null ) {
            return ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            
            if( o != null && o instanceof MqCacheImple ) {
                
                NamingManager.remove( naming ) ;
                
                if( delete == true ) {
                    ( ( MqCacheImple )o ).deleteCacheToDestroy() ;
                }
                
            }
            
        }
        
    }
    
    /**
     * キャッシュ取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @param naming 取得対象のネーミングを設定します.
     * @return MqCache キャッシュ情報が返されます.
     */
    public static final MqCache get( String naming ) {
        
        Object o = null ;
        MqCache ret = null ;
        
        if( ( naming = MqCacheFactory.createNaming( naming ) ) == null ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            
            o = NamingManager.get( naming ) ;
            if( o != null && o instanceof MqCache ) {
                ret = ( MqCache )o ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されているキャッシュ名一覧を取得.
     * <BR><BR>
     * 登録されているキャッシュ名一覧を取得します.
     * <BR>
     * @return String[] キャッシュ名一覧が返されます.
     */
    public static final String[] getNames() {
        
        int i ;
        int len ;
        int namingLen ;
        
        String[] ret = null ;
        
        synchronized( SYNC ) {
            
            ret = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( ret != null && ( len = ret.length ) > 0 ) {
                namingLen = NAMING_PLUS.length() ;
                for( i = 0 ; i < len ; i ++ ) {
                    
                    ret[ i ] = ret[ i ].substring( 0,ret[ i ].length() - namingLen ) ;
                    
                }
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 登録されている情報長を取得.
     * <BR><BR>
     * 登録されている情報長を取得します.
     * <BR>
     * @return int 登録されている情報長が返されます.
     */
    public static final int size() {
        
        int ret = 0 ;
        
        synchronized( SYNC ) {
            
            String[] names = SearchPortion.searchString(
                NamingManager.getNames(),
                new StringBuffer().append( "*" ).append( NAMING_PLUS ).toString() ) ;
            
            if( names != null ) {
                ret = names.length ;
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 同期オブジェクトを取得.
     * <BR><BR>
     * 同期オブジェクトを取得します.
     * <BR>
     * @return Object 同期オブジェクトが返されます.
     */
    public static final Object getSync() {
        return SYNC ;
    }
    
    
    /**
     * キャッシュファクトリ名生成.
     */
    private static final String createNaming( String name ) {
        
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        return new StringBuffer().append( name ).append( NAMING_PLUS ).toString() ;
        
    }
}

