/*
 * @(#)ExecutionResultSuccessSwitch.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.channel.service.receive ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.NumberTable;
import org.maachang.queue.access.status.ChannelStatus;
import org.maachang.queue.main.channel.Channel;
import org.maachang.queue.main.channel.ChannelFactory;
import org.maachang.queue.main.channel.SendChannel;
import org.maachang.queue.main.channel.protocol.ChannelProtocol;
import org.maachang.queue.main.channel.protocol.ProtocolHeartBeat;
import org.maachang.queue.main.channel.protocol.ProtocolObject;
import org.maachang.queue.main.channel.protocol.ProtocolResultSuccess;
import org.maachang.queue.main.channel.service.receive.core.CoreReceiveChild;
import org.maachang.queue.main.channel.service.send.ChannelOption;
import org.maachang.queue.main.channel.service.send.SendQueueManage;
import org.maachang.queue.main.channel.service.send.SendService;
import org.maachang.queue.main.connect.Connect;
import org.maachang.queue.main.connect.ConnectFactory;
import org.maachang.queue.main.queue.base.core.QSendSeq;

/**
 * 受信電文[ProtocolResultSuccess]処理実装.
 *
 * @version 2006/12/21
 * @author  Masahito Suzuki
 * @since   MaachangQ 1.00
 */
public class ExecutionResultSuccessSwitch implements ExecutionSwitch {
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( ExecutionResultSuccessSwitch.class ) ;
    
    /**
     * 戻りエラープロトコルプーリング.
     */
    private final NumberTable pool = new NumberTable() ;
    
    /**
     * プロトコルタイプを取得.
     * <BR><BR>
     * プロトコルタイプを取得します.
     * <BR>
     * @return int プロトコルタイプが返されます.<BR>
     *             [ProtocolObject.PROTOCOL_DATA]が返された場合、データオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_HEARTBEAT]が返された場合、ハートビートオブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_SUCCESS]が返された場合、正常オブジェクトです.<BR>
     *             [ProtocolObject.PROTOCOL_ERROR]が返された場合ｍエラーオブジェクトです.
     */
    public int getType() {
        return ProtocolObject.PROTOCOL_SUCCESS ;
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param threadNum スレッド項番が設定されます.
     * @param child 処理対象の要素が設定されます.
     * @param sync 同期オブジェクト.
     */
    public void execution( int threadNum,CoreReceiveChild child,Synchronized sync ) {
        
        try {
            
            // プロトコルをプールオブジェクトから取得.
            ProtocolResultSuccess resultSuccess = ( ProtocolResultSuccess )pool.get(
                threadNum ) ;
            
            // 情報が存在しない場合は、生成して、プールに設定.
            if( resultSuccess == null ) {
                resultSuccess = new ProtocolResultSuccess() ;
                pool.add( threadNum,resultSuccess ) ;
            }
            
            // 電文情報から、戻り正常電文を生成.
            ChannelProtocol.getTelegramByProtocolResultSuccess(
                    resultSuccess,child.getBinResource() ) ;
            
            // チャネルを取得.
            Channel channel = ChannelFactory.getToID(
                resultSuccess.getChannelId() ) ;
            
            // 取得したチャネルが不正な場合.
            if( channel == null || channel.isChannel() == false ||
                ( channel instanceof SendChannel ) == false ) {
                // 処理しない.
                return ;
            }
            
            // 接続先のコネクションを取得.
            Connect conn = ConnectFactory.get(
                ( ( SendChannel)channel ).getConnectName() ) ;
            
            // 対象チャネルのコネクションポートと、
            // 受信されたときのコネクションポートが一致しない場合.
            if( conn == null || conn.getPort() != resultSuccess.getReceivePort() ) {
                // 処理しない.
                return ;
            }
            
            // 戻り正常電文の処理元タイプで処理分離.
            switch( resultSuccess.getReturnType() ) {
                case ProtocolObject.PROTOCOL_DATA :
                    // データ電文処理.
                    this.resultSuccessByData( child,channel,conn,resultSuccess,sync ) ;
                    break ;
                case ProtocolObject.PROTOCOL_HEARTBEAT :
                    // ハートビート電文処理.
                    this.resultSuccessByHeartBeat( child,channel,conn,resultSuccess,sync ) ;
                    break ;
            }
            
        } catch( NullPointerException nul ) {
            throw nul ;
        } catch( OutOfMemoryError me ) {
            LOG.error( "[(Receive)ResultSuccess]OutOfMemoryError",me ) ;
        } catch( Exception e ) {
            LOG.warn( "[(Receive)ResultSuccess]受信処理時に例外",e ) ;
        }
        
    }
    
    /**
     * ハートビート電文系正常処理.
     */
    private final void resultSuccessByHeartBeat(
        CoreReceiveChild child,Channel channel,Connect con,
        ProtocolResultSuccess resultSuccess,Synchronized sync ) {
        
        // チャネルオプションを取得.
        ChannelOption opt = ( ChannelOption )(
            ( SendChannel )channel ).getOption() ;
        
        // チャネルオプションが存在しない場合.
        if( opt == null ) {
            return ;
        }
        
        synchronized( opt ) {
            
            // 対象のハートビートプロトコルを取得.
            ProtocolHeartBeat heart = opt.getSendProtocol() ;
            
            // 対象送信チャネルIDと、受信されたときのIDが一致しない場合.
            if( heart == null || resultSuccess.getId() != heart.getId() ) {
                // 処理しない.
                return ;
            }
            
            opt.clearSendCount() ;
            
            // チャネルステータスが、シャットダウン中か、
            // 計画停止中以外の場合.
            if( channel.getState() != ChannelStatus.STATE_DOWN &&
                channel.getState() != ChannelStatus.STATE_SHUTDOWN &&
                channel.getState() != ChannelStatus.STATE_PLANNED_STOP ) {
                
                channel.setState( ChannelStatus.STATE_SUCCESS ) ;
                
            }
            
        }
        
    }
    
    /**
     * データ電文系正常処理.
     */
    private final void resultSuccessByData(
        CoreReceiveChild child,Channel channel,Connect con,
        ProtocolResultSuccess resultSuccess,Synchronized sync ) {
        
        // データ電文送信管理オブジェクトを取得.
        SendQueueManage man = SendService.getSendQueueManage() ;
        
        // データ電文送信管理オブジェクトが存在しない.
        if( man == null ) {
            return ;
        }
        
        // 対象の受信情報を破棄.
        man.receiveChannelId(
            resultSuccess.getQueueManagerName(),resultSuccess.getName(),
            QSendSeq.getNowNextIdByNowId( resultSuccess.getId() ),
            resultSuccess.getChannelId() ) ;
        
    }
    
}

