/*
 * @(#)MonitorManagerThread.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.manager.service ;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.maachang.commons.exception.BaseException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.thread.ExecutionThread;
import org.maachang.commons.thread.LoopThread;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.access.status.QueueManagerStatus;
import org.maachang.queue.access.status.QueueStatus;
import org.maachang.queue.main.manager.BaseQueueManager;
import org.maachang.queue.main.manager.QueueManagerFactory;
import org.maachang.queue.main.queue.QueueKey;
import org.maachang.queue.main.queue.base.BaseQueue;

/**
 * キューマネージャ情報監視スレッド.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
class MonitorManagerThread extends ExecutionThread {
    
    /**
     * 待機時間.
     */
    private static final int WAIT_TIME = 250 ;
    
    /**
     * ログオブジェクト.
     */
    private static final Log LOG = LogFactory.getLog( MonitorManagerThread.class ) ;
    
    /**
     * ループスレッド.
     */
    private final LoopThread thread = new LoopThread() ;
    
    /**
     * 同期処理.
     */
    private final Synchronized sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     */
    public MonitorManagerThread(){
        
        sync.create() ;
        
        try{
            
            thread.create( this ) ;
            thread.startThread() ;
            
        }catch( Exception e ){
            this.destroy() ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * <BR>
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public final void destroy() {
        sync.clear() ;
        thread.clear() ;
    }
    
    /**
     * すべてのキューマネージャをシャットダウン中に設定.
     * <BR><BR>
     * すべてのキューマネージャをシャットダウン中に設定します.
     */
    public final void shutdownManager() {
        
        String[] names = QueueManagerFactory.getNames() ;
        if( names != null && names.length > 0 ) {
            int len = names.length ;
            
            for( int i = 0 ; i < len ; i ++ ) {
                
                UtilCom.idleTime() ;
                
                BaseQueueManager man = ( BaseQueueManager )QueueManagerFactory.
                    get( names[ i ] ) ;
                
                if( man == null || man.isQueueManager() == false ) {
                    continue ;
                }
                
                man.setState( QueueManagerStatus.STATE_SHUTDOWN ) ;
            }
            
        }
        
    }
    
    /**
     * スレッド状態を取得.
     * <BR><BR>
     * スレッド状態を取得します.
     * <BR>
     * @return boolean スレッド状態が返されます.<BR>
     *                 [true]が返された場合、スレッドは実行中です.<BR>
     *                 [false]が返された場合、スレッドは停止中です.
     */
    public final boolean isThread()
    {
        boolean ret ;
        
        try{
            synchronized( sync.get() ){
                ret = thread.isThread() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 実行初期化処理をサポートします.
     * <BR><BR>
     * 実行初期化処理をサポートします.<BR>
     * この処理は、スレッド処理が開始された時に呼び出されます.
     * <BR>
     * @param obj 実行開始時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void init( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行終了化処理をサポートします.
     * <BR><BR>
     * 実行終了化処理をサポートします.<BR>
     * この処理は、スレッド処理が終了された時に呼び出されます.
     * <BR>
     * @param obj 実行終了時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void exit( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * ストップ処理をサポートします。
     * <BR><BR>
     * ストップ処理をサポートします。<BR>
     * この処理は、スレッドでのストップ処理に対して呼び出し実行されます.
     * <BR>
     * @param obj ストップ時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void stop( Object obj )
        throws ExecutionException
    {
        
    }
    
    /**
     * 実行処理をサポートします。
     * <BR><BR>
     * 実行処理をサポートします。<BR>
     * この処理は、スレッドでの実行処理に対して呼び出し実行されます.
     * <BR>
     * @param obj 実行時に設定されます.
     * @exception ExecutionException 実行例外
     */
    protected final void execution( Object obj )
        throws ExecutionException {
        
        try{
            
            UtilCom.idleSleep( WAIT_TIME ) ;
            
            String[] names = QueueManagerFactory.getNames() ;
            if( names != null && names.length > 0 ) {
                int len = names.length ;
                
                if( len > 0 ) {
                    for( int i = 0 ; i < len ; i ++ ) {
                        
                        UtilCom.idleSleep( WAIT_TIME ) ;
                        
                        BaseQueueManager man = ( BaseQueueManager )QueueManagerFactory.
                            get( names[ i ] ) ;
                        
                        if( man == null || man.isQueueManager() == false ) {
                            continue ;
                        }
                        
                        // キューマネージャ内のキュー情報一覧を取得.
                        QueueKey[] keys = man.getKeyElements() ;
                        
                        if( keys != null && keys.length > 0 ) {
                            
                            int errorCnt = 0 ;
                            int lenJ = keys.length ;
                            
                            // 存在するキューのステータスをチェック.
                            for( int j = 0 ; j < lenJ ; j ++ ) {
                                
                                UtilCom.idleSleep( WAIT_TIME ) ;
                                
                                BaseQueue queue = ( BaseQueue )( ( BaseQueueManager )man ).getQueue(
                                    keys[ i ] ) ;
                                if( queue == null || queue.isQueue() == false ) {
                                    continue ;
                                }
                                
                                // ステータスが「正常」「警告」「キューフル」以外の場合.
                                if( queue.getState().getState() != QueueStatus.STATE_SUCCESS &&
                                    queue.getState().getState() != QueueStatus.STATE_WARNING &&
                                    queue.getState().getState() != QueueStatus.STATE_FULL ) {
                                    // エラーカウントを１インクリメント.
                                    errorCnt ++ ;
                                    break ;
                                }
                                
                            }
                            
                            // キューマネージャに属するすべてのキューがエラー状態で、
                            // 現在のキューマネージャステータスが
                            // 「停止」「シャットダウン中」「開始中」「エラー」「計画停止」
                            // 以外のステータスの場合は「エラー」とする.
                            if( errorCnt == lenJ &&
                                man.getState() != QueueManagerStatus.STATE_DOWN &&
                                man.getState() != QueueManagerStatus.STATE_SHUTDOWN &&
                                man.getState() != QueueManagerStatus.STATE_STARTUP &&
                                man.getState() != QueueManagerStatus.STATE_ERROR &&
                                man.getState() != QueueManagerStatus.STATE_PLANNED_STOP ) {
                                
                                man.setState( QueueManagerStatus.STATE_ERROR ) ;
                            }
                            // ステータスが正常で、キュー数が0の場合は、エラーとする.
                            else if( lenJ == 0 &&
                                man.getState() == QueueManagerStatus.STATE_SUCCESS ) {
                                
                                man.setState( QueueManagerStatus.STATE_ERROR ) ;
                            }
                            // どれか１つのキューマネージャが利用可能な場合.
                            else if( errorCnt != lenJ &&
                                ( man.getState() == QueueManagerStatus.STATE_STARTUP ||
                                    man.getState() == QueueManagerStatus.STATE_ERROR ) ) {
                                
                                man.setState( QueueManagerStatus.STATE_SUCCESS ) ;
                                
                            }
                            
                        }
                        
                    }
                }
                else {
                    UtilCom.idleTime() ;
                }
                
            }
            else {
                UtilCom.idleTime() ;
            }
            
        }catch( NullPointerException nul ){
            throw new ExecutionException(
                nul,ExecutionException.LEVEL_STOP
            ) ;
        }catch( BaseException be ){
            LOG.error( "エラーが発生しました", be ) ;
        }catch( Exception e ){
            LOG.error( "エラーが発生しました", e ) ;
        }finally{
            
        }
        
    }
    
}

