/*
 * @(#)QArrayChild.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base.core ;

import java.util.Date;

import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.UtilCom;
import org.maachang.queue.main.queue.QueueDef;

/**
 * キュー配列子要素.
 * <BR><BR>
 * キュー配列の子要素を示します.
 *  
 * @version 2005/12/29
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QArrayChild
{
    
    /**
     * ステータス : 無効.
     */
    public static final int STATE_NOT = 0x00000000 ;
    
    /**
     * ステータス : 生成された.
     */
    public static final int STATE_CREATE = 0x00000001 ;
    
    /**
     * ステータス : 確定されているオブジェクト.
     */
    public static final int STATE_TO_COMMIT = 0x00000002 ;
    
    /**
     * ステータス : 削除対象のオブジェクト.
     */
    public static final int STATE_TO_DELETE = 0x00000004 ;
    
    /**
     * ステータス : 削除受理のオブジェクト.
     */
    public static final int STATE_TO_DELETE_OK = 0x00000005 ;
    
    
    
    /**
     * デフォルト優先順位.
     */
    protected static final int DEF_PRIORITY = QueueDef.DEF_PRIORITY ;
    
    /**
     * 基本最大プライオリティ.
     */
    protected static final int MAX_BASE_PRIORITY = QueueDef.MAX_BASE_PRIORITY ;
    
    /**
     * 最大優先順位.
     */
    protected static final int MAX_PRIORITY = QueueDef.MAX_PRIORITY ;
    
    /**
     * 最小優先順位.
     */
    protected static final int MIN_PRIORITY = QueueDef.MIN_PRIORITY ;
    
    
    
    /**
     * このオブジェクトの前に存在する子要素.
     */
    protected QArrayChild m_before = null ;
    
    /**
     * このオブジェクトの後に存在する子要素.
     */
    protected QArrayChild m_next = null ;
    
    /**
     * 優先順位.
     */
    protected int m_priority = DEF_PRIORITY ;
    
    /**
     * オブジェクトID.
     */
    protected long m_id = -1L ;
    
    /**
     * 送信ID.
     */
    protected long m_sendID = 0L ;
    
    /**
     * オプション.
     */
    protected Object m_option = null ;
    
    /**
     * オブジェクト確定フラグ.
     */
    protected boolean m_commitFlag = false ;
    
    /**
     * このオブジェクトを検索するKEY.
     */
    protected String[] m_key = new String[ 2 ] ;
    
    /**
     * このオブジェクトに格納する内容.
     */
    protected final Object[] m_value = new Object[ 2 ] ;
    
    /**
     * プロセスID.
     */
    protected String[] m_processId = new String[ 2 ] ;
    
    /**
     * オブジェクトステータス.
     */
    protected final int[] m_state = new int[ 2 ] ;
    
    /**
     * Expire.
     */
    protected final long[] m_expire = new long[ 2 ] ;
    
    /**
     * 生成時間.
     */
    protected final long[] m_createDate = new long[ 2 ] ;
    
    /**
     * QueueArrayオブジェクト.
     */
    protected QArray m_qarray = null ;
    
    /**
     * 同期オブジェクト.
     */
    protected Synchronized m_sync = null ;
    
    
    
    /**
     * コンストラクタ.
     */
    protected QArrayChild(){}
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象要素を設定してオブジェクトを生成します.
     * <BR>
     * @param qarray 対象のQArrayオブジェクトを設定します.
     * @param value 設定対象の要素オブジェクトを設定します.
     * @param priority 対象の優先順位を設定します.<BR>
     *                 設定最小値は[0]です.<BR>
     *                 設定最大値は[10000]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     */
    public QArrayChild( QArray qarray,Object value,int priority,long expire )
    {
        m_before = null ;
        m_next = null ;
        
        m_qarray = qarray ;
        m_sync = qarray.m_sync ;
        
        expire = ( expire <= 0 ) ? -1 : expire ;
        
        m_id = qarray.getID() ;
        
        m_commitFlag = qarray.m_autoCommitFlag ;
        m_priority = ( priority >= MAX_PRIORITY ) ?
            MAX_PRIORITY : 
        (
            ( priority <= MIN_PRIORITY ) ?
                MIN_PRIORITY : priority
        ) ;
        
        m_value[ 0 ] = value ;
        m_value[ 1 ] = null ;
        m_key[ 0 ] = "" ;
        m_key[ 1 ] = null ;
        m_processId[ 0 ] = "" ;
        m_processId[ 1 ] = null ;
        
        m_state[ 0 ] = STATE_CREATE ;
        m_state[ 1 ] = STATE_NOT ;
        
        m_expire[ 0 ] = expire ;
        m_expire[ 1 ] = Long.MAX_VALUE ;
        
        m_createDate[ 0 ] = System.currentTimeMillis() ;
        m_createDate[ 1 ] = -1L ;
        
        m_sendID = 0L ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * オブジェクトをクリア.
     * <BR><BR>
     * オブジェクトをクリアします.
     */
    public void clear()
    {
        boolean flg = false ;
        try{
            synchronized( m_sync.get() ){
                
                this.cut() ;
                
                m_before = null ;
                m_next = null ;
                m_qarray = null ;
                m_sync = null ;
                
                m_id = -1L ;
                m_sendID = 0L ;
                m_option = null ;
                m_commitFlag = false ;
                m_priority = DEF_PRIORITY ;
                
                m_key[ 0 ] = null ;
                m_key[ 1 ] = null ;
                m_value[ 0 ] = null ;
                m_value[ 1 ] = null ;
                
                m_processId[ 0 ] = null ;
                m_processId[ 1 ] = null ;
                
                m_state[ 0 ] = STATE_NOT ;
                m_state[ 1 ] = STATE_NOT ;
                
                m_expire[ 0 ] = Long.MAX_VALUE ;
                m_expire[ 1 ] = Long.MAX_VALUE ;
                
                m_createDate[ 0 ] = -1L ;
                m_createDate[ 1 ] = -1L ;
                
                flg = true ;
                
            }
        }catch( Exception e ){
        }
        
        if( flg == false ){
            
            m_before = null ;
            m_next = null ;
            m_qarray = null ;
            m_sync = null ;
            
            m_id = -1L ;
            m_sendID = 0L ;
            m_option = null ;
            m_commitFlag = false ;
            m_priority = DEF_PRIORITY ;
            
            m_key[ 0 ] = null ;
            m_key[ 1 ] = null ;
            m_value[ 0 ] = null ;
            m_value[ 1 ] = null ;
            
            m_processId[ 0 ] = null ;
            m_processId[ 1 ] = null ;
            
            m_state[ 0 ] = STATE_NOT ;
            m_state[ 1 ] = STATE_NOT ;
            
            m_expire[ 0 ] = Long.MAX_VALUE ;
            m_expire[ 1 ] = Long.MAX_VALUE ;
            
            m_createDate[ 0 ] = -1L ;
            m_createDate[ 1 ] = -1L ;
            
        }
        
    }
    
    
    /**
     * 確定処理.
     * <BR><BR>
     * 確定処理を実施します.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、このオブジェクトは破棄する必要があります.
     */
    protected boolean commit()
    {
        boolean ret = false ;
        
        try{
            synchronized( m_sync.get() ){
                if(
                    this.getAutoCommit() == false &&
                    m_commitFlag == false
                )
                {
                    this.decDeleteSize() ;
                    
                    if(
                        m_state[ 0 ] != STATE_CREATE &&
                        m_state[ 0 ] != STATE_TO_COMMIT
                    )
                    {
                        ret = true ;
                    }
                    else{
                        
                        m_key[ 1 ] = null ;
                        m_value[ 1 ] = null ;
                        m_processId[ 1 ] = null ;
                        m_state[ 1 ] = STATE_NOT ;
                        m_expire[ 1 ] = Long.MAX_VALUE ;
                        m_createDate[ 1 ] = -1L ;
                        
                        m_state[ 0 ] = STATE_TO_COMMIT ;
                        m_commitFlag = true ;
                        
                    }
                }
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ロールバック処理.
     * <BR><BR>
     * 設定内容を前回情報に戻します.
     * <BR>
     * @return boolean 処理結果が返されます.<BR>
     *                 [true]が返された場合、このオブジェクトは破棄する必要があります.
     */
    protected boolean rollback()
    {
        boolean ret = false ;
        
        try{
            synchronized( m_sync.get() ){
                if(
                    this.getAutoCommit() == false &&
                    m_commitFlag == false
                )
                {
                    this.decDeleteSize() ;
                    
                    if(
                        m_state[ 0 ] == STATE_CREATE ||
                        m_state[ 1 ] == STATE_CREATE
                    )
                    {
                        ret = true ;
                    }
                    else{
                        
                        m_key[ 0 ] = m_key[ 1 ] ;
                        m_key[ 1 ] = null ;
                        m_value[ 0 ] = m_value[ 1 ] ;
                        m_value[ 1 ] = null ;
                        m_processId[ 0 ] = m_processId[ 1 ] ;
                        m_processId[ 1 ] = null ;
                        m_state[ 0 ] = m_state[ 1 ] ;
                        m_state[ 1 ] = STATE_NOT ;
                        m_expire[ 0 ] = m_expire[ 1 ] ;
                        m_expire[ 1 ] = Long.MAX_VALUE ;
                        m_createDate[ 0 ] = m_createDate[ 1 ] ;
                        m_createDate[ 1 ] = -1L ;
                        
                        m_commitFlag = true ;
                        
                    }
                }
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * このオブジェクトを削除.
     * <BR><BR>
     * このオブジェクトを削除します.
     */
    protected void delete()
    {
        
        try{
            synchronized( m_sync.get() ){
                
                if(
                    this.getAutoCommit() == false &&
                    m_state[ 0 ] != STATE_TO_DELETE_OK &&
                    m_state[ 0 ] != STATE_TO_DELETE &&
                    m_state[ 0 ] != STATE_NOT
                )
                {
                    if( m_key[ 1 ] == null ){
                        m_key[ 1 ] = m_key[ 0 ] ;
                        m_key[ 0 ] = null ;
                    }
                    
                    if( m_value[ 1 ] == null ){
                        m_value[ 1 ] = m_value[ 0 ] ;
                        m_value[ 0 ] = null ;
                    }
                    
                    if( m_processId[ 1 ] == null ) {
                        m_processId[ 1 ] = m_processId[ 0 ] ;
                        m_processId[ 0 ] = null ;
                    }
                    
                    if( m_state[ 1 ] == STATE_NOT ){
                        m_state[ 1 ] = m_state[ 0 ] ;
                        m_state[ 0 ] = STATE_TO_DELETE ;
                    }
                    
                    if( m_expire[ 1 ] == Long.MAX_VALUE ){
                        m_expire[ 1 ] = m_expire[ 0 ] ;
                        m_expire[ 0 ] = Long.MAX_VALUE ;
                    }
                    
                    if( m_createDate[ 1 ] == -1L ){
                        m_createDate[ 1 ] = m_createDate[ 0 ] ;
                        m_createDate[ 0 ] = -1L ;
                    }
                    
                    m_qarray.m_deleteSize ++ ;
                    this.commitOff() ;
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    
    /**
     * このオブジェクトを配列条件から外す.
     * <BR><BR>
     * このオブジェクトを配列条件から外します.
     */
    protected void cut()
    {
        try{
            synchronized( m_sync.get() ){
                
                if( m_before != null || m_next != null ){
                    if( m_before != null ){
                        if( m_before == m_qarray.m_table ){
                            if( m_next != null ){
                                m_qarray.m_table.m_next = m_next ;
                                m_next.m_before = m_qarray.m_table ;
                                m_qarray.m_size -- ;
                            }
                            else{
                                m_qarray.m_table.m_before = null ;
                                m_qarray.m_table.m_next = null ;
                                m_qarray.m_last = null ;
                                m_qarray.m_size = 0 ;
                            }
                        }
                        else{
                            if( m_next == null ){
                                m_before.m_next = null ;
                                m_qarray.m_last = m_before ;
                                m_qarray.m_size -- ;
                            }
                            else{
                                m_before.m_next = m_next ;
                                m_next.m_before = m_before ;
                                m_qarray.m_size -- ;
                            }
                        }
                    }
                }
                
                this.decDeleteSize() ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * オプション情報を設定.
     * <BR><BR>
     * オプション情報を設定します.
     * <BR>
     * @param option 対象のオプション情報を設定します.
     */
    public void setOption( Object option ) {
        
        try {
            synchronized( m_sync.get() ) {
                m_option = option ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * オプション情報を取得.
     * <BR><BR>
     * オプション情報を取得します.
     * <BR>
     * @return Object オプション情報が返されます.
     */
    public Object getOption() {
        
        Object ret = null ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = m_option ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のキー情報を設定.
     * <BR><BR>
     * 対象のキー情報を設定します.
     * <BR>
     * @param key 対象のキー情報を設定します.
     */
    public void setKey( String key )
    {
        try{
            synchronized( m_sync.get() ){
                
                if(
                    m_state[ 0 ] != STATE_TO_DELETE &&
                    m_state[ 0 ] != STATE_TO_DELETE_OK
                )
                {
                    if(
                        this.getAutoCommit() == false &&
                        ( m_key[ 1 ] == null || m_key[ 1 ].length() <= 0 ) &&
                        m_key[ 0 ] != null
                    )
                    {
                        m_key[ 1 ] = m_key[ 0 ] ;
                        m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    }
                    
                    m_key[ 0 ] = ( key == null || key.length() <= 0 ) ? "" : key ;
                    m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    m_createDate[ 0 ] = System.currentTimeMillis() ;
                    this.commitOff() ;
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 設定されているキー情報を取得.
     * <BR><BR>
     * 設定されているキー情報を取得します.
     * <BR>
     * @return String 設定されているキー情報が返されます.
     */
    public String getKey()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_key[ 0 ] == null || m_key[ 0 ].length() <= 0 ) ?
                    "" : m_key[ 0 ] ;
            }
        }catch( Exception e ){
            ret = "" ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のプロセスID情報を設定.
     * <BR><BR>
     * 対象のプロセスID情報を設定します.
     * <BR>
     * @param key 対象のプロセスID情報を設定します.
     */
    public void setProcessId( String processId )
    {
        try{
            synchronized( m_sync.get() ){
                
                if(
                    m_state[ 0 ] != STATE_TO_DELETE &&
                    m_state[ 0 ] != STATE_TO_DELETE_OK
                )
                {
                    if(
                        this.getAutoCommit() == false &&
                        ( m_processId[ 1 ] == null || m_processId[ 1 ].length() <= 0 ) &&
                        m_processId[ 0 ] != null
                    )
                    {
                        m_processId[ 1 ] = m_processId[ 0 ] ;
                        m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    }
                    
                    m_processId[ 0 ] = ( processId == null || processId.length() <= 0 ) ?
                        "" : processId ;
                    m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    m_createDate[ 0 ] = System.currentTimeMillis() ;
                    this.commitOff() ;
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 設定されているプロセスID情報を取得.
     * <BR><BR>
     * 設定されているプロセスID情報を取得します.
     * <BR>
     * @return String 設定されているプロセスID情報が返されます.
     */
    public String getProcessId()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( m_processId[ 0 ] == null || m_processId[ 0 ].length() <= 0 ) ?
                    "" : m_processId[ 0 ] ;
            }
        }catch( Exception e ){
            ret = "" ;
        }
        
        return ret ;
    }
    
    /**
     * オブジェクトを設定.
     * <BR><BR>
     * 対象のオブジェクト情報を設定します.
     * <BR>
     * @param value 設定対象のオブジェクトを設定します.
     */
    public void setValue( Object value )
    {
        try{
            synchronized( m_sync.get() ){
                
                if(
                    m_state[ 0 ] != STATE_TO_DELETE &&
                    m_state[ 0 ] != STATE_TO_DELETE_OK
                )
                {
                    if(
                        this.getAutoCommit() == false &&
                        m_value[ 1 ] == null &&
                        m_value[ 0 ] != null
                    )
                    {
                        m_value[ 1 ] = m_value[ 0 ] ;
                        m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    }
                    
                    m_value[ 0 ] = value ;
                    m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    m_createDate[ 0 ] = System.currentTimeMillis() ;
                    this.commitOff() ;
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 格納されているオブジェクト情報を取得.
     * <BR><BR>
     * 格納されているオブジェクト情報を取得します.
     * <BR>
     * @return Object 格納されているオブジェクト情報が返されます.
     */
    public Object getValue()
    {
        Object ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_value[ 0 ] ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * Expireを設定.
     * <BR><BR>
     * 対象のExpire情報を設定します.
     * <BR>
     * @param expire 設定対象のExpireを設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     */
    public void setExpire( long expire )
    {
        
        expire = ( expire <= 0 ) ? -1 : expire ;
        
        try{
            synchronized( m_sync.get() ){
                
                if(
                    m_state[ 0 ] != STATE_TO_DELETE &&
                    m_state[ 0 ] != STATE_TO_DELETE_OK
                )
                {
                    if(
                        this.getAutoCommit() == false &&
                        m_expire[ 1 ] == Long.MAX_VALUE &&
                        m_expire[ 0 ] != Long.MAX_VALUE
                    )
                    {
                        m_expire[ 1 ] = m_expire[ 0 ] ;
                        m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    }
                    
                    m_expire[ 0 ] = expire ;
                    m_createDate[ 1 ] = m_createDate[ 0 ] ;
                    m_createDate[ 0 ] = System.currentTimeMillis() ;
                    this.commitOff() ;
                }
                
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 格納されているExpire情報を取得.
     * <BR><BR>
     * 格納されているExpire情報を取得します.
     * <BR>
     * @return long 格納されているExpire情報が返されます.<BR>
     *              [Long.MAX_VALUE]が返された場合、情報は設定されていません.
     */
    public long getExpire()
    {
        long ret = Long.MAX_VALUE ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_expire[ 0 ] ;
            }
        }catch( Exception e ){
            ret = Long.MAX_VALUE ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているID情報を取得.
     * <BR><BR>
     * 設定されているID情報が返されます.
     * <BR>
     * @return long 設定されているID情報が返されます.
     */
    public long getID()
    {
        long ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_id ;
            }
        }catch( Exception e ){
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されているシーケンスIDを取得.
     * <BR><BR>
     * 設定されているシーケンスIDが返されます.
     * <BR>
     * @return int 設定されているシーケンスIDが返されます.
     */
    public int getSequenceID()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = ( int )( ( m_id & 0x7fffffff00000000L ) >> 32L ) ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 送信IDをLongで設定.
     * <BR><BR>
     * 対象の送信IDをLongで設定します.
     * <BR>
     * @param id 対象の送信ID(Long)を設定します.
     */
    public void setSendIdToLong( long id ) {
        try {
            synchronized( m_sync.get() ) {
                m_sendID = id ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * 送信IDを設定.
     * <BR><BR>
     * 対象の送信IDを設定します.
     * <BR>
     * @param id 対象の送信IDを設定します.
     * @param nextId 対象の次送信IDを設定します.
     */
    public void setSendId( int id,int nextId ) {
        try {
            synchronized( m_sync.get() ) {
                m_sendID = ( long )(
                    ( ( long )id & 0x000000007fffffffL ) |
                    ( ( ( long )nextId & 0x000000007fffffffL ) << 32L ) ) ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * 送信IDを取得.
     * <BR><BR>
     * 送信IDを取得します.
     * <BR>
     * @return long 送信IDが返されます.
     */
    public long getSendId() {
        long ret = 0L ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = m_sendID ;
            }
        } catch( Exception e ) {
            ret = 0L ;
        }
        
        return ret ;
    }
    
    /**
     * 現在の送信IDを取得.
     * <BR><BR>
     * 現在の送信IDを取得します.
     * <BR>
     * @return int 現在の送信IDが返されます.
     */
    public int getNowId() {
        
        int ret = -1 ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = ( int )
                    ( m_sendID & 0x000000007fffffffL ) ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 次送信IDを取得.
     * <BR><BR>
     * 次送信IDを取得します.
     * <BR>
     * @return int 次送信IDが返されます.
     */
    public int getNextSendId() {
        
        int ret = -1 ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = ( int )(
                    ( m_sendID & 0x7fffffff00000000L ) >> 32L ) ;
            }
        } catch( Exception e ) {
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 設定されている日付情報を取得.
     * <BR><BR>
     * 設定されている日付情報を取得します.
     * <BR>
     * @return Date 設定されている日付情報が返されます.
     */
    public Date getDate()
    {
        Date ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = UtilCom.getIntByDate(
                    ( int )( m_id & 0x00000000ffffffffL )
                ) ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 生成時間[yyyymmdd hhmmss ms]を取得.
     * <BR><BR>
     * このデータが生成された生成時間[ミリ秒]を取得します.
     * <BR>
     * @return long 生成された生成時間が返されます.
     */
    public long getTime()
    {
        long ret = -1L ;
        
        try{
            synchronized( m_sync.get() ) {
                ret = m_createDate[ 0 ] ;
            }
        }catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
    }
    
    /**
     * 生成時間[yyyymmdd hhmmss ms]の時間を更新.
     * <BR><BR>
     * 管理されている生成時間を更新します.
     */
    public void resetTime() {
        try {
            synchronized( m_sync.get() ) {
                m_createDate[ 0 ] = System.currentTimeMillis() ;
            }
        } catch( Exception e ) {
        }
    }
    
    /**
     * プライオリティを再設定.
     * <BR><BR>
     * プライオリティを再設定します.<BR>
     * また再設定した場合、再配置は行われません.
     * <BR>
     * @param priority 対象のプライオリティ値を設定します.
     */
    public void resetPriority( int priority ) {
        
        priority = ( priority <= MIN_PRIORITY ) ? MIN_PRIORITY : priority ;
        priority = ( priority >= MAX_BASE_PRIORITY ) ? MAX_BASE_PRIORITY : priority ;
        
        try {
            
            synchronized( m_sync.get() ) {
                m_priority = priority ;
            }
            
        } catch( Exception e ) {
            
        }
    }
    
    /**
     * 設定されている優先順位を取得.
     * <BR><BR>
     * 設定されている優先順位を取得します.
     * <BR>
     * @return int 設定されている優先順位が返されます.
     */
    public int getPriority()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_priority ;
            }
        }catch( Exception e ){
            ret = -1 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のステータス情報を取得.
     * <BR><BR>
     * 現在のステータス情報を取得します.
     * <BR>
     * @return int 現在のステータス情報が返されます.
     */
    public int getState()
    {
        int ret ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = m_state[ 0 ] ;
            }
        } catch( Exception e ) {
            ret = STATE_NOT ;
        }
        
        return ret ;
    }
    
    /**
     * 文字列に変換.
     * <BR><BR>
     * オブジェクトを文字列に変換します.
     * <BR>
     * @return String 文字列が返されます.
     */
    public String toString()
    {
        StringBuffer buf = null ;
        String ret = null ;
        
        buf = new StringBuffer() ;
        
        try{
            synchronized( m_sync.get() ){
                
                buf.append( "value:[" ) ;
                if( m_value[ 0 ] != null ){
                    buf.append( m_value[ 0 ] ) ;
                }
                else{
                    buf.append( "null" ) ;
                }
                buf.append( "] key:[" ) ;
                if( m_key[ 0 ] != null || m_key[ 0 ].length() > 0 ){
                    buf.append( m_key[ 0 ] ) ;
                }
                else{
                    buf.append( "null" ) ;
                }
                buf.append( "] processId:[" ) ;
                if( m_processId[ 0 ] != null || m_processId[ 0 ].length() > 0 ){
                    buf.append( m_processId[ 0 ] ) ;
                }
                else{
                    buf.append( "null" ) ;
                }
                buf.append( "] expire:[" ) ;
                if( m_expire[ 0 ] < Long.MAX_VALUE ){
                    buf.append( m_expire[ 0 ] ) ;
                }
                else{
                    buf.append( "null" ) ;
                }
                buf.append( "] id:" ) ;
                buf.append( m_id ) ;
                
                buf.append( " priority:" ) ;
                buf.append( m_priority ) ;
                
                buf.append( " state:" ) ;
                buf.append( m_state[ 0 ] ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
    /**
     * オブジェクト有効チェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean オブジェクト有効チェック内容が返されます.<BR>
     *                 [true]が返された場合有効です.<BR>
     *                 [false]が返された場合無効です.
     */
    public boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 確定状態を取得.
     * <BR><BR>
     * 現在の確定状態を取得します.
     * <BR>
     * @return boolean 現在の確定状態が返されます.
     */
    protected boolean isCommit()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_commitFlag ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * オートコミット条件を取得.
     */
    private final boolean getAutoCommit()
    {
        return m_qarray.m_autoCommitFlag ;
    }
    
    /**
     * コミットフラグをOFFに設定.
     */
    private final void commitOff()
    {
        if( m_qarray.m_autoCommitFlag == false ){
            m_commitFlag = false ;
        }
        else{
            m_commitFlag = true ;
        }
    }
    
    /**
     * カウントを１デクリメント.
     */
    private final void decDeleteSize()
    {
        if( m_state[ 0 ] == STATE_TO_DELETE ){
            m_state[ 0 ] = STATE_TO_DELETE_OK ;
            m_qarray.m_deleteSize = ( m_qarray.m_deleteSize <= 0 ) ?
                0 : m_qarray.m_deleteSize - 1 ;
        }
    }
    
}

