/*
 * @(#)QTable.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.main.queue.base.core ;

import java.util.Date;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.array.ObjectArray;

/**
 * キュー管理テーブル.
 * <BR><BR>
 * キュー情報を管理するテーブルオブジェクトです.
 *  
 * @version 2006/01/01
 * @author  masahito suzuki
 * @since   MaachangQ 1.00
 */
public class QTable
{
    
    /**
     * データ管理オブジェクト.
     */
    private QArray m_qarray = null ;
    
    /**
     * キーワード管理オブジェクト.
     */
    private final CharTable m_table = new CharTable() ;
    
    /**
     * データ変更テンポラリオブジェクト.
     */
    private final ObjectArray m_tmp = new ObjectArray() ;
    
    /**
     * キュ送信ID管理.
     */
    private QSendSeq m_sendId = null ;
    
    /**
     * コミット時に有効データが存在する時のフラグ.
     */
    private volatile boolean m_useCommitDataFlag = false ;
    
    
    
    /**
     * 同期オブジェクト.
     */
    protected final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public QTable()
    {
        
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * オブジェクトを生成します.
     */
    public void create()
    {
        this.create( false ) ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * オブジェクトを生成します.
     * <BR>
     * @param autoCommit オートコミットを設定します.<BR>
     *                   [true]を設定した場合、オートコミットはONになります.<BR>
     *                   [false]を設定した場合、オートコミットはOFFになります.
     */
    public void create( boolean autoCommit )
    {
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                m_sendId = new QSendSeq() ;
                m_qarray = new QArray() ;
                m_qarray.create( m_sync,autoCommit ) ;
            }
        }catch( Exception e ){
            this.clear() ;
        }
    }
    
    /**
     * オブジェクトをクリア.
     * <BR><BR>
     * オブジェクトをクリアします.
     */
    public void clear()
    {
        try{
            synchronized( m_sync.get() ){
                m_tmp.clear() ;
                m_table.clear() ;
                if( m_qarray != null ) {
                    m_qarray.clear() ;
                }
                m_qarray = null ;
                m_useCommitDataFlag = false ;
                m_sendId = null ;
            }
        }catch( Exception e ){
        }
        
        m_qarray = null ;
        m_useCommitDataFlag = false ;
        m_sendId = null ;
        m_sync.clear() ;
        
    }
    
    /**
     * 確定処理を実行.
     * <BR><BR>
     * 確定処理を実行します.
     */
    public void commit()
    {
        int i ;
        int len ;
        boolean useCommitData = false ;
        
        String key = null ;
        QArrayChild ch = null ;
        ObjectArray tmp = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( this.getAutoCommitFlag() == true ){
                    return ;
                }
                
                tmp = m_tmp ;
                len = tmp.size() ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    ch = ( QArrayChild )tmp.get( i ) ;
                    
                    if( ch != null && ch.isUse() == true ){
                    //if( ch != null ) {
                        
                        key = this.getQArrayChildByKey( ch ) ;
                        
                        if( ch.commit() == true ){
                            
                            if( key != null && key.length() > 0 ) {
                                this.removeTable( key,ch ) ;
                            }
                            ch.clear() ;
                            
                        }
                        else {
                            useCommitData = true ;
                        }
                        
                        ch = null ;
                    }
                    
                }
                
                tmp.clear() ;
                m_useCommitDataFlag = ( useCommitData == true ) ?
                    true : m_useCommitDataFlag ;
                
            }
            
        }catch( Exception e ){
        }finally{
            key = null ;
            ch = null ;
            tmp = null ;
        }
    }
    
    /**
     * １つのQArrayChildをコミット.
     * <BR><BR>
     * １つのQArrayChildをコミットします.
     * <BR>
     * @param ch コミット対象のQArrayChildを設定します.
     */
    public void commit( QArrayChild ch ) {
        
        int i ;
        int len ;
        boolean useCommitData = false ;
        
        String key = null ;
        ObjectArray tmp = null ;
        
        if( ch == null ) {
            return ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( this.getAutoCommitFlag() == true ){
                    return ;
                }
                
                tmp = m_tmp ;
                len = tmp.size() ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    if( tmp.get( i ) == ch ) {
                        
                        tmp.remove( i ) ;
                        
                        key = this.getQArrayChildByKey( ch ) ;
                        
                        if( ch.commit() == true ){
                            
                            if( key != null && key.length() > 0 ) {
                                this.removeTable( key,ch ) ;
                            }
                            ch.clear() ;
                            
                        }
                        else {
                            useCommitData = true ;
                        }
                        
                        break ;
                        
                    }
                    
                }
                
                m_useCommitDataFlag = ( useCommitData == true ) ?
                    true : m_useCommitDataFlag ;
                
            }
            
        }catch( Exception e ){
        }finally{
            key = null ;
            tmp = null ;
        }
        
    }
    
    /**
     * 以前の内容にロールバック.
     * <BR><BR>
     * 以前の内容にロールバックします.
     */
    public void rollback()
    {
        int i ;
        int len ;
        
        String key = null ;
        QArrayChild ch = null ;
        ObjectArray tmp = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( this.getAutoCommitFlag() == true ){
                    return ;
                }
                
                tmp = m_tmp ;
                len = tmp.size() ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    ch = ( QArrayChild )tmp.get( i ) ;
                    
                    if( ch != null && ch.isUse() == true ){
                    //if( ch != null ) {
                        
                        key = this.getQArrayChildByKey( ch ) ;
                        
                        if( ch.rollback() == true ){
                            
                            if( key != null && key.length() > 0 ) {
                                this.removeTable( key,ch ) ;
                            }
                            ch.clear() ;
                            
                        }
                        ch = null ;
                    }
                    
                }
                
                tmp.clear() ;
                
            }
            
        }catch( Exception e ){
        }finally{
            key = null ;
            ch = null ;
            tmp = null ;
        }
    }
    
    /**
     * １つのQArrayChildをロールバック.
     * <BR><BR>
     * １つのQArrayChildをロールバックします.
     * <BR>
     * @param ch ロールバック対象のQArrayChildを設定します.
     */
    public void rollback( QArrayChild ch ) {
        
        int i ;
        int len ;
        
        String key = null ;
        ObjectArray tmp = null ;
        
        if( ch == null ) {
            return ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( this.getAutoCommitFlag() == true ){
                    return ;
                }
                
                tmp = m_tmp ;
                len = tmp.size() ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    if( tmp.get( i ) == ch ) {
                        
                        tmp.remove( i ) ;
                        
                        if( ch.rollback() == true ){
                            
                            if(
                                ( key = this.getQArrayChildByKey( ch ) ) != null &&
                                key.length() > 0
                            )
                            {
                                this.removeTable( key,ch ) ;
                            }
                            ch.clear() ;
                            
                        }
                        
                        break ;
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
        }finally{
            key = null ;
            tmp = null ;
        }
        
    }
    
    /**
     * 情報を追加.
     * <BR><BR>
     * 対象の情報を追加します.
     * <BR>
     * @param name 対象のキーワードを設定します.
     * @param value 対象の要素を設定します.
     * @param processId 対象のプロセスIDを設定します.
     * @param priority 対象の優先順位を設定します.<BR>
     *                 設定できる最小値は[0]です.<BR>
     *                 設定できる最大値は[10000]です.
     * @param expire 対象のExpire値を設定します.<BR>
     *               [0]以下を設定した場合、未設定となります.
     * @return QArrayChild 生成されたQArrayChildが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public QArrayChild add( String name,Object value,String processId,
        int priority,long expire )
        throws InputException,AccessException
    {
        QArrayChild ret = null ;
        
        if( value == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                ret = m_qarray.add( value,priority,expire ) ;
                
                if( name != null && name.length() > 0 ){
                    ret.setKey( name ) ;
                    this.putTable( name,ret ) ;
                }
                if( processId != null && processId.length() > 0 ){
                    ret.setProcessId( processId ) ;
                }
                
                this.putTmp( ret ) ;
                
            }
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除して取得します.
     * <BR>
     * @param no 指定項番を指定して取得します.
     * @return Object 対象のオブジェクトが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public Object remove( int no )
    {
        QArrayChild ch = null ;
        String key = null ;
        Object ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                ch = m_qarray.get( no ) ;
                
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    
                    ret = ch.getValue() ;
                    
                    if( this.getAutoCommitFlag() == true ){
                        
                        if( ( key = this.getQArrayChildByKey( ch ) ) != null && key.length() > 0 ){
                            this.removeTable( key,ch ) ;
                        }
                        
                        ch.clear() ;
                        
                    }
                    else{
                        
                        if(
                            ch.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                            ch.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                        ) {
                            
                            ch.delete() ;
                            this.putTmp( ch ) ;
                            
                        }
                        
                    }
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
            key = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 対象の情報を削除して取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 対象の要素項番を設定します.
     * @return Object 対象のオブジェクトが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public Object remove( String key,int no )
    {
        int i ;
        int len ;
        int cnt ;
        
        ObjectArray ary = null ;
        QArrayChild ch = null ;
        Object ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        if( key == null || key.length() <= 0 ){
            return this.remove( no ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                ary = ( ObjectArray )m_table.get( key ) ;
                
                if( ary != null && ( len = ary.size() ) > 0 ){
                    
                    if( this.getAutoCommitFlag() == true ){
                        if( ( ch = ( QArrayChild )ary.remove( no ) ) != null ){
                            ret = ch.getValue() ;
                            ch.clear() ;
                        }
                    }
                    else{
                        for( i = 0,cnt = 0 ; i < len ; i ++ ){
                            if(
                                ( ch = ( QArrayChild )ary.get( i ) ) != null &&
                                (
                                    ch.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                                    ch.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                                )
                            )
                            {
                                if( cnt >= no ){
                                    ret = ch.getValue() ;
                                    ch.delete() ;
                                    this.putTmp( ch ) ;
                                    break ;
                                }
                                cnt ++ ;
                            }
                        }
                    }
                    
                }
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ary = null ;
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 情報削除.
     * <BR><BR>
     * 指定したQArrayChildを指定して情報を削除します.
     * <BR>
     * @param ch 削除対象のQArrayChildを設定します.
     * @return Object 対象のオブジェクトが返されます.<BR>
     *                情報が存在しない場合[null]が返されます.
     */
    public Object remove( QArrayChild ch ) {
        
        String key = null ;
        Object ret = null ;
        
        if( ch == null ) {
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( this.getAutoCommitFlag() == true ){
                    
                    if(
                        ( key = this.getQArrayChildByKey( ch ) ) != null &&
                        key.length() > 0
                    )
                    {
                        this.removeTable( key,ch ) ;
                    }
                    
                    ret = ch.getValue() ;
                    ch.clear() ;
                    
                }
                else {
                    
                    if( 
                        ch.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                        ch.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                    ) {
                        
                        ret = ch.getValue() ;
                        ch.delete() ;
                        this.putTmp( ch ) ;
                        
                    }
                    
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            key = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象要素を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return Object 対象の要素が返されます.
     */
    public Object get( int no )
    {
        QArrayChild ch = null ;
        Object ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getValue() ;
                }
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象要素を取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return Object 対象の要素が返されます.
     */
    public Object get( String key,int no )
    {
        QArrayChild ch = null ;
        Object ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( key,no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getValue() ;
                }
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のキー名を取得.
     * <BR><BR>
     * 対象要素のキー名を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 対象のキー名が返されます.
     */
    public String getKey( int no )
    {
        QArrayChild ch = null ;
        String ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getKey() ;
                }
            }
        }catch( Exception e ){
            ret = "" ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素の優先順位を取得.
     * <BR><BR>
     * 対象要素の優先順位を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return int 対象の優先順位が返されます.
     */
    public int getPriority( int no )
    {
        int ret ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return 0 ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getPriority() ;
                }
                else{
                    ret = 0 ;
                }
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素の優先順位を取得.
     * <BR><BR>
     * 対象要素の優先順位を取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return int 対象の優先順位が返されます.
     */
    public int getPriority( String key,int no )
    {
        int ret ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return 0 ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( key,no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getPriority() ;
                }
                else{
                    ret = 0 ;
                }
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素の生成時間を取得.
     * <BR><BR>
     * 対象要素の生成時間を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return Date 対象の生成時間が返されます.
     */
    public Date getDate( int no )
    {
        QArrayChild ch = null ;
        Date ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getDate() ;
                }
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素の生成時間を取得.
     * <BR><BR>
     * 対象要素の生成時間を取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return Date 対象の生成時間が返されます.
     */
    public Date getDate( String key,int no )
    {
        QArrayChild ch = null ;
        Date ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( key,no ) ;
                if( ch != null && ch.isUse() == true ){
                //if( ch != null ) {
                    ret = ch.getDate() ;
                }
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のプロセスIDを取得.
     * <BR><BR>
     * 対象要素のプロセスIDを取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return String 対象のプロセスIDが返されます.
     */
    public String getProcessId( int no )
    {
        String ret = null ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                ret = ch.getProcessId() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のプロセスIDを取得.
     * <BR><BR>
     * 対象要素のプロセスIDを取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return String 対象のプロセスIDが返されます.
     */
    public String getProcessId( String key,int no )
    {
        String ret = null ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( key,no ) ;
                ret = ch.getProcessId() ;
            }
        }catch( Exception e ){
            ret = null ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のExpire+生成時間を取得.
     * <BR><BR>
     * 対象要素のExpire+生成時間を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return long 対象のExpire+生成時間が返されます.<BR>
     *              [-1L]が返された場合、Expireは無効です.
     */
    public long getExpire( int no )
    {
        long ret = -1L ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return -1L ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( no ) ;
                ret = this.getExpire( ch ) ;
            }
        }catch( Exception e ){
            ret = -1L ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のExpire+生成時間を取得.
     * <BR><BR>
     * 対象要素のExpire+生成時間を取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return long 対象のExpire+生成時間が返されます.<BR>
     *              [-1L]が返された場合、Expireは無効です.
     */
    public long getExpire( String key,int no )
    {
        long ret = -1L ;
        QArrayChild ch = null ;
        
        if( no < 0 ){
            return -1L ;
        }
        
        try{
            synchronized( m_sync.get() ){
                ch = this.getQArrayChild( key,no ) ;
                ret = this.getExpire( ch ) ;
            }
        }catch( Exception e ){
            ret = -1L ;
        }finally{
            ch = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素のExpire+生成時間を取得.
     * <BR><BR>
     * 対象要素のExpire+生成時間を取得します.
     * <BR>
     * @param ch 対象のQArrayChildを取得します.
     * @return long 対象のExpire+生成時間が返されます.<BR>
     *              [Long.MAX_VALUE ;]が返された場合、Expireは無効です.
     */
    public long getExpire( QArrayChild ch )
    {
        long ret = Long.MAX_VALUE ;
        
        if( ch == null ){
            return Long.MAX_VALUE ;
        }
        
        try{
            synchronized( m_sync.get() ){
                if(
                    ch.getExpire() != -1L &&
                    ch.getExpire() != Long.MAX_VALUE
                )
                {
                    ret = ch.getTime() + ch.getExpire() ;
                }
                else {
                    ret = Long.MAX_VALUE ;
                }
            }
        }catch( Exception e ){
            ret = Long.MAX_VALUE ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象となる要素情報を取得します.
     * <BR>
     * @param no 取得対象の項番を設定します.
     * @return QArrayChild 対象のQArrayChild要素が返されます.
     */
    public QArrayChild getQArrayChild( int no )
    {
        QArrayChild ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        try{
            
            ret = m_qarray.get( no ) ;
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素を取得.
     * <BR><BR>
     * 対象となる要素情報を取得します.
     * <BR>
     * @param key 対象のキー条件を設定します.
     * @param no 取得対象の項番を設定します.
     * @return QArrayChild 対象のQArrayChild要素が返されます.
     */
    public QArrayChild getQArrayChild( String key,int no )
    {
        int i ;
        int len ;
        int cnt ;
        
        ObjectArray ary = null ;
        QArrayChild ret = null ;
        
        if( no < 0 ){
            return null ;
        }
        
        if( key == null || key.length() <= 0 ){
            return this.getQArrayChild( no ) ;
        }
        
        try{
            
            ary = ( ObjectArray )m_table.get( key ) ;
            
            if( ary != null && ( len = ary.size() ) > 0 ){
                
                if( this.getAutoCommitFlag() == true ){
                    if( ( ret = ( QArrayChild )ary.get( no ) ) == null ){
                        ret = null ;
                    }
                }
                else{
                    
                    for( i = 0,cnt = 0 ; i < len ; i ++ ){
                        if(
                            ( ret = ( QArrayChild )ary.get( i ) ) != null &&
                            (
                                ret.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                                ret.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                            )
                        )
                        {
                            if( cnt >= no ){
                                break ;
                            }
                            cnt ++ ;
                        }
                        
                        ret = null ;
                    }
                }
                
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            ary = null ;
        }
        
        return ret ;
    }
    
    /**
     * 要素格納数を取得.
     * <BR><BR>
     * 格納されている要素数を取得.
     * <BR>
     * @return int 格納されている要素数が返されます.
     */
    public int size()
    {
        int ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_qarray.size() ;
            }
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * キー名群を取得.
     * <BR><BR>
     * 対象の検索ワードキー名群を取得します.
     * <BR>
     * @return String[] 対象のキー名群が返されます.
     */
    public String[] getKeys()
    {
        int i,j ;
        int len ;
        int sz ;
        
        String[] tmp = null ;
        String[] ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                
                if(
                    ( tmp = m_table.getNames() ) != null &&
                    ( len = tmp.length ) > 0
                )
                {
                    
                    if( this.getAutoCommitFlag() == true ){
                        
                        ret = tmp ;
                        
                    }
                    else{
                        
                        for( i = 0,sz = 0 ; i < len ; i ++ ){
                            if( this.getElements( tmp[ i ] ) <= 0 ){
                                tmp[ i ] = null ;
                            }
                            else{
                                sz ++ ;
                            }
                        }
                        
                        if( sz > 0 ){
                            ret = new String[ sz ] ;
                            for( i = 0,j = 0 ; i < len ; i ++ ){
                                if( tmp[ i ] != null ){
                                    ret[ j ] = tmp [ i ] ;
                                    j ++ ;
                                }
                            }
                        }
                        
                    }
                    
                }
                
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * キー名に対する要素数を取得.
     * <BR><BR>
     * 対象のキー名に対する要素数を取得します.
     * <BR>
     * @param key 対象のキー名を設定します.
     * @return int キー名に対する要素数が返されます.
     */
    public int getElements( String key )
    {
        int i ;
        int len ;
        int ret ;
        
        ObjectArray ary = null ;
        QArrayChild ch = null ;
        
        if( key == null || key.length() <= 0 ){
            return 0 ;
        }
        
        try{
            synchronized( m_sync.get() ){
                
                if( ( ary = ( ObjectArray )m_table.get( key ) ) == null ){
                    ret = 0 ;
                }
                else if( ( len = ary.size() ) <= 0 ){
                    ret = 0 ;
                }
                else if( this.getAutoCommitFlag() == true ){
                    ret = len ;
                }
                else{
                    for( i = 0,ret = 0 ; i < len ; i ++ ){
                        if(
                            ( ch = ( QArrayChild )ary.get( i ) ) != null &&
                            (
                                ch.m_state[ 0 ] == QArrayChild.STATE_CREATE ||
                                ch.m_state[ 0 ] == QArrayChild.STATE_TO_COMMIT
                            )
                        )
                        {
                            ret ++ ;
                        }
                    }
                }
            }
        }catch( Exception e ){
            ret = 0 ;
        }finally{
            ary = null ;
            ch = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * コミット有効データフラグをリセット.
     * <BR><BR>
     * コミット有効データフラグをリセットします.
     */
    public void resetUseCommitDataFlag() {
        
        try {
            synchronized( m_sync.get() ) {
                m_useCommitDataFlag = false ;
            }
        } catch( Exception e ) {
        }
        
    }
    
    /**
     * コミット有効データフラグを取得.
     * <BR><BR>
     * コミット処理により、有効データが存在する参考値です.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効データが存在します.<BR>
     *                 [false]が返された場合、有効データは存在しません.
     */
    public boolean isUseCommitDataFlag() {
        
        boolean ret = false ;
        
        try {
            synchronized( m_sync.get() ) {
                if( this.getAutoCommitFlag() == false ) {
                    ret = m_useCommitDataFlag ;
                }
                else {
                    ret = true ;
                }
            }
        } catch( Exception e ) {
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューID情報を取得.
     * <BR><BR>
     * キュー内のID情報を取得します.
     * <BR>
     * @return long このキューが発行するID情報が返されます.
     */
    public long getQueueID() {
        
        long ret = -1L ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = m_qarray.getID() ;
            }
        } catch( Exception e ) {
            ret = -1L ;
        }
        
        return ret ;
        
    }
    
    /**
     * 送信ID管理オブジェクトを取得.
     * <BR><BR>
     * 送信ID管理オブジェクトを取得します.
     * <BR>
     * @reurn QSendSeq 送信ID管理オブジェクトが返されます.
     */
    public QSendSeq getSendSequence() {
        
        QSendSeq ret = null ;
        
        try {
            synchronized( m_sync.get() ) {
                ret = m_sendId ;
            }
        } catch( Exception e ) {
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * オートコミット情報を取得.
     * <BR><BR>
     * このオブジェクトがオートコミットであるか取得します.
     * <BR>
     * @return boolean オートコミットであるか取得します.<BR>
     *                 [true]が返された場合、オートコミットです.<BR>
     *                 [false]が返された場合、オートコミットではありません.
     */
    public boolean isAutoCommit()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = this.getAutoCommitFlag() ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * このオブジェクトが有効であるか取得.
     * <BR><BR>
     * このオブジェクトが有効であるか取得します.
     * <BR>
     * @return boolean このオブジェクトが有効であるか取得します.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = true ;
            }
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * 指定の要素を、テーブルから削除.
     */
    private final void removeTable( String key,QArrayChild ch )
    {
        int i ;
        int len ;
        
        ObjectArray ary = null ;
        QArrayChild tmp = null ;
        
        if( ( ary = ( ObjectArray )m_table.get( key ) ) != null ){
            
            len = ary.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( ( tmp = ( QArrayChild )ary.get( i ) ) != null ){
                    
                    if( ch == tmp ){
                        ary.remove( i ) ;
                        if( len == 1 ){
                            try{
                                m_table.remove( key ) ;
                            }catch( Exception e ){
                            }
                        }
                        break ;
                    }
                    
                }
            }
            
        }
        
    }
    
    /**
     * キーワード管理オブジェクトに情報を追加.
     */
    private final void putTable( String name,QArrayChild ch )
    {
        int i ;
        int len ;
        
        int pt ;
        QArrayChild tmp = null ;
        ObjectArray oa = null ;
        
        if( ch != null && ch.isUse() == true ){
        //if( ch != null ) {
            try{
                if( ( oa = ( ObjectArray )m_table.get( name ) ) == null ){
                    
                    oa = new ObjectArray() ;
                    m_table.add( name,oa ) ;
                    oa.add( ch ) ;
                    
                }
                else{
                    
                    pt = ch.m_priority ;
                    
                    if(
                        ( tmp = ( QArrayChild )oa.get( 0 ) ) != null &&
                        tmp.m_priority < pt
                    )
                    {
                        this.bitween( oa,ch,0 ) ;
                    }
                    else if(
                        ( tmp = ( QArrayChild )oa.get( oa.size() - 1 ) ) != null &&
                        tmp.m_priority >= pt
                    )
                    {
                        oa.add( ch ) ;
                    }
                    else{
                        
                        len = oa.size() - 1 ;
                        for( i = 1 ; i < len ; i ++ ){
                            
                            if(
                                ( tmp = ( QArrayChild )oa.get( i ) ) != null &&
                                tmp.m_priority > pt
                            )
                            {
                                this.bitween( oa,ch,i ) ;
                                break ;
                            }
                            
                        }
                        
                    }
                }
            }catch( Exception e ){
            }finally{
                tmp = null ;
                oa = null ;
            }
        }
    }
    /**
     * 確定・削除条件で利用するテンポラリ領域に条件を設定.
     */
    private final void putTmp( QArrayChild ch )
    {
        int i ;
        int len ;
        boolean flg ;
        
        ObjectArray tmp = null ;
        
        if( ch != null && ch.isUse() == true && this.getAutoCommitFlag() == false ){
        //if( ch != null  && this.getAutoCommitFlag() == false ) {
            try{
                
                tmp = m_tmp ;
                len = tmp.size() ;
                
                for( i = 0,flg = false ; i < len ; i ++ ) {
                    if( tmp.get( i ) == ch ) {
                        flg = true ;
                        break ;
                    }
                }
                if( flg == false ) {
                    m_tmp.add( ch ) ;
                }
                
            }catch( Exception e ){
            }finally{
                tmp = null ;
            }
        }
    }
    
    /**
     * オートコミットフラグを取得.
     */
    private final boolean getAutoCommitFlag()
    {
        return m_qarray.m_autoCommitFlag ;
    }
    
    /**
     * 指定位置に情報を追加.
     */
    private final void bitween( ObjectArray out,QArrayChild ch,int no )
        throws Exception
    {
        int len ;
        
        Object[] objs = null ;
        
        len = out.size() ;
        
        if( no == 0 ){
            
            out.add( new Object() ) ;
            objs = out.getObjects() ;
            System.arraycopy( objs,0,objs,1,len ) ;
            objs[ 0 ] = ch ;
            
        }
        else if( no >= len ){
            
            out.add( ch ) ;
            
        }
        else{
            
            out.add( new Object() ) ;
            objs = out.getObjects() ;
            System.arraycopy( objs,no,objs,no+1,len-no ) ;
            objs[ no ] = ch ;
            
        }
    }
    
    /**
     * 対象QArrayChildから、有効なKey名を取得.
     */
    private final String getQArrayChildByKey( QArrayChild ch ) {
        
        if( ch.m_key[ 0 ] != null && ch.m_key[ 0 ].length() > 0 ) {
            return ch.m_key[ 0 ] ;
        }
        else if( ch.m_key[ 1 ] != null && ch.m_key[ 1 ].length() > 0 ) {
            return ch.m_key[ 1 ] ;
        }
        
        return null ;
        
    }
}

