/*
 * @(#)ExecutionClientMq.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access ;

import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.MessageBean;
import org.maachang.queue.access.protocol.MessageProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;


/**
 * クライアント側キュー実行処理.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionClientMq {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * 対象キューマネージャ存在チェック.
     * <BR><BR>
     * 対象キューマネージャの存在チェックを行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void checkManager( ConnectObject connect,String manager )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createCheckManager( outId,manager ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],MessageProtocol.TYPE_CONNECT_MANAGER,telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 指定キュー存在チェック.
     * <BR><BR>
     * 対象キューの存在チェックを行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @param queue 対象のキューを設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @return boolean オートコミットモードが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean checkQueue( ConnectObject connect,String manager,String queue,int queueType )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createCheckQueue(
                outId,manager,queue,queueType ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                return ResultTelegram.checkResultByMessageBoolean(
                    outId[ 0 ],MessageProtocol.TYPE_CONNECT_QUEUE,telegram ) ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 対象データ送信処理.
     * <BR><BR>
     * 対象データを送信します.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @param queue 対象のキューを設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param key 対象のキーコードを設定します.
     * @param processId 対象のプロセスIDを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param expire 対象のExpire情報を設定します.
     * @param priority 対象のプライオリティ値を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void sendQueue( ConnectObject connect,String manager,String queue,
        int queueType,String key,String processId,byte[] message,int expire,int priority )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createSendMessage(
                outId,manager,queue,queueType,key,processId,
                message,expire,priority ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],MessageProtocol.TYPE_SEND,telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 電文受信処理.
     * <BR><BR>
     * 電文情報を受信します.
     * <BR>
     * @param result 受信結果を格納する格納用データを設定します.
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param key 対象のキー情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void receiveQueue( MqValue result,ConnectObject connect,String manager,
        String queue,String key )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createReceiveMessage(
                outId,manager,queue,key ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int resultType = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( resultType == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],MessageProtocol.TYPE_RECEIVE,telegram ) ;
                
            }
            // 受信電文がメッセージの場合.
            else if( resultType == MessageProtocol.CATEGORY_TYPE_MESSAGE ) {
                
                MessageBean out = new MessageBean() ;
                MessageTelegram.analysisDataTelegram( out,telegram ) ;
                
                if( out.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                result.setValue( out.getMessage() ) ;
                result.setExpire( out.getExpire() ) ;
                result.setPriority( out.getPriority() ) ;
                result.setKey( out.getKey() ) ;
                result.setProcessId( out.getProcessId() ) ;
                
                return ;
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 対象キューのトランザクション開始.
     * <BR><BR>
     * 対象キューのトランザクションを開始します.
     * <BR>
     * @param mode 対象の処理モードを設定します.<BR>
     *             [true]を設定した場合、コミットします.<BR>
     *             [false]を設定した場合、ロールバックします.
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @param queue 対象のキューを設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param timeout 対象のタイムアウト値を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void transactionQueue( ConnectObject connect,
        String manager,String queue,int queueType,long timeout )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createTransaction(
                outId,manager,queue,queueType ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,timeout ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],MessageProtocol.TYPE_TRANSACTION,telegram ) ;
                
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 対象キューをコミット or ロールバック.
     * <BR><BR>
     * 対象キューをコミット or ロールバックします.
     * <BR>
     * @param mode 対象の処理モードを設定します.<BR>
     *             [true]を設定した場合、コミットします.<BR>
     *             [false]を設定した場合、ロールバックします.
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param manager 対象のキューマネージャを設定します.
     * @param queue 対象のキューを設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void commitOrRollbackQueue( boolean mode,ConnectObject connect,
        String manager,String queue,int queueType )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = MessageTelegram.createCommitOrRollback(
                mode,outId,manager,queue,queueType ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                int execType = ( mode == true ) ?
                    MessageProtocol.TYPE_COMMIT :
                    MessageProtocol.TYPE_ROLLBACK ;
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult( outId[ 0 ],execType,telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
}

