/*
 * @(#)MaachangQAccessDriver.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access ;

import java.net.InetAddress;
import java.net.Socket;

import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.net.CreateSocket;


/**
 * MaachangQ接続.
 * <BR><BR>
 * MaachangQ接続オブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class MaachangQAccessDriver {
    
    private MaachangQAccessDriver() { }
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * キューマネージャに接続.
     * <BR><BR>
     * キューマネージャに接続します.
     * <BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを設定します.
     *             [false]を設定した場合、通常ソケットを設定します.
     * @param manager 接続先のキューマネージャ名を設定します.
     * @return MaachangQManager 接続されたキューマネージャオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final MaachangQManager connect( boolean mode,String manager )
        throws MaachangQException {
        return MaachangQAccessDriver.connect(
            mode,manager,null,MaachangQAccessDriver.getDefaultPort( mode ) ) ;
    }
    
    /**
     * キューマネージャに接続.
     * <BR><BR>
     * キューマネージャに接続します.
     * <BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを設定します.
     *             [false]を設定した場合、通常ソケットを設定します.
     * @param manager 接続先のキューマネージャ名を設定します.
     * @param port 接続先のポート番号を設定します.
     * @return MaachangQManager 接続されたキューマネージャオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final MaachangQManager connect( boolean mode,String manager,int port )
        throws MaachangQException {
        return MaachangQAccessDriver.connect( mode,manager,null,port ) ;
    }
    
    /**
     * キューマネージャに接続.
     * <BR><BR>
     * キューマネージャに接続します.
     * <BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを設定します.
     *             [false]を設定した場合、通常ソケットを設定します.
     * @param manager 接続先のキューマネージャ名を設定します.
     * @param address 接続先のアドレスを設定します.
     * @return MaachangQManager 接続されたキューマネージャオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final MaachangQManager connect( boolean mode,String manager,InetAddress address )
        throws MaachangQException {
        return MaachangQAccessDriver.connect(
            mode,manager,address,MaachangQAccessDriver.getDefaultPort( mode ) ) ;
    }
    
    /**
     * キューマネージャに接続.
     * <BR><BR>
     * キューマネージャに接続します.
     * <BR>
     * @param mode 対象のモードを設定します.<BR>
     *             [true]を設定した場合、SSLソケットを設定します.
     *             [false]を設定した場合、通常ソケットを設定します.
     * @param manager 接続先のキューマネージャ名を設定します.
     * @param address 接続先のアドレスを設定します.
     * @param port 接続先のポート番号を設定します.
     * @return MaachangQManager 接続されたキューマネージャオブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final MaachangQManager connect( boolean mode,String manager,InetAddress address,int port )
        throws MaachangQException {
        
        if( port < 0 || port > 65535 || manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            throw new MaachangQException( "引数は不正です" ) ;
        }
        
        try {
            synchronized( SYNC ) {
                
                // アドレスがnullの場合、ローカルホスト接続.
                if( address == null ) {
                    address = InetAddress.getByName( "127.0.0.1" ) ;
                }
                
                Socket socket = CreateSocket.clientSocket( mode,address,port ) ;
                ConnectObject connect = new ConnectObject( socket ) ;
                
                ExecutionClientMq.checkManager( connect,manager ) ;
                return new MaachangQManagerImple( connect,manager ) ;
                
            }
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * デフォルトポートを取得.
     */
    private static final int getDefaultPort( boolean mode ) {
        if( mode == true ) {
            return MaachangQAccessDefine.SSL_TELEGRAM_PORT ;
        }
        return MaachangQAccessDefine.TELEGRAM_PORT ;
    }
    
}

