/*
 * @(#)ExecutionQueueManager.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.info.CreateQueueManagerInfo;
import org.maachang.queue.access.admin.info.QueueManagerInfo;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.protocol.ResultProtocol;
import org.maachang.queue.access.protocol.ResultTelegram;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerBean;
import org.maachang.queue.access.protocol.admin.AdminQueueManagerProtocol;


/**
 * 管理キューマネージャ実行処理.
 *  
 * @version 2007/01/12
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class ExecutionQueueManager {
    
    /**
     * デフォルトタイムアウト.
     */
    public static final long DEFAULT_TIMEOUT = MaachangQAccessDefine.DEFAULT_TIMEOUT ;
    
    /**
     * キューマネージャ追加処理.
     * <BR><BR>
     * キューマネージャ追加処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param info 対象の追加キューマネージャ情報を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void addQueueManager(
        ConnectObject connect,int userId,boolean userOwner,CreateQueueManagerInfo info )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createAddQueueManager(
                outId,userOwner,userId,info ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_ADD_QMANAGER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キューマネージャ削除処理.
     * <BR><BR>
     * キューマネージャ削除処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void removeQueueManager(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createRemoveQueueManager(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_REMOVE_QMANAGER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キューマネージャ停止処理.
     * <BR><BR>
     * キューマネージャ停止処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void stopQueueManager(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createStopQueueManager(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_STOP_QMANAGER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キューマネージャ開始処理.
     * <BR><BR>
     * キューマネージャ開始処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final void startQueueManager(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createStartQueueManager(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_START_QMANAGER,
                    telegram ) ;
                return ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キューマネージャ情報取得処理.
     * <BR><BR>
     * キューマネージャ情報取得処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return QueueManagerInfo キューマネージャ構成情報が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final QueueManagerInfo getQueueManager(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createGetQueueManager(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_GET_QMANAGER,
                    telegram ) ;
                
            }
            // 受信電文が、キューマネージャ情報の場合.
            else if( result == AdminQueueManagerProtocol.CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ) {
                
                // キューマネージャ情報を解析.
                AdminQueueManagerBean bean = new AdminQueueManagerBean() ;
                QueueManagerTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                if( bean.getName() == null || bean.getName().length() <= 0 ) {
                    throw new MaachangQException(
                        "対象名(" + name + ")のキューマネージャ名は存在しません",
                        MaachangQErrorCode.ERROR_NOT_QUEUE_MANAGER ) ;
                }
                
                QueueManagerInfo ret = new QueueManagerInfo() ;
                ret.setQueueManager( bean.getName() ) ;
                ret.setUseCache( ( bean.getMaxSector() <= 0 ) ? false : true ) ;
                ret.setMaxCacheSize( bean.getMaxSector() ) ;
                ret.setUseCacheSize( bean.getUseSector() ) ;
                ret.setState( bean.getState() ) ;
                
                return ret ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 存在するキューマネージャ名群処理.
     * <BR><BR>
     * 存在するキューマネージャ名群処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return String[] キューマネージャ名群が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final String[] getNames(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createGetNamesQueueManager(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_GET_QMANAGER_NAMES,
                    telegram ) ;
                
            }
            // 受信電文が、キューマネージャ情報の場合.
            else if( result == AdminQueueManagerProtocol.CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ) {
                
                // キューマネージャ情報を解析.
                AdminQueueManagerBean bean = new AdminQueueManagerBean() ;
                QueueManagerTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getParams() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * 存在するキューマネージャ数処理.
     * <BR><BR>
     * 存在するキューマネージャ数処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @return int キューマネージャ数が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final int getSize(
        ConnectObject connect,int userId,boolean userOwner )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createGetSizeQueueManager(
                outId,userOwner,userId ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                // 処理戻り条件をチェック.
                ResultTelegram.checkResult(
                    outId[ 0 ],AdminQueueManagerProtocol.TYPE_GET_QMANAGER_SIZE,
                    telegram ) ;
                
            }
            // 受信電文が、キューマネージャ情報の場合.
            else if( result == AdminQueueManagerProtocol.CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ) {
                
                // キューマネージャ情報を解析.
                AdminQueueManagerBean bean = new AdminQueueManagerBean() ;
                QueueManagerTelegram.analysisTelegram( bean,telegram ) ;
                
                // IDが一致しない場合.
                if( bean.getId() != outId[ 0 ] ) {
                    throw new MaachangQException( "受信電文は不正です" ) ;
                }
                
                return bean.getSize() ;
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
    /**
     * キューマネージャ開始処理.
     * <BR><BR>
     * キューマネージャ開始処理を行います.
     * <BR>
     * @param connect 対象のコネクションオブジェクトを設定します.
     * @param userId 対象のユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @return boolean 存在結果が返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public static final boolean isQueueManager(
        ConnectObject connect,int userId,boolean userOwner,String name )
        throws MaachangQException {
        
        try {
            
            int[] outId = new int[ 1 ] ;
            byte[] telegram = QueueManagerTelegram.createUseQueueManager(
                outId,userOwner,userId,name ) ;
            
            connect.send( telegram ) ;
            telegram = ConnectObject.receive( connect,DEFAULT_TIMEOUT ) ;
            
            int result = CommonProtocol.protocolByCategory( telegram ) ;
            
            // 受信電文が、処理戻り値の場合.
            if( result == ResultProtocol.CATEGORY_TYPE_RESULT ) {
                
                try {
                    // 処理戻り条件をチェック.
                    ResultTelegram.checkResult(
                        outId[ 0 ],AdminQueueManagerProtocol.TYPE_IS_QMANAGER,
                        telegram ) ;
                    
                    return true ;
                } catch( Exception e ) {
                    return false ;
                }
                
            }
            
            throw new MaachangQException( "受信電文は不正です" ) ;
            
        } catch( MaachangQException fm ) {
            throw fm ;
        } catch( Exception e ) {
            throw new MaachangQException( e ) ;
        }
        
    }
    
}

