/*
 * @(#)MaachangQConnectAdminImple.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.admin ;

import java.net.InetAddress;
import java.util.ArrayList;

import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.net.ConnectObject;
import org.maachang.queue.access.protocol.login.LoginBean;


/**
 * MaachangQ管理者接続情報.
 * <BR><BR>
 * MaachangQ管理者接続情報を表すオブジェクト.
 *  
 * @version 2007/01/02
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
class MaachangQConnectAdminImple implements MaachangQConnectAdmin {
    
    /**
     * ログインユーザ名.
     */
    private String loginUserName = null ;
    
    /**
     * ログインユーザパスワード.
     */
    private String loginUserPasswd = null ;
    
    /**
     * ログインユーザID.
     */
    private int userId = -1 ;
    
    /**
     * ログインユーザ権限.
     */
    private boolean userOwner = false ;
    
    /**
     * コネクションオブジェクト.
     */
    private ConnectObject connect = null ;
    
    /**
     * コネクション中キュー管理オブジェクト.
     */
    private ArrayList queueList = null ;
    
    /**
     * ユーザ管理.
     */
    private AdminUserImple adminUser = null ;
    
    /**
     * マネージャ管理.
     */
    private AdminQueueManagerImple adminQueueManager = null ;
    
    /**
     * チャネル管理.
     */
    private AdminChannelImple adminChannel = null ;
    
    /**
     * コネクション管理.
     */
    private AdminConnectImple adminConnect = null ;
    
    /**
     * キャッシュ管理.
     */
    private AdminCacheImple adminCache = null ;
    
    
    /**
     * コンストラクタ.
     */
    private MaachangQConnectAdminImple() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象条件を設定してオブジェクトを生成します.
     * <BR>
     * @param connect 対象のコネクション名を設定します.
     * @param userId 対象のログインユーザIDを設定します.
     * @param userOwner 対象のユーザ権限を設定します.
     * @param loginUser 対象のログインユーザ名を設定します.
     * @param loginPasswd 対象のログインパスワードを設定します.
     */
    protected MaachangQConnectAdminImple( ConnectObject connect,int userId,boolean userOwner,
        String loginUser,String loginPasswd ) {
        
        this.connect = connect ;
        this.userId = userId ;
        this.userOwner = userOwner ;
        this.loginUserName = loginUser ;
        this.loginUserPasswd = loginPasswd ;
        
        this.queueList = new ArrayList() ;
        this.adminUser = new AdminUserImple( this ) ;
        this.adminQueueManager = new AdminQueueManagerImple( this ) ;
        this.adminChannel = new AdminChannelImple( this ) ;
        this.adminConnect = new AdminConnectImple( this ) ;
        this.adminCache = new AdminCacheImple( this ) ;
        
    }
    
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     * <BR>
     * @exception Exception 例外.
     */
    protected void finalize() throws Exception {
        this.close() ;
    }
    
    /**
     * コネクションをクローズ.
     * <BR><BR>
     * コネクションをクローズします.
     */
    public void close() {
        if( this.connect != null ) {
            try {
                ExecutionLogin.logout(
                    connect,userId,userOwner ) ;
            } catch( Exception e ) {
            }
            try {
                this.connect.destroy() ;
            } catch( Exception e ) {
            }
        }
        this.connect = null ;
        this.loginUserName = null ;
        this.loginUserPasswd = null ;
        this.userId = -1 ;
        this.userOwner = false ;
        this.queueList = null ;
        this.adminUser = null ;
        this.adminQueueManager = null ;
        this.adminChannel = null ;
        this.adminConnect = null ;
        this.adminCache = null ;
    }
    
    /**
     * 再度ログイン.
     * <BR><BR>
     * 再度ログイン処理を行います.
     * <BR>
     * @param user 対象のログインユーザ名を設定します.
     * @param passwd 対象のログインパスワードを設定します.
     * @exception MaachangQException MaachangQ例外.
     */
    public void login( String user,String passwd )
        throws MaachangQException {
        
        if( user == null || ( user = user.trim() ).length() <= 0 ) {
            throw new MaachangQException( "引数は不正です" ) ;
        }
        
        // ログイン処理.
        LoginBean bean = ExecutionLogin.login( connect,user,passwd ) ;
        
        // ログイン結果を再設定.
        this.loginUserName = user ;
        this.loginUserPasswd = passwd ;
        this.userId = bean.getId() ;
        this.userOwner = bean.isRootOwner() ;
        
    }
    
    /**
     * MaachangQシャットダウン.
     * <BR><BR>
     * 接続先のMaachangQをシャットダウンします.<BR>
     * このメソッドは、管理者権限を持たない場合、処理できません.
     * <BR>
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized void shutdown() throws MaachangQException {
        if( this.isOpen() == true ) {
            ExecutionLogin.shutdown(
                connect,loginUserName,loginUserPasswd ) ;
        }
    }
    
    /**
     * ユーザ管理オブジェクトを取得.
     * <BR><BR>
     * ユーザ管理オブジェクトを取得します.
     * <BR>
     * @return AdminUser ユーザ管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminUser getAdminUser()
        throws MaachangQException {
        return adminUser ;
    }
    
    /**
     * キューマネージャ管理オブジェクトを取得.
     * <BR><BR>
     * キューマネージャ管理オブジェクトを取得します.
     * <BR>
     * @return AdminQueueManager キューマネージャ管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminQueueManager getAdminQueueManager()
        throws MaachangQException {
        return adminQueueManager ;
    }
    
    /**
     * キュー管理オブジェクトを取得.
     * <BR><BR>
     * キュー管理オブジェクトを取得します.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     * @return AdminQueue キュー管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminQueue getAdminQueue( String manager )
        throws MaachangQException {
        
        if( this.isOpen() == false ) {
            return null ;
        }
        
        if( manager == null ||
            ( manager = manager.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        // キューマネージャ存在確認.
        ExecutionQueueManager.isQueueManager( connect,userId,userOwner,manager ) ;
        
        // 対象キュー情報を取得.
        AdminQueueImple ret = null ;
        
        int no = this.checkManager( manager ) ;
        if( no <= -1 ) {
            ret = new AdminQueueImple( this,manager ) ;
            queueList.add( ret ) ;
        }
        else {
            ret = ( AdminQueueImple )queueList.get( no ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューマネージャを削除した場合に呼び出す処理.
     * <BR><BR>
     * キューマネージャを削除した場合に呼び出す必要があります.
     * <BR>
     * @param manager 対象のキューマネージャ名を設定します.
     */
    protected synchronized void removeQueueManagerTo( String manager ) {
        if( this.isOpen() == false ) {
            return ;
        }
        int no = this.checkManager( manager ) ;
        if( no >= 0 ) {
            queueList.remove( no ) ;
        }
    }
    
    /**
     * チャネル管理オブジェクトを取得.
     * <BR><BR>
     * チャネル管理オブジェクトを取得します.
     * <BR>
     * @return AdminQueueManager チャネル管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminChannel getAdminChannel()
        throws MaachangQException {
        return adminChannel ;
    }
    
    /**
     * コネクション管理オブジェクトを取得.
     * <BR><BR>
     * コネクション管理オブジェクトを取得します.
     * <BR>
     * @return AdminConnect コネクション管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminConnect getAdminConnect()
        throws MaachangQException {
        return adminConnect ;
    }
    
    /**
     * キャッシュ管理オブジェクトを取得.
     * <BR><BR>
     * キャッシュ管理オブジェクトを取得します.
     * <BR>
     * @return AdminCache キャッシュ管理オブジェクトが返されます.
     * @exception MaachangQException MaachangQ例外.
     */
    public synchronized AdminCache getAdminCache()
        throws MaachangQException {
        return adminCache ;
    }
    
    /**
     * 現在接続中のユーザ名を取得.
     * <BR><BR>
     * 現在接続中のユーザ名を取得します.
     * <BR>
     * @return String 現在接続中のユーザ名が返されます.
     */
    public synchronized String getUserName() {
        return loginUserName ;
    }
    
    /**
     * 接続先アドレスを取得.
     * <BR><BR>
     * 接続先のアドレスを取得します.
     * <BR>
     * @return InetAddress 接続先アドレスが返されます.
     */
    public synchronized InetAddress getInetAddress() {
        
        try {
            return connect.getRemoteAddress() ;
        } catch( Exception e ) {
            return null ;
        }
        
    }
    
    /**
     * 接続先ポート番号を取得.
     * <BR><BR>
     * 接続先ポート番号を取得します.
     * <BR>
     * @return int 接続先ポート番号が返されます.
     */
    public synchronized int getPort() {
        
        try {
            return connect.getRemotePort() ;
        } catch( Exception e ) {
            return -1 ;
        }
        
    }
    
    /**
     * 接続ユーザの管理者権限を取得.
     * <BR><BR>
     * 接続ユーザの管理者権限を取得します.
     * <BR>
     * @return boolean 権限情報が返されます.<BR>
     *                 [true]が返された場合、コネクションユーザは管理者です.<BR>
     *                 [false]が返された場合、コネクションユーザは、一般ユーザです.
     */
    public synchronized boolean isRootOwner() {
        return userOwner ;
    }
    
    /**
     * キューマネージャがオープン中かチェック.
     * <BR><RB>
     * 対象のキューマネージャがオープン中であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オープン中です.<BR>
     *                 [false]が返された場合、クローズしています.
     */
    public synchronized boolean isOpen() {
        
        if( connect != null && userId != -1 ) {
            return ( connect.isClose() == false ) ? true : false ;
        }
        return false ;
        
    }
    
    /**
     * ログインパスワードを取得.
     * <BR><BR>
     * ログインパスワードを取得します.
     * <BR>
     * @return String ログインパスワードが返されます.
     */
    protected synchronized String getLoginPasswd() {
        return loginUserPasswd ;
    }
    
    /**
     * ログインIDを取得.
     * <BR><BR>
     * ログインIDを取得します.
     * <BR>
     * @return int ログインIDが返されます.
     */
    protected synchronized int getUserId() {
        return userId ;
    }
    
    /**
     * コネクションオブジェクトを取得.
     * <BR><BR>
     * コネクションオブジェクトを取得します.
     * <BR>
     * @return ConnectObject コネクションオブジェクトが返されます.
     */
    protected synchronized ConnectObject getConnect() {
        return connect ;
    }
    
    
    /**
     * 対象マネージャが存在するかチェック.
     */
    private final int checkManager( String name ) {
        
        AdminQueueImple conn = null ;
        int len = queueList.size() ;
        
        for( int i = 0 ; i < len ; i ++ ) {
            
            conn = ( AdminQueueImple )queueList.get( i ) ;
            if( name.equals( conn.getQueueManagerName() ) ) {
                return i ;
            }
            
        }
        
        return -1 ;
    }
}

