/*
 * @(#)MessageProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol ;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * メッセージ送受信用プロトコル.
 * <BR><BR>
 * メッセージ送受信用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/08
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class MessageProtocol {
    
    /**
     * メッセージ用ヘッダ.
     */
    public static final byte[] HEADER_MESSAGEE = {
        ( byte )0x0000093,( byte )0x000000e5 } ;
    
    /**
     * 処理カテゴリ : 電文送受信系.
     */
    public static final int CATEGORY_TYPE_MESSAGE = 0x10000000 ;
    
    /**
     * 処理タイプ : キューマネージャ接続.
     */
    public static final int TYPE_CONNECT_MANAGER = 0x00000001 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : キュー接続.
     */
    public static final int TYPE_CONNECT_QUEUE = 0x00000002 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : 電文送信.
     */
    public static final int TYPE_SEND = 0x00000003 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : 電文受信.
     */
    public static final int TYPE_RECEIVE = 0x00000004 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : トランザクション開始.
     */
    public static final int TYPE_TRANSACTION = 0x00000005 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : コミット電文.
     */
    public static final int TYPE_COMMIT = 0x00000006 | CATEGORY_TYPE_MESSAGE ;
    
    /**
     * 処理タイプ : ロールバック電文.
     */
    public static final int TYPE_ROLLBACK = 0x00000007 | CATEGORY_TYPE_MESSAGE ;
    
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_CONNECT_MANAGER :
            case TYPE_CONNECT_QUEUE :
            case TYPE_SEND :
            case TYPE_RECEIVE :
            case TYPE_TRANSACTION :
            case TYPE_COMMIT :
            case TYPE_ROLLBACK :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param type 処理タイプを設定します.
     * @param bean 対象のメッセージBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int type,MessageBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return MessageProtocol.createProtocol(
            id,type,bean.getQueueManager(),bean.getQueue(),bean.getQueueType(),
            bean.getKey(),bean.getProcessId(),bean.getMessage(),
            bean.getExpire(),bean.getPriority() ) ;
        
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param type 処理タイプを設定します.
     * @param manager 対象のキューマネージャ名を設定します.
     * @param queue 対象のキュー名を設定します.
     * @param queueType 対象のキュータイプを設定します.
     * @param key 対象のキーコードを設定します.
     * @param processId 対象のプロセスIDを設定します.
     * @param message 対象のメッセージ情報を設定します.
     * @param expire 対象のExpire情報を設定します.
     * @param priority 対象のプライオリティ値を設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int type,String manager,
        String queue,int queueType,String key,String processId,byte[] message,
        int expire,int priority )
        throws Exception {
        
        int messageLen ;
        
        // メッセージが不正な場合.
        if( message == null || ( messageLen = message.length ) <= 0 ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 処理タイプが不正な場合.
        if( MessageProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        int keyLen = 0 ;
        byte[] keyBin = null ;
        if( key != null && ( key = key.trim() ).length() >= 0 ) {
            keyBin = key.getBytes( CommonProtocol.CHARSET ) ;
            keyLen = keyBin.length ;
        }
        
        int processLen = 0 ;
        byte[] processBin = null ;
        if( processId != null && ( processId = processId.trim() ).length() >= 0 ) {
            processBin = processId.getBytes( CommonProtocol.CHARSET ) ;
            processLen = keyBin.length ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // expire値.
            4 +                 // priority値.
            4 +                 // キー長.
            keyLen +            // キー.
            4 +                 // プロセスID長.
            processLen +        // プロセスID.
            4 +                 // メッセージ長.
            messageLen ;        // メッセージ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseProtocol.createCommonTelegram(
            nextPnt,binaryLength,type,manager,queue,queueType ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // expire値.
        expire = ( expire <= 0 ) ? -1 : expire ;
        ConvertBinary.convertInt( telegram,pnt,expire ) ;
        pnt += 4 ;
        
        // priority値.
        priority = ( priority == -1 ) ?
            MaachangQAccessDefine.DEFAULT_PRIORITY : priority ;
        priority = ( priority <= MaachangQAccessDefine.MIN_PRIORITY ) ?
            MaachangQAccessDefine.MIN_PRIORITY : priority ;
        priority = ( priority >= MaachangQAccessDefine.MAX_PRIORITY ) ?
            MaachangQAccessDefine.MAX_PRIORITY : priority ;
        ConvertBinary.convertInt( telegram,pnt,priority ) ;
        pnt += 4 ;
        
        // キー長.
        ConvertBinary.convertInt( telegram,pnt,keyLen ) ;
        pnt += 4 ;
        
        // キー.
        if( keyLen > 0 ) {
            System.arraycopy( keyBin,0,telegram,pnt,keyLen ) ;
            pnt += keyLen ;
            keyBin = null ;
        }
        
        // プロセスID長.
        ConvertBinary.convertInt( telegram,pnt,processLen ) ;
        pnt += 4 ;
        
        // プロセスID.
        if( processLen > 0 ) {
            System.arraycopy( processBin,0,telegram,pnt,processLen ) ;
            pnt += processLen ;
            processBin = null ;
        }
        
        // メッセージ長.
        ConvertBinary.convertInt( telegram,pnt,messageLen ) ;
        pnt += 4 ;
        
        // メッセージ
        System.arraycopy( message,0,telegram,pnt,messageLen ) ;
        pnt += messageLen ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_MESSAGEE,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( MessageBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // Expire値を取得.
        int expire = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // Priority値を取得.
        int priority = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キー長を取得.
        int keyLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キーを取得.
        String key = null ;
        if( keyLen > 0 ) {
            byte[] keyBin = new byte[ keyLen ] ;
            System.arraycopy( telegram,pnt,keyBin,0,keyLen ) ;
            key = new String( keyBin,CommonProtocol.CHARSET ) ;
            keyBin = null ;
            pnt += keyLen ;
        }
        
        // プロセスID長を取得.
        int processLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // プロセスIDを取得.
        String processId = null ;
        if( processLen > 0 ) {
            byte[] processBin = new byte[ processLen ] ;
            System.arraycopy( telegram,pnt,processBin,0,processLen ) ;
            processId = new String( processBin,CommonProtocol.CHARSET ) ;
            processBin = null ;
            pnt += processLen ;
        }
        
        // メッセージ長を取得.
        int messageLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        byte[] message = new byte[ messageLen ] ;
        System.arraycopy( telegram,pnt,message,0,messageLen ) ;
        
        // Beanに設定.
        out.setExpire( expire ) ;
        out.setPriority( priority ) ;
        out.setKey( key ) ;
        out.setProcessId( processId ) ;
        out.setMessage( message ) ;
        
    }
    
}

