/*
 * @(#)AdminChannelProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import java.net.InetAddress;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者ユーザ用プロトコル.
 * <BR><BR>
 * 管理者ユーザ用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminChannelProtocol {
    
    /**
     * ユーザ処理用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_CHANNEL = {
        ( byte )0x000000ac,( byte )0x000000a1 } ;
    
    /**
     * 処理カテゴリ : 管理者用チャネル系.
     */
    public static final int CATEGORY_TYPE_ADMIN_CHANNEL = 0x85000000 ;
    
    /**
     * 処理タイプ : チャネル追加.
     */
    public static final int TYPE_ADD_CHANNEL = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル変更.
     */
    public static final int TYPE_SET_CHANNEL = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル削除.
     */
    public static final int TYPE_REMOVE_CHANNEL = 0x00000003 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル停止.
     */
    public static final int TYPE_STOP_CHANNEL = 0x00000004 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル開始.
     */
    public static final int TYPE_START_CHANNEL = 0x00000005 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル取得.
     */
    public static final int TYPE_GET_CHANNEL = 0x00000006 |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : 存在するチャネル一覧を取得.
     */
    public static final int TYPE_GET_CHANNEL_NAMES = 0x00000007 |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : 存在するチャネル数を取得.
     */
    public static final int TYPE_GET_CHANNEL_SIZE = 0x00000008 |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    /**
     * 処理タイプ : チャネル存在チェック.
     */
    public static final int TYPE_IS_CHANNEL = 0x00000009 |
        CATEGORY_TYPE_ADMIN_CHANNEL ;
    
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_CHANNEL :
            case TYPE_SET_CHANNEL :
            case TYPE_REMOVE_CHANNEL :
            case TYPE_STOP_CHANNEL :
            case TYPE_START_CHANNEL :
            case TYPE_GET_CHANNEL :
            case TYPE_GET_CHANNEL_NAMES :
            case TYPE_GET_CHANNEL_SIZE :
            case TYPE_IS_CHANNEL :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminChannelBean bean )
        throws Exception {
        
        if( id < 0 || bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminChannelProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getChannelType(),bean.getQueueManager(),
            bean.getHost(),bean.getPort(),bean.getConnectName(),
            bean.getUseCb32Word(),bean.getState(),bean.getNumbers(),
            bean.getSize(),bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name チャネル名を設定します.
     * @param channelType チャネルタイプを設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先ポートを設定します.
     * @param connectName コネクション名を設定します.
     * @param useCb32Word 利用暗号ワードを設定します.
     * @param state 対象のステータスを設定します.
     * @param numbers 対象の数値パラメータを設定します.
     * @param size 存在するチャネル名一覧を設定します.
     * @param params 処理パラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,
        boolean rootOwner,int type,String name,int channelType,
        String queueManager,InetAddress host,int port,String connectName,
        String useCb32Word,int state,int[] numbers,int size,String[] params )
        throws Exception {
        
        String hostString = null ;
        
        if( host != null ) {
            hostString = host.getHostName() ;
        }
        
        return AdminChannelProtocol.createProtocol(
            id,userId,rootOwner,type,name,channelType,
            queueManager,hostString,port,connectName,
            useCb32Word,state,numbers,size,params ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name チャネル名を設定します.
     * @param channelType チャネルタイプを設定します.
     * @param queueManager 対象のキューマネージャ名を設定します.
     * @param host 接続先のホスト名を設定します.
     * @param port 接続先ポートを設定します.
     * @param connectName コネクション名を設定します.
     * @param useCb32Word 利用暗号ワードを設定します.
     * @param state 対象のステータスを設定します.
     * @param numbers 対象の数値パラメータを設定します.
     * @param size 存在するチャネル名一覧を設定します.
     * @param params 処理パラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,
        boolean rootOwner,int type,String name,int channelType,
        String queueManager,String host,int port,String connectName,
        String useCb32Word,int state,int[] numbers,int size,String[] params )
        throws Exception {
        
        // 引数が不正.
        if( channelType != MaachangQAccessDefine.TYPE_SEND &&
                channelType != MaachangQAccessDefine.TYPE_RECEIVE ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        // 処理タイプが不正な場合.
        if( AdminChannelProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // キューマネージャ名をバイナリ変換.
        int managerLen = 0 ;
        byte[] managerBin = null ;
        if( queueManager != null &&
            ( queueManager = queueManager.trim().toLowerCase() ).length() > 0 ) {
            managerBin = queueManager.getBytes( CommonProtocol.CHARSET ) ;
            managerLen = managerBin.length ;
        }
        
        // ホスト名をバイナリ変換.
        int hostLen = 0 ;
        byte[] hostBin = null ;
        if( host != null && ( host = host.trim() ).length() > 0 ) {
            hostBin = host.getBytes( CommonProtocol.CHARSET ) ;
            hostLen = hostBin.length ;
        }
        
        // コネクション名をバイナリ変換.
        int connectLen = 0 ;
        byte[] connectBin = null ;
        if( connectName != null &&
            ( connectName = connectName.trim().toLowerCase() ).length() > 0 ) {
            connectBin = connectName.getBytes( CommonProtocol.CHARSET ) ;
            connectLen = connectBin.length ;
        }
        
        // 利用暗号ワードをバイナリ変換.
        int cb32WordLen = 0 ;
        byte[] cb32WordBin = null ;
        if( useCb32Word != null && useCb32Word.length() > 0 ) {
            cb32WordBin = useCb32Word.getBytes( CommonProtocol.CHARSET ) ;
            cb32WordLen = cb32WordBin.length ;
        }
        
        // 数値パラメータ数を取得.
        int numbersLen = 0 ;
        if( numbers != null ) {
            numbersLen = numbers.length ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // チャネルタイプ.
            4 +                 // キューマネージャ名長.
            managerLen +        // キューマネージャ名.
            4 +                 // ホスト名長.
            hostLen +           // ホスト名.
            4 +                 // ポート番号.
            4 +                 // コネクション名長.
            connectLen +        // コネクション名.
            4 +                 // 利用暗号ワード長.
            cb32WordLen +       // 利用暗号ワード.
            4 +                 // ステータス.
            4 +                 // 数値パラメータ数.
            numbersLen * 4 ;    // 数値パラメータ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // チャネルタイプ.
        ConvertBinary.convertInt( telegram,pnt,channelType ) ;
        pnt += 4 ;
        
        // キューマネージャ名長.
        ConvertBinary.convertInt( telegram,pnt,managerLen ) ;
        pnt += 4 ;
        
        // キューマネージャ名.
        if( managerLen > 0 ) {
            System.arraycopy( managerBin,0,telegram,pnt,managerLen ) ;
            pnt += managerLen ;
            managerBin = null ;
        }
        
        // ホスト名長.
        ConvertBinary.convertInt( telegram,pnt,hostLen ) ;
        pnt += 4 ;
        
        // ホスト名.
        if( hostLen > 0 ) {
            System.arraycopy( hostBin,0,telegram,pnt,hostLen ) ;
            pnt += hostLen ;
            hostBin = null ;
        }
        
        // ポート番号.
        ConvertBinary.convertInt( telegram,pnt,port ) ;
        pnt += 4 ;
        
        // コネクション名長.
        ConvertBinary.convertInt( telegram,pnt,connectLen ) ;
        pnt += 4 ;
        
        // コネクション名.
        if( connectLen > 0 ) {
            System.arraycopy( connectBin,0,telegram,pnt,connectLen ) ;
            pnt += connectLen ;
            connectBin = null ;
        }
        
        // コネクション名長.
        ConvertBinary.convertInt( telegram,pnt,cb32WordLen ) ;
        pnt += 4 ;
        
        // 利用暗号ワード.
        if( cb32WordLen > 0 ) {
            System.arraycopy( cb32WordBin,0,telegram,pnt,cb32WordLen ) ;
            pnt += cb32WordLen ;
            cb32WordBin = null ;
        }
        
        // ステータス.
        ConvertBinary.convertInt( telegram,pnt,state ) ;
        pnt += 4 ;
        
        // 数値パラメータ数.
        ConvertBinary.convertInt( telegram,pnt,numbersLen ) ;
        pnt += 4 ;
        
        // 数値パラメータ.
        for( int i = 0 ; i < numbersLen ; i ++ ) {
            ConvertBinary.convertInt( telegram,pnt,numbers[ i ] ) ;
            pnt += 4 ;
        }
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_CHANNEL,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminChannelBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // チャネルタイプを取得.
        int channelType = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キューマネージャ名長を取得.
        int managerLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // マネージャ名を取得.
        String manager = null ;
        if( managerLen > 0 ) {
            byte[] managerBin = new byte[ managerLen ] ;
            System.arraycopy( telegram,pnt,managerBin,0,managerLen ) ;
            manager = new String( managerBin,CommonProtocol.CHARSET ) ;
            managerBin = null ;
            pnt += managerLen ;
        }
        
        // ホスト名長を取得.
        int hostLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // ホスト名を取得.
        String host = null ;
        if( hostLen > 0 ) {
            byte[] hostBin = new byte[ hostLen ] ;
            System.arraycopy( telegram,pnt,hostBin,0,hostLen ) ;
            host = new String( hostBin,CommonProtocol.CHARSET ) ;
            hostBin = null ;
            pnt += hostLen ;
        }
        
        // ポート番号を取得.
        int port = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // コネクション名長を取得.
        int connectLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // コネクション名を取得.
        String connect = null ;
        if( connectLen > 0 ) {
            byte[] connectBin = new byte[ connectLen ] ;
            System.arraycopy( telegram,pnt,connectBin,0,connectLen ) ;
            connect = new String( connectBin,CommonProtocol.CHARSET ) ;
            connectBin = null ;
            pnt += connectLen ;
        }
        
        // コネクション名長を取得.
        int cb32WordLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 利用暗号ワードを取得.
        String useCb32Word = null ;
        if( cb32WordLen > 0 ) {
            byte[] cb32WordBin = new byte[ cb32WordLen ] ;
            System.arraycopy( telegram,pnt,cb32WordBin,0,cb32WordLen ) ;
            useCb32Word = new String( cb32WordBin,CommonProtocol.CHARSET ) ;
            cb32WordBin = null ;
            pnt += cb32WordLen ;
        }
        
        // ステータスを取得.
        int state = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 数値パラメータ数を取得.
        int numbersLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 数値パラメータ.
        int[] numbers = null ;
        if( numbersLen > 0 ) {
            numbers = new int[ numbersLen ] ;
            for( int i = 0 ; i < numbersLen ; i ++ ) {
                numbers[ i ] = ConvertBinary.convertInt( pnt,telegram ) ;
                pnt += 4 ;
            }
        }
        
        // Beanに設定.
        out.setChannelType( channelType ) ;
        out.setQueueManager( manager ) ;
        out.setHost( host ) ;
        out.setPort( port ) ;
        out.setConnectName( connect ) ;
        out.setUseCb32Word( useCb32Word ) ;
        out.setState( state ) ;
        out.setNumbers( numbers ) ;
        
    }
    
}

