/*
 * @(#)AdminConnectProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import java.net.InetAddress;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者コネクション用プロトコル.
 * <BR><BR>
 * 管理者コネクション用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminConnectProtocol {
    
    /**
     * コネクション用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_CONNECT = {
        ( byte )0x00000ac,( byte )0x000000ce } ;
    
    /**
     * 処理カテゴリ : 管理者用コネクション系.
     */
    public static final int CATEGORY_TYPE_ADMIN_CONNECT = 0x86000000 ;
    
    /**
     * 処理タイプ : コネクション追加.
     */
    public static final int TYPE_ADD_CONNECT = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : コネクション設定.
     */
    public static final int TYPE_SET_CONNECT = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : コネクション削除.
     */
    public static final int TYPE_REMOVE_CONNECT = 0x00000003 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : コネクション情報取得.
     */
    public static final int TYPE_GET_CONNECT = 0x00000014 |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : 存在するコネクション名群取得.
     */
    public static final int TYPE_GET_CONNECT_NAMES = 0x00000015 |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : 存在するコネクション数取得.
     */
    public static final int TYPE_GET_CONNECT_SIZE = 0x00000016 |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプ : コネクションが存在するか取得.
     */
    public static final int TYPE_IS_CONNECT = 0x00000017 |
        CATEGORY_TYPE_ADMIN_CONNECT ;
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_CONNECT :
            case TYPE_SET_CONNECT :
            case TYPE_REMOVE_CONNECT :
            case TYPE_GET_CONNECT :
            case TYPE_GET_CONNECT_NAMES :
            case TYPE_GET_CONNECT_SIZE :
            case TYPE_IS_CONNECT :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminConnectBean bean )
        throws Exception {
        
        if( id < 0 || bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminConnectProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getAddress(),bean.getPort(),
            bean.getCacheName(),bean.getBuffer(),
            bean.isCb32(),bean.getSize(),bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のコネクション名を設定します.
     * @param address 対象のバインド先アドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param cacheName 対象のキャッシュ名を設定します.
     * @param buffer 対象の送受信バッファ長を設定します.
     * @param cb32 対象の暗号フラグを設定します.
     * @param size 対象のコネクションサイズを設定します.
     * @param params 対象のパラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,boolean rootOwner,int type,
        String name,InetAddress address,int port,String cacheName,int buffer,
        boolean cb32,int size,String[] params )
        throws Exception {
        
        String addressString = null ;
        
        if( address != null ) {
            addressString = address.getHostName() ;
        }
        
        return AdminConnectProtocol.createProtocol(
            id,userId,rootOwner,type,name,addressString,port,
            cacheName,buffer,cb32,size,params ) ;
        
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のコネクション名を設定します.
     * @param address 対象のバインド先アドレスを設定します.
     * @param port 対象のポート番号を設定します.
     * @param cacheName 対象のキャッシュ名を設定します.
     * @param buffer 対象の送受信バッファ長を設定します.
     * @param cb32 対象の暗号フラグを設定します.
     * @param size 対象のコネクションサイズを設定します.
     * @param params 対象のパラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,boolean rootOwner,int type,
        String name,String address,int port,String cacheName,int buffer,
        boolean cb32,int size,String[] params )
        throws Exception {
        
        // 処理タイプが不正な場合.
        if( AdminConnectProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // バインドアドレスをバイナリ変換.
        int addressLen = 0 ;
        byte[] addressBin = null ;
        if( address != null && ( address = address.trim() ).length() > 0 ) {
            addressBin = address.getBytes( CommonProtocol.CHARSET ) ;
            addressLen = addressBin.length ;
        }
        
        // キャッシュ名をバイナリ変換.
        int cacheLen = 0 ;
        byte[] cacheBin = null ;
        if( cacheName != null &&
            ( cacheName = cacheName.trim().toLowerCase() ).length() > 0 ) {
            cacheBin = cacheName.getBytes( CommonProtocol.CHARSET ) ;
            cacheLen = cacheBin.length ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // バインドアドレス長.
            addressLen +        // バインドアドレス.
            4 +                 // ポート番号.
            4 +                 // キャッシュ名長.
            cacheLen +          // キャッシュ名.
            4 +                 // 送受信バッファ.
            1 ;                 // 暗号フラグ.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // バインドアドレス長.
        ConvertBinary.convertInt( telegram,pnt,addressLen ) ;
        pnt += 4 ;
        
        // バインドアドレス.
        if( addressLen > 0 ) {
            System.arraycopy( addressBin,0,telegram,pnt,addressLen ) ;
            pnt += addressLen ;
            addressBin = null ;
        }
        
        // バインドポート番号.
        ConvertBinary.convertInt( telegram,pnt,port ) ;
        pnt += 4 ;
        
        // キャッシュ名長.
        ConvertBinary.convertInt( telegram,pnt,cacheLen ) ;
        pnt += 4 ;
        
        // キャッシュ名.
        if( cacheLen > 0 ) {
            System.arraycopy( cacheBin,0,telegram,pnt,cacheLen ) ;
            pnt += cacheLen ;
            cacheBin = null ;
        }
        
        // 送受信バッファ.
        ConvertBinary.convertInt( telegram,pnt,buffer ) ;
        pnt += 4 ;
        
        // 暗号フラグ.
        ConvertBinary.convertBoolean( telegram,pnt,cb32 ) ;
        pnt += 1 ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_CONNECT,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminConnectBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // バインドアドレス長を取得.
        int addressLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // バインドアドレスを取得.
        String address = null ;
        if( addressLen > 0 ) {
            byte[] addressBin = new byte[ addressLen ] ;
            System.arraycopy( telegram,pnt,addressBin,0,addressLen ) ;
            address = new String( addressBin,CommonProtocol.CHARSET ) ;
            addressBin = null ;
            pnt += addressLen ;
        }
        
        // ポート番号を取得.
        int port = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キャッシュ名長を取得.
        int cacheLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // キャッシュ名を取得.
        String cache = null ;
        if( cacheLen > 0 ) {
            byte[] cacheBin = new byte[ cacheLen ] ;
            System.arraycopy( telegram,pnt,cacheBin,0,cacheLen ) ;
            cache = new String( cacheBin,CommonProtocol.CHARSET ) ;
            cacheBin = null ;
            pnt += cacheLen ;
        }
        
        // 送受信バッファを取得.
        int buffer = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 暗号フラグを取得.
        boolean cb32 = ConvertBinary.convertBoolean( pnt,telegram ) ;
        pnt += 1 ;
        
        // Beanに設定.
        out.setAddress( address ) ;
        out.setPort( port ) ;
        out.setCacheName( cache ) ;
        out.setBuffer( buffer ) ;
        out.setCb32( cb32 ) ;
        
    }
    
}

