/*
 * @(#)AdminQueueManagerProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者キューマネージャ用プロトコル.
 * <BR><BR>
 * 管理者キューマネージャ用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminQueueManagerProtocol {
    
    /**
     * キューマネージャ用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_QMANAGER = {
        ( byte )0x00000a9,( byte )0x0000003e } ;
    
    /**
     * 処理カテゴリ : 管理者用キューマネージャ系.
     */
    public static final int CATEGORY_TYPE_ADMIN_QUEUE_MANAGER = 0x84000000 ;
    
    /**
     * 処理タイプ : マネージャ追加.
     */
    public static final int TYPE_ADD_QMANAGER = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : マネージャ削除.
     */
    public static final int TYPE_REMOVE_QMANAGER = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : マネージャ停止.
     */
    public static final int TYPE_STOP_QMANAGER = 0x00000003 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : マネージャ開始.
     */
    public static final int TYPE_START_QMANAGER = 0x00000004 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : キューマネージャ情報を取得.
     */
    public static final int TYPE_GET_QMANAGER = 0x00000015 |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : 存在するキューマネージャ名群を取得.
     */
    public static final int TYPE_GET_QMANAGER_NAMES = 0x00000016 |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : 存在するキューマネージャ数を取得.
     */
    public static final int TYPE_GET_QMANAGER_SIZE = 0x00000017 |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプ : キューマネージャが存在するか取得.
     */
    public static final int TYPE_IS_QMANAGER = 0x00000018 |
        CATEGORY_TYPE_ADMIN_QUEUE_MANAGER ;
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_QMANAGER :
            case TYPE_REMOVE_QMANAGER :
            case TYPE_STOP_QMANAGER :
            case TYPE_START_QMANAGER :
            case TYPE_GET_QMANAGER :
            case TYPE_GET_QMANAGER_NAMES :
            case TYPE_GET_QMANAGER_SIZE :
            case TYPE_IS_QMANAGER :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminQueueManagerBean bean )
        throws Exception {
        
        if( bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminQueueManagerProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getMaxSector(),bean.getUseSector(),
            bean.getState(),bean.getSize(),bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコルを生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のキューマネージャ名を設定します.
     * @param maxSector 最大セクタ数を設定します.
     * @param useSector 現在利用セクタ数を設定します.
     * @param state 対象ステータスを設定します.
     * @param size 対象のキューマネージャサイズを設定します.
     * @param params 対象のパラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,boolean rootOwner,int type,
        String name,int maxSector,int useSector,int state,int size,String[] params )
        throws Exception {
        
        // 処理タイプが不正な場合.
        if( AdminQueueManagerProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 +                 // 最大セクタ長.
            4 +                 // 現在セクタ長.
            4 ;                 // ステータス.
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 最大セクタ長.
        maxSector = ( maxSector <= -1 ) ? -1 : maxSector ;
        ConvertBinary.convertInt( telegram,pnt,maxSector ) ;
        pnt += 4 ;
        
        // 現在セクタ長.
        useSector = ( useSector <= -1 ) ? -1 : useSector ;
        ConvertBinary.convertInt( telegram,pnt,useSector ) ;
        pnt += 4 ;
        
        // ステータス.
        state = ( state <= -1 ) ? -1 : state ;
        ConvertBinary.convertInt( telegram,pnt,state ) ;
        pnt += 4 ;
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_QMANAGER,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminQueueManagerBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // 最大セクタ数を取得.
        int maxSector = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // 現在セクタ数を取得.
        int useSector = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // ステータスを取得.
        int state = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // Beanに設定.
        out.setMaxSector( maxSector ) ;
        out.setUseSector( useSector ) ;
        out.setState( state ) ;
        
    }
    
}

