/*
 * @(#)AdminUserProtocol.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.queue.access.protocol.admin ;

import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.protocol.CommonProtocol;
import org.maachang.queue.access.util.ConvertBinary;


/**
 * 管理者ユーザ用プロトコル.
 * <BR><BR>
 * 管理者ユーザ用プロトコルを表すオブジェクト.
 *  
 * @version 2007/01/05
 * @author  masahito suzuki
 * @since   MaachangQ-Access 1.00
 */
public class AdminUserProtocol {
    
    /**
     * ユーザ処理用ヘッダ.
     */
    public static final byte[] HEADER_ADMIN_USER = {
        ( byte )0x000000ac,( byte )0x000000e7 } ;
    
    /**
     * 処理カテゴリ : 管理者用ユーザ系.
     */
    public static final int CATEGORY_TYPE_ADMIN_USER = 0x82000000 ;
    
    /**
     * 処理タイプ : ユーザ追加.
     */
    public static final int TYPE_ADD_USER = 0x00000001 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : ユーザ削除.
     */
    public static final int TYPE_REMOVE_USER = 0x00000002 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : 指定ユーザチェンジパスワード.
     */
    public static final int TYPE_SET_USER_CHANGE_PASSWD = 0x00000003 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : 自ユーザチェンジパスワード.
     */
    public static final int TYPE_THIS_USER_CHANGE_PASSWD = 0x00000004 |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : ルート権限を変更.
     */
    public static final int TYPE_CHANGE_OWNER = 0x00000005 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : ユーザ情報一覧を取得.
     */
    public static final int TYPE_GET_USERS = 0x00000016 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : ユーザサイズを取得.
     */
    public static final int TYPE_USER_SIZE = 0x00000017 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : 対象ユーザに対するルート権限を取得.
     */
    public static final int TYPE_USER_BY_ROOT_OWNER = 0x00000018 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプ : 対象ユーザが存在するかチェック.
     */
    public static final int TYPE_IS_USER = 0x00000019 |
        CommonProtocol.MASK_ROOT_OWNER |
        CATEGORY_TYPE_ADMIN_USER ;
    
    /**
     * 処理タイプチェック.
     * <BR><BR>
     * 処理タイプをチェックします.
     * <BR>
     * @param type 処理タイプを設定します.
     * @return boolean 処理結果が返されます.<BR>
     *                  [true]が返された場合、処理タイプは存在します.<BR>
     *                  [false]が返された場合、処理タイプは存在しません.
     */
    public static final boolean checkType( int type ) {
        switch( type ) {
            case TYPE_ADD_USER :
            case TYPE_REMOVE_USER :
            case TYPE_SET_USER_CHANGE_PASSWD :
            case TYPE_THIS_USER_CHANGE_PASSWD :
            case TYPE_CHANGE_OWNER :
            case TYPE_GET_USERS :
            case TYPE_USER_SIZE :
            case TYPE_USER_BY_ROOT_OWNER :
            case TYPE_IS_USER :
            return true ;
        }
        return false ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param bean 設定対象のBeanを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol(
        int id,boolean rootOwner,AdminUserBean bean )
        throws Exception {
        
        if( id < 0 || bean == null ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        return AdminUserProtocol.createProtocol(
            id,bean.getUserId(),rootOwner,bean.getType(),
            bean.getName(),bean.getFlags(),
            bean.getSize(),bean.getParams() ) ;
    }
    
    /**
     * プロトコルを生成.
     * <BR><BR>
     * プロトコル処理を生成します.
     * <BR>
     * @param id プロトコルIDを設定します.
     * @param userId ログインユーザIDを設定します.
     * @param rootOwner ルートオーナを設定します.
     * @param type 処理タイプを設定します.
     * @param name 対象のユーザ名を設定します.
     * @param flags 処理フラグを設定します.
     * @param size ユーザサイズを設定します.
     * @param params 処理パラメータを設定します.
     * @return byte[] 対象のバイナリを設定します.
     * @exception Exception 例外.
     */
    public static final byte[] createProtocol( int id,int userId,
        boolean rootOwner,int type,String name,boolean[] flags,int size,
        String[] params )
        throws Exception {
        
        // 処理タイプが不正な場合.
        if( AdminUserProtocol.checkType( type ) == false ) {
            throw new MaachangQException(
                "処理タイプ(" + type + ")が不正です",
                MaachangQErrorCode.ERROR_NOT_TYPE ) ;
        }
        
        // 電文情報長を計算.
        int binaryLength = 
            4 ;                 // フラグパラメータ数.
        
        int paramLen = 0 ;
        
        // フラグパラメータが存在する場合.
        if( flags != null && flags.length > 0 ) {
            paramLen = flags.length ;
            binaryLength += paramLen  ;
        }
        
        // 電文情報を生成.
        int[] nextPnt = new int[ 1 ] ;
        byte[] telegram = BaseAdminProtocol.createCommonTelegram(
            nextPnt,binaryLength,id,userId,rootOwner,type,name,size,params ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // フラグパラメータ数.
        ConvertBinary.convertInt( telegram,pnt,paramLen ) ;
        pnt += 4 ;
        
        // フラグパラメータを設定.
        for( int i = 0 ; i < paramLen ; i ++ ) {
            ConvertBinary.convertBoolean( telegram,pnt,flags[ i ] ) ;
            pnt += 1 ;
        }
        
        // チェックコード設定.
        CommonProtocol.setCheckCode( telegram ) ;
        
        // 電文情報生成.
        return CommonProtocol.createProtocol( HEADER_ADMIN_USER,id,telegram ) ;
        
    }
    
    /**
     * プロトコルを解析.
     * <BR><BR>
     * プロトコルを解析します.<BR>
     * この処理で渡される電文データは、あらかじめ[CommonProtocol.analysisProtocol()]で
     * 解析しておく必要があります.
     * <BR>
     * @param out 解析結果を格納するBeanを設定します.
     * @param telegram 解析対象の電文データを設定します.
     * @exception Exception 例外.
     */
    public static final void analysisProtocol( AdminUserBean out,byte[] telegram )
        throws Exception {
        
        // 電文情報を解析.
        int[] nextPnt = new int[ 1 ] ;
        BaseAdminProtocol.analysisCommonTelegram( nextPnt,out,telegram ) ;
        int pnt = nextPnt[ 0 ] ;
        
        // フラグパラメータ数を取得.
        int flagLen = ConvertBinary.convertInt( pnt,telegram ) ;
        pnt += 4 ;
        
        // フラグパラメータ情報群を取得.
        boolean[] flags = null ;
        if( flagLen > 0 ) {
            flags = new boolean[ flagLen ] ;
            for( int i = 0 ; i < flagLen ; i ++ ) {
                flags[ i ] = ConvertBinary.convertBoolean( pnt,telegram ) ;
                pnt += 1 ;
            }
            
        }
        
        // Beanに設定.
        out.setFlags( flags ) ;
        
    }
    
}

