/*
 * @(#)AdminConnectCommand.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.command.admin ;

import java.net.InetAddress;

import org.maachang.command.core.AnalysisCommand;
import org.maachang.command.core.ConnectParam;
import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.AdminConnect;
import org.maachang.queue.access.admin.MaachangQConnectAdmin;
import org.maachang.queue.access.admin.info.ConnectInfo;
import org.maachang.queue.access.admin.info.CreateConnectInfo;

/**
 * 管理者用コネクションコマンド.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ-Command 1.00
 */
public class AdminConnectCommand {
    
    /**
     * 実行条件.
     */
    private static final String EXEC_HEAD = "x" ;
    
    /**
     * コネクション名
     */
    private static final String CONNECT_NAME = "n" ;
    
    /**
     * キャッシュ名.
     */
    private static final String CACHE_NAME = "a" ;
    
    /**
     * 暗号フラグ.
     */
    private static final String CB32_FLAG = "f" ;
    
    /**
     * 暗号ワード群.
     */
    private static final String CB32_WORDS = "w" ;
    
    
    
    /**
     * 実行タイプ : リスト表示.
     * list.
     */
    private static final String EXEC_LIST = "l" ;
    
    /**
     * 実行タイプ : キャッシュ生成.
     * create.
     */
    private static final String EXEC_CREATE = "c" ;
    
    /**
     * 実行タイプ : キャッシュ設定.
     * set.
     */
    private static final String EXEC_SET = "s" ;
    
    /**
     * 実行タイプ : キャッシュ削除.
     * remove.
     */
    private static final String EXEC_DELETE = "r" ;
    
    
    
    /**
     * コンストラクタ.
     */
    private AdminConnectCommand() {
        
    }
    
    /**
     * メイン.
     * <BR><BR>
     * @param args
     * @exception Exception
     */
    public static final void main( String[] args )
        throws Exception {
        try {
            new AdminConnectCommand().execution( args ) ;
        } catch( Exception e ) {
            AnalysisCommand.outputError( e ) ;
        }
    }
    
    /**
     * 実行処理.
     * <BR><BR>
     * @param args
     * @exception Exception
     */
    private final void execution( String[] args )
        throws Exception {
        
        String exec = this.execType( args ) ;
        String connectName = this.getConnectName( args ) ;
        String cacheName = this.getCacheName( args ) ;
        boolean cb32Flag = this.getCb32Flag( args ) ;
        String[] cb32Words = this.getCb32Words( args ) ;
        ConnectParam connParam = AnalysisCommand.analysisConnect( args ) ;
        
        MaachangQConnectAdmin admin = AnalysisCommand.loginToConnect( args ) ;
        
        AdminConnect connect = admin.getAdminConnect() ;
        
        // コネクション条件が設定されていない場合は、
        // デフォルトポートを設定.
        if( connParam == null ) {
            connParam = new ConnectParam() ;
            connParam.setPort( MaachangQAccessDefine.DEFAULT_CHANNEL_PORT ) ;
            connParam.setAddress( null ) ;
        }
        
        // リスト表示.
        if( EXEC_LIST.equals( exec ) ) {
            if( connectName != null && connectName.length() > 0 ) {
                this.outputList( true,connect,connectName ) ;
            }
            else {
                String[] names = connect.getNames() ;
                this.outputNames( true,names ) ;
            }
        }
        // 生成.
        else if( EXEC_CREATE.equals( exec ) ) {
            
            if( connectName == null || connectName.length() <= 0 ) {
                throw new MaachangQException(
                    "対象コネクション名は存在しません" ) ;
            }
            
            if( connParam.getPort() < -1 ) {
                throw new MaachangQException(
                    "対象コネクションポートは不正です" ) ;
            }
            
            CreateConnectInfo info = new CreateConnectInfo() ;
            info.setName( connectName ) ;
            info.setMqCacheName( cacheName ) ;
            info.setBuffer( -1 ) ;
            if( connParam.getAddress() == null ) {
                info.setBindAddress( null ) ;
            }
            else {
                info.setBindAddress(
                    InetAddress.getByName( connParam.getAddress() ) ) ;
            }
            info.setBindPort( connParam.getPort() ) ;
            info.setCb32( cb32Flag ) ;
            info.setCb32Word( cb32Words ) ;
            
            connect.addConnect( info ) ;
            
            System.out.println( connectName +
                " コネクション作成に成功しました" ) ;
        }
        // 再設定.
        else if( EXEC_SET.equals( exec ) ) {
            
            if( connectName == null || connectName.length() <= 0 ) {
                throw new MaachangQException(
                    "対象コネクション名は存在しません" ) ;
            }
            
            ConnectInfo info = new ConnectInfo() ;
            info.setName( connectName ) ;
            info.setCb32( cb32Flag ) ;
            info.setCb32Word( cb32Words ) ;
            
            connect.setConnect( info ) ;
            
            System.out.println( connectName +
                " コネクション設定に成功しました" ) ;
        }
        // 削除.
        else if( EXEC_DELETE.equals( exec ) ) {
            
            if( connectName == null || connectName.length() <= 0 ) {
                throw new MaachangQException(
                    "対象コネクション名は存在しません" ) ;
            }
            
            connect.removeConnect( connectName ) ;
            
            System.out.println( connectName +
                " コネクション削除に成功しました" ) ;
        }
        
    }
    
    /**
     * 実行条件を解析.
     */
    private final String execType( String[] args ) {
        String execType = AnalysisCommand.getHeader( args,EXEC_HEAD ) ;
        if( execType == null || ( execType = execType.trim().toLowerCase() ).length() <= 0 ) {
            return EXEC_LIST ;
        }
        else if( execType.startsWith( EXEC_LIST ) == true ) {
            return EXEC_LIST ;
        }
        else if( execType.startsWith( EXEC_CREATE ) == true ) {
            return EXEC_CREATE ;
        }
        else if( execType.startsWith( EXEC_SET ) == true ) {
            return EXEC_SET ;
        }
        else if( execType.startsWith( EXEC_DELETE ) == true ) {
            return EXEC_DELETE ;
        }
        else {
            return EXEC_LIST ;
        }
        
    }
    
    /**
     * コネクション名を取得.
     */
    private final String getConnectName( String[] args ) {
        return AnalysisCommand.getParamStringLowerCase( args,CONNECT_NAME ) ;
    }
    
    /**
     * キャッシュ名を取得.
     */
    private final String getCacheName( String[] args ) {
        return AnalysisCommand.getParamStringLowerCase( args,CACHE_NAME ) ;
    }
    
    /**
     * 暗号フラグを取得.
     */
    private final boolean getCb32Flag( String[] args ) {
        return AnalysisCommand.getParamBoolean( args,CB32_FLAG ) ;
    }
    
    /**
     * 暗号ワードを取得.
     */
    private final String[] getCb32Words( String[] args ) {
        return AnalysisCommand.getParamArray( args,CB32_WORDS ) ;
    }
    
    /**
     * リスト表示.
     */
    private final void outputList( boolean mode,AdminConnect connect,String name )
        throws Exception {
        
        ConnectInfo info = connect.getConnect( name ) ;
        if( info != null ) {
            System.out.println( AnalysisCommand.CUT_CODE ) ;
            System.out.println( "name    :" + info.getName() ) ;
            System.out.println( "cache   :" + info.getMqCacheName() ) ;
            System.out.println( "buffer  :" + info.getBuffer() ) ;
            if( info.getBindAddress() != null ) {
                System.out.println( "addr    :" + info.getBindAddress().getHostName() ) ;
            }
            System.out.println( "port    :" + info.getBindPort() ) ;
            System.out.println( "cb32    :" + info.isCb32() ) ;
            if( info.getCb32Word() != null && info.getCb32Word().length > 0 ) {
                StringBuffer buf = new StringBuffer() ;
                int len = info.getCb32Word().length ;
                for( int i = 0 ; i < len ; i ++ ) {
                    buf.append( info.getCb32Word()[ i ] ) ;
                    buf.append( " " ) ;
                }
                System.out.println( "words   :" + buf.toString() ) ;
            }
            
        }
        else if( mode == true ) {
            System.out.println( "not connect." ) ;
        }
    }
    
    /**
     * 一覧名表示.
     */
    private final void outputNames( boolean mode,String[] names ) {
        if( names != null && names.length > 0 ) {
            System.out.println( AnalysisCommand.CUT_CODE ) ;
            int len = names.length ;
            for( int i = 0 ; i < len ; i ++ ) {
                System.out.println( "name    :" + names[ i ] ) ;
            }
        }
        else if( mode == true ) {
            System.out.println( "not cache." ) ;
        }
    }
    
}

