/*
 * @(#)AnalysisCommand.java
 *
 * Copyright (c) 2007 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.command.core ;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;

import org.maachang.queue.access.MaachangQAccessDefine;
import org.maachang.queue.access.MaachangQErrorCode;
import org.maachang.queue.access.MaachangQException;
import org.maachang.queue.access.admin.MaachangQAccessAdminDriver;
import org.maachang.queue.access.admin.MaachangQConnectAdmin;

/**
 * コマンド用解析プログラム.
 *  
 * @version 2007/01/25
 * @author  masahito suzuki
 * @since   MaachangQ-Command 1.00
 */
public class AnalysisCommand {
    
    /**
     * ヘッダ情報.
     */
    public static final String HEAD = "-" ;
    
    /**
     * ユーザパラメータ接頭語.
     */
    public static final String USER_HEAD = "o" ;
    
    /**
     * キューパラメータ接頭語.
     */
    public static final String QUEUE_HEAD = "q" ;
    
    /**
     * キューコネクションパラメータ接頭語.
     */
    public static final String CONNECT_HEAD = "c" ;
    
    /**
     * チャネルパラメータ接頭語.
     */
    public static final String CHANNEL_HEAD = "h" ;
    
    /**
     * ユーザ名区切り条件.
     */
    public static final String USER_CUT = "@" ;
    
    /**
     * コネクション開始区切り.
     */
    public static final String CONNECT_CUT = "/" ;
    
    /**
     * コネクションポート区切り.
     */
    public static final String PORT_CUT = ":" ;
    
    /**
     * CSV区切り文字.
     */
    public static final String CUT_CSV = "," ;
    
    /**
     * キューマネージャ名区切り条件.
     */
    public static final String MANAGER_CUT = "@" ;
    
    /**
     * キュータイプ区切り.
     */
    public static final String QUEUE_TYPE_CUT = ":" ;
    
    /**
     * 受信タイプ.
     */
    public static final String TYPE_RECEIVE = "r" ;
    
    /**
     * 送信タイプ.
     */
    public static final String TYPE_SEND = "s" ;
    
    /**
     * ローカルホストアドレス.
     */
    public static final String LOCAL_HOST = "127.0.0.1" ;
    
    /**
     * 区切り文字.
     */
    public static final String CUT_CODE =
        "--------+-----------------------------------------" ;
    
    /**
     * 指定頭文字の情報を取得.
     * <BR><BR>
     * 指定頭文字の情報を取得します.
     * <BR>
     * @param args 対象のパラメータが設定されます.
     * @param head 対象の頭文字を設定します.
     * @return String 対象の文字列が返されます.
     */
    public static final String getHeader( String[] args,String head ) {
        
        if( head == null || ( head = head.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        if( args == null || args.length <= 0 ) {
            throw new IllegalArgumentException( "パラメータが足りません" ) ;
        }
        
        int len = args.length ;
        int i = 0 ;
        
        head = HEAD+head ;
        
        for( i = 0 ; i < len ; i ++ ) {
            if( args[ i ] != null &&
                args[ i ].toLowerCase().startsWith( head ) == true ) {
                return args[ i ].substring( head.length() ).trim() ;
            }
        }
        
        return null ;
    }
    
    /**
     * ログインして、管理者オブジェクトを取得.
     * <BR><BR>
     * ログインして、管理者オブジェクトを取得します.
     * <BR>
     * @param args 対象のパラメータを設定します.
     * @return MaachangQConnectAdmin 管理者コネクションオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final MaachangQConnectAdmin loginToConnect( String[] args )
        throws Exception {
        return AnalysisCommand.loginToConnect( args,false ) ;
    }
    
    /**
     * ログインして、管理者オブジェクトを取得.
     * <BR><BR>
     * ログインして、管理者オブジェクトを取得します.
     * <BR>
     * @param args 対象のパラメータを設定します.
     * @param output ログイン状態を表示するか設定します.
     * @return MaachangQConnectAdmin 管理者コネクションオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final MaachangQConnectAdmin loginToConnect( String[] args,boolean output )
        throws Exception {
        
        LoginParam param = AnalysisCommand.analysisLogin( args ) ;
        
        if( param == null ) {
            throw new MaachangQException(
                "ログイン条件が設定されていません\n" +
                "      パラメータ[-Ouser@passwd/address:port]で設定してください" ) ;
        }
        
        if( output == true ) {
            System.out.println( "-------+------------------------------------------" ) ;
            System.out.println( "user   :" + param.getUser() ) ;
            System.out.println( "addr   :" + param.getAddress() ) ;
            System.out.println( "port   :" + param.getPort() ) ;
            System.out.println( "-------+------------------------------------------" ) ;
            System.out.println() ;
        }
        
        return MaachangQAccessAdminDriver.connect(
            false,
            param.getUser(),param.getPasswd(),
            InetAddress.getByName( param.getAddress() ),
            param.getPort() ) ;
    }
    
    /**
     * 管理者用ログイン条件を解析.
     * <BR><BR>
     * 管理者用ログイン条件を解析します.
     * <BR>
     * @param args 対象のパラメータが設定されます.
     * @return LoginParam 解析された情報が返されます.
     * @exception Exception 例外.
     */
    public static final LoginParam analysisLogin( String[] args )
        throws Exception {
        
        // -Ouser@passwd/address:port
        
        LoginParam ret = new LoginParam() ;
        int pnt,pnt2 ;
        
        
        String param = AnalysisCommand.getHeader( args,USER_HEAD ) ;
        if( param == null || param.length() <= 0 ) {
            return null ;
        }
        
        pnt = param.indexOf( USER_CUT ) ;
        if( pnt > 0 ) {
            ret.setUser( param.substring( 0,pnt ) ) ;
            pnt2 = param.indexOf( CONNECT_CUT,pnt ) ;
            if( pnt2 <= -1 ) {
                ret.setPasswd( param.substring( pnt+USER_CUT.length(),param.length() ) ) ;
                pnt = -1 ;
            }
            else {
                ret.setPasswd( param.substring( pnt+USER_CUT.length(),pnt2 ) ) ;
                pnt = pnt2 ;
            }
        }
        else {
            pnt = param.indexOf( CONNECT_CUT ) ;
            if( pnt > 0 ) {
                ret.setUser( param.substring( 0,pnt ) ) ;
            }
            else {
                ret.setUser( param ) ;
                return ret ;
            }
        }
        
        if( pnt > 0 ) {
            
            if( ( pnt2 = param.indexOf( PORT_CUT,pnt ) ) > 0 ) {
                ret.setAddress( param.substring( pnt+CONNECT_CUT.length(),pnt2 ) ) ;
                try {
                    ret.setPort( Integer.parseInt(
                        param.substring( pnt2+PORT_CUT.length(),param.length() ) ) ) ;
                } catch( Exception e ) {
                    ret.setPort( -1 ) ;
                }
            }
            else {
                ret.setAddress( param.substring(
                    pnt+CONNECT_CUT.length(),param.length() ) ) ;
            }
            
        }
        
        if( ret.getAddress() == null || ret.getAddress().trim().length() <= 0 ) {
            ret.setAddress( LOCAL_HOST ) ;
        }
        if( ret.getPort() <= 0 ) {
            ret.setPort( MaachangQAccessDefine.ADMIN_PORT ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * キューパラメータを解析.
     * <BR><BR>
     * キューパラメータを解析します.
     * <BR>
     * @param args 対象のパラメータを設定します.
     * @return QueueParam キューパラメータが返されます.
     * @exception Exception 例外.
     */
    public static final QueueParam getQueueParam( String[] args )
        throws Exception {
        
        // -Qmanager@queue:[send/receive]
        
        QueueParam ret = new QueueParam() ;
        int pnt,pnt2 ;
        String type = null ;
        
        String param = AnalysisCommand.getHeader( args,QUEUE_HEAD ) ;
        if( param == null || param.length() <= 0 ) {
            return null ;
        }
        
        pnt =  param.indexOf( MANAGER_CUT ) ;
        if( pnt > 0 ) {
            ret.setManager( param.substring( 0,pnt ) ) ;
            pnt2 = param.indexOf( QUEUE_TYPE_CUT,pnt ) ;
            if( pnt2 <= -1 ) {
                ret.setQueue(
                    param.substring( pnt+MANAGER_CUT.length(),param.length() ) ) ;
            }
            else {
                ret.setQueue(
                    param.substring( pnt+MANAGER_CUT.length(),pnt2 ) ) ;
                type = param.substring(
                    pnt2+QUEUE_TYPE_CUT.length(),param.length() ) ;
            }
        }
        else {
            pnt = param.indexOf( QUEUE_TYPE_CUT ) ;
            if( pnt > 0 ) {
                ret.setManager( param.substring( 0,pnt ) ) ;
            }
            else {
                ret.setManager( param ) ;
            }
        }
        
        if( type != null ) {
            type = type.trim().toLowerCase() ;
            char ch = type.charAt( 0 ) ;
            if( ch == TYPE_RECEIVE.charAt( 0 ) ) {
                ret.setType( MaachangQAccessDefine.TYPE_RECEIVE ) ;
            }
            else {
                ret.setType( MaachangQAccessDefine.TYPE_SEND ) ;
            }
        }
        
        if( ret.getManager() == null ) {
            throw new IllegalArgumentException(
                "キューマネージャ名が存在しません" ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * チャネルパラメータを解析.
     * <BR><BR>
     * チャネルパラメータを解析します.
     * <BR>
     * @param args 対象のパラメータを設定します.
     * @return ChannelParam チャネルパラメータが返されます.
     * @exception Exception 例外.
     */
    public static final ChannelParam getChannelParam( String[] args )
        throws Exception {
        
        // -Hmanager@channel:[send/receive]
        
        ChannelParam ret = new ChannelParam() ;
        int pnt,pnt2 ;
        String type = null ;
        
        String param = AnalysisCommand.getHeader( args,CHANNEL_HEAD ) ;
        if( param == null || param.length() <= 0 ) {
            return null ;
        }
        
        pnt =  param.indexOf( MANAGER_CUT ) ;
        if( pnt > 0 ) {
            ret.setManager( param.substring( 0,pnt ) ) ;
            pnt2 = param.indexOf( QUEUE_TYPE_CUT,pnt ) ;
            if( pnt2 <= -1 ) {
                ret.setChannel(
                    param.substring( pnt+MANAGER_CUT.length(),param.length() ) ) ;
            }
            else {
                ret.setChannel(
                    param.substring( pnt+MANAGER_CUT.length(),pnt2 ) ) ;
                type = param.substring(
                    pnt2+QUEUE_TYPE_CUT.length(),param.length() ) ;
            }
        }
        else {
            pnt = param.indexOf( QUEUE_TYPE_CUT,pnt ) ;
            if( pnt > 0 ) {
                ret.setChannel( param.substring( 0,pnt ) ) ;
                type = param.substring(
                    pnt+QUEUE_TYPE_CUT.length(),param.length() ) ;
            }
            else {
                ret.setChannel( param ) ;
            }
        }
        
        if( type != null ) {
            type = type.trim().toLowerCase() ;
            char ch = type.charAt( 0 ) ;
            if( ch == TYPE_SEND.charAt( 0 ) ) {
                ret.setType( MaachangQAccessDefine.TYPE_SEND ) ;
            }
            else {
                ret.setType( MaachangQAccessDefine.TYPE_RECEIVE ) ;
            }
        }
        else {
            if( ret.getManager() == null ) {
                ret.setType( MaachangQAccessDefine.TYPE_RECEIVE ) ;
            }
            else {
                ret.setType( MaachangQAccessDefine.TYPE_SEND ) ;
            }
        }
        
        return ret ;
        
    }
    
    /**
     * キューコネクション情報を取得.
     * <BR><BR>
     * キューコネクション情報を取得します.
     * <BR>
     * @param args 対象のパラメータが設定されます.
     * @return QueueConnectParam 解析された情報が返されます.
     * @exception Exception 例外.
     */
    public static final ConnectParam analysisConnect( String[] args )
        throws Exception {
        
        // -Caddress:port
        
        ConnectParam ret = new ConnectParam() ;
        int pnt ;
        
        String param = AnalysisCommand.getHeader( args,CONNECT_HEAD ) ;
        if( param == null || param.length() <= 0 ) {
            return null ;
        }
        
        pnt = param.indexOf( PORT_CUT ) ;
        if( pnt > 0 ) {
            ret.setAddress( param.substring( 0,pnt ) ) ;
            try {
                ret.setPort( Integer.parseInt(
                    param.substring( pnt+PORT_CUT.length(),param.length() ) ) ) ;
            } catch( Exception e ) {
                ret.setPort( -1 ) ;
            }
        }
        else {
            ret.setAddress( param ) ;
            ret.setPort( -1 ) ;
        }
        
        if( ret.getAddress() == null || ret.getAddress().trim().length() <= 0 ) {
            ret.setAddress( LOCAL_HOST ) ;
        }
        
        return ret ;
        
    }
    
    /**
     * CSV形式パラメータを解析.
     * <BR><BR>
     * CSV形式パラメータを解析します.
     * <BR>
     * @param csv 解析対象のCVS情報を設定します.
     * @return String[] 解析された文字列が返されます.
     */
    public static final String[] cutCsv( String csv ) {
        
        if( csv == null || ( csv = csv.trim() ).length() <= 0 ) {
            return null ;
        }
        
        ArrayList lst = new ArrayList() ;
        
        int pnt = 0 ;
        int bef = 0 ;
        
        for( ;; ) {
            
            if( ( pnt = csv.indexOf( CUT_CSV,bef ) ) <= -1 ) {
                lst.add( csv.substring( bef,csv.length() ) ) ;
                break ;
            }
            
            lst.add( csv.substring( bef,pnt ) ) ;
            bef = pnt+CUT_CSV.length() ;
            
        }
        
        if( lst.size() > 0 ) {
            int len = lst.size() ;
            String[] ret = new String[ len ] ;
            for( int i = 0 ; i < len ; i ++ ) {
                ret[ i ] = ( String )lst.get( i ) ;
            }
            return ret ;
        }
        
        return null ;
        
    }
    
    /**
     * 指定ファイル名の情報を取得.
     * <BR><BR>
     * 指定ファイル名の情報を取得します.
     * <BR>
     * @param fileName 対象のファイル名を設定します.
     * @return byte[] 対象のバイナリ情報が返されます.
     * @exception IOException IO例外.
     */
    public static final byte[] getFile( String file )
        throws IOException {
        
        long len ;
        
        byte[] ret = null ;
        
        try{
            
            if( file == null || file.trim().length() <= 0 ){
                throw new IllegalArgumentException(
                    "ファイル名は不正です" ) ;
            }
            
            File fp = new File( file ) ;
            len = ( fp.exists() == true ) ? 
                fp.length() : -1L ;
            
            if( len <= -1L ){
                throw new IOException(
                    "対象のファイル名(" + file +
                    ")はファイルが存在しません"
                ) ;
            }
            
            if( len > 0x000000007fffffffL ){
                throw new IOException(
                    "指定されたファイル名(" + file +
                    ")はファイルサイズが大きいため取得できません"
                ) ;
            }
            
            ret = new byte[ ( int )( len & 0x000000007fffffffL ) ] ;
            FileInputStream buf = new FileInputStream( file ) ;
            buf.read( ret ) ;
            buf.close() ;
            
        }catch( IOException io ){
            throw io ;
        }
        
        return ret ;
        
    }
    
    /**
     * バイナリをファイル内容に設定.
     * <BR><BR>
     * バイナリをファイル内容に設定します.
     * <BR>
     * @param file 対象のファイル名を設定します.
     * @param binary 設定対象のバイナリを設定します.
     * @exception IOException IO例外.
     */
    public static final void setFile( String file,byte[] binary )
        throws IOException {
        
        if( file == null || file.trim().length() <= 0 || binary == null ){
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        try{
            
            FileOutputStream buf = new FileOutputStream( file ) ;
            buf.write( binary ) ;
            buf.close() ;
            
        }catch( IOException io ){
            throw io ;
        }
    }
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * @param args 引数群.
     * @param head 対象ヘッダ文字列.
     * @return String 取得された文字列.
     */
    public static final String getParamString( String[] args,String head ) {
        String name = AnalysisCommand.getHeader( args,head ) ;
        if( name == null || ( name = name.trim() ).length() <= 0 ) {
            return null ;
        }
        return name ;
    }
    
    /**
     * パラメータを文字列で取得.
     * <BR><BR>
     * @param args 引数群.
     * @param head 対象ヘッダ文字列.
     * @return String 取得された文字列.
     */
    public static final String getParamStringLowerCase( String[] args,String head ) {
        String name = AnalysisCommand.getHeader( args,head ) ;
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        return name ;
    }
    
    /**
     * パラメータを数値で取得.
     * <BR><BR>
     * @param args 引数群.
     * @param head 対象ヘッダ文字列.
     * @return int 取得された数値.
     */
    public static final int getParamInt( String[] args,String head ) {
        String name = AnalysisCommand.getHeader( args,head ) ;
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return -1 ;
        }
        try {
            int ret = Integer.parseInt( name ) ;
            if( ret <= 0 ) {
                return -1 ;
            }
            else {
                return ret ;
            }
        } catch( Exception e ) {
        }
        
        return -1 ;
    }
    
    /**
     * パラメータをフラグで取得.
     * <BR><BR>
     * @param args 引数群.
     * @param head 対象ヘッダ文字列.
     * @return boolean 取得されたBoolean.
     */
    public static final boolean getParamBoolean( String[] args,String head ) {
        String name = AnalysisCommand.getHeader( args,head ) ;
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return false ;
        }
        
        return ( name.startsWith( "t" ) == true || name.startsWith( "on" ) == true ) ?
            true : false ;
    }
    
    /**
     * パラメータを文字配列で取得.
     * <BR><BR>
     * @param args 引数群.
     * @param head 対象ヘッダ文字列.
     * @return String[] 取得された文字列群.
     */
    public static final String[] getParamArray( String[] args,String head ) {
        String name = AnalysisCommand.getHeader( args,head ) ;
        if( name == null || ( name = name.trim().toLowerCase() ).length() <= 0 ) {
            return null ;
        }
        
        return AnalysisCommand.cutCsv( name ) ;
    }
    
    /**
     * エラーメッセージを表示.
     * <BR><BR>
     * エラーメッセージを表示します.
     * <BR>
     * @param exception 対象の例外オブジェクトを設定します.
     */
    public static final void outputError( Exception exception ) {
        
        if( exception instanceof MaachangQException ) {
            MaachangQException fe = ( MaachangQException )exception ;
            if( fe.getErrorCode() == MaachangQErrorCode.UNKNOWN_ERROR ) {
                System.out.println( "ERROR " + fe.getMessage() ) ;
            }
            else {
                System.out.println( "ERROR " + fe.getErrorCode() +
                    " " + fe.getMessage() ) ;
            }
        }
        else {
            System.out.println( "ERROR " + exception.getMessage() ) ;
        }
        
        //exception.printStackTrace() ;
    }
}

