/*
 * @(#)CommonsBean.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.beans;

import org.maachang.commons.conv.MD5;
import org.maachang.commons.def.BaseDef;
import org.maachang.commons.def.MaachangCommonDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.GetterSetter;
import org.maachang.commons.util.UtilCom;


/**
 * Bean基本処理.
 * 
 * @version 1.00, 2005/10/15
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public class CommonsBean
{
    
    /**
     * デフォルトロード名.
     */
    private static final String DEF_LOAD_CB32 = "def/bean/insert" ;
    
    /**
     * CB32ヘッダ名.
     */
    private static final String CB32_HEADER = "\\\\\\\\@" ;
    /**
     * CB32符合名.
     */
    private static String CB32_PLUS = null ;
    private static final String CB32_PLUS_FIRST = "@[" ;
    private static final String CB32_PLUS_END = "]////" ;
    
    /**
     * CB32デフォルト符号名.
     */
    private static final String CB32DEF_PLUS = "6pbao8c0do050346x2sx1x025617c8ph" ;
    
    /**
     * コンストラクタ.
     */
    private CommonsBean(){}
    
    /**
     * 名前を指定してBeanオブジェクトを生成.
     * <BR><BR>
     * 名前を指定してBeanオブジェクトを生成します.
     * <BR>
     * @param name 生成対象のBeanオブジェクトを生成します.
     * @return BaseBean 生成されたオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean create( String name )
        throws InputException,AccessException
    {
        return CommonsBean.create( name,null ) ;
    }
    
    /**
     * 名前を指定してBeanオブジェクトを生成.
     * <BR><BR>
     * 名前を指定してBeanオブジェクトを生成します.
     * <BR>
     * @param name 生成対象のBeanオブジェクトを生成します.
     * @param loader 対象のクラスローダを設定します.<BR>
     *               デフォルトのローダを利用する場合[null]を設定します.
     * @return BaseBean 生成されたオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean create( String name,ClassLoader loader )
        throws InputException,AccessException
    {
        Object ret = null ;
        ret = UtilCom.createObject( name,loader ) ;
        if( ( ret instanceof BaseBean ) == false ){
            throw new AccessException(
                "指定オブジェクト名(" + name +
                ")はBaseBeanではありません"
            ) ;
        }
        
        return ( BaseBean )ret ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをセーブ.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをセーブします.
     * <BR>
     * @param name セーブ対象のファイル名を設定します.
     * @param bean セーブ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void save( String name,BaseBean bean )
        throws InputException,AccessException
    {
        CommonsBean.save( true,CommonsBean.createCB32( null ),name,bean ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをセーブ.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをセーブします.
     * <BR>
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param name セーブ対象のファイル名を設定します.
     * @param bean セーブ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void save( boolean mode,String name,BaseBean bean )
        throws InputException,AccessException
    {
        CommonsBean.save( mode,CommonsBean.createCB32( null ),name,bean ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをセーブ.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをセーブします.
     * <BR>
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name セーブ対象のファイル名を設定します.
     * @param bean セーブ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void save( String cb32,String name,BaseBean bean )
        throws InputException,AccessException
    {
        CommonsBean.save( true,CommonsBean.createCB32( cb32 ),name,bean ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをセーブ.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをセーブします.
     * <BR>
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name セーブ対象のファイル名を設定します.
     * @param bean セーブ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void save( boolean mode,String cb32,String name,BaseBean bean )
        throws InputException,AccessException
    {
        SerializeUtil.putCb32Serialize( mode,CommonsBean.createCB32( cb32 ),name,bean ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをロード.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをロードします.
     * <BR>
     * @param name ロード対象のファイル名を設定します.
     * @return BaseBean ロードされたBaseBeanオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean load( String name )
        throws InputException,AccessException
    {
        return CommonsBean.load( true,CommonsBean.createCB32( null ),name ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをロード.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをロードします.
     * <BR>
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param name ロード対象のファイル名を設定します.
     * @return BaseBean ロードされたBaseBeanオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean load( boolean mode,String name )
        throws InputException,AccessException
    {
        return CommonsBean.load( mode,CommonsBean.createCB32( null ),name ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをロード.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをロードします.
     * <BR>
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name ロード対象のファイル名を設定します.
     * @return BaseBean ロードされたBaseBeanオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean load( String cb32,String name )
        throws InputException,AccessException
    {
        return CommonsBean.load( true,CommonsBean.createCB32( cb32 ),name ) ;
    }
    
    /**
     * 指定ファイル名でBeanオブジェクトをロード.
     * <BR><BR>
     * 指定したファイル名でBeanオブジェクトをロードします.
     * <BR>
     * @param mode 暗号モードを設定します.<BR>
     *             [true]を設定した場合、現在ディレクトリ名を暗号コードの一部にします.<BR>
     *             [false]を設定した場合、通常の条件で暗号化させます.
     * @param cb32 暗号基本値を設定します.<BR>
     *             [null]を設定した場合、デフォルト条件となります.
     * @param name ロード対象のファイル名を設定します.
     * @return BaseBean ロードされたBaseBeanオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final BaseBean load( boolean mode,String cb32,String name )
        throws InputException,AccessException
    {
        Object ret = null ;
        
        ret = SerializeUtil.getCb32Serialize( mode,CommonsBean.createCB32( cb32 ),name ) ;
        if( ( ret instanceof BaseBean ) == false ){
            throw new AccessException(
                "ロード対象のオブジェクト(" + name +
                ")はBaseBeanではありません"
            ) ;
        }
        
        return ( BaseBean )ret ;
    }
    
    /**
     * 指定Beanのフィールド内容を取得.
     * <BR><BR>
     * 指定Beanのフィールド内容を取得します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param in 取得対象のオブジェクト情報を設定します.
     * @param names 取得対象のフィールド名を設定します.
     * @return Object[] 取得されたフィールド情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Object[] getter( BaseBean in,String[] names )
        throws InputException,AccessException
    {
        return GetterSetter.getter( in,names ) ;
    }
    
    /**
     * Beanのフィールドに対してオブジェクトを挿入.
     * <BR><BR>
     * Beanのフィールドに対してオブジェクトを挿入します.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param out 挿入対象のBeanオブジェクトを設定します.
     * @param names 追加対象のフィールド名を設定します.<BR>
     *              また、この条件は引数[objs]に紐付けされます.
     * @param objs 追加対象のオブジェクト群を設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void setter( BaseBean out,String[] names,Object[] objs )
        throws InputException,AccessException
    {
        GetterSetter.setter( out,names,objs ) ;
    }
    
    /**
     * Beanフィールドをコピー.
     * <BR><BR>
     * Beanフィールドで一致する内容をコピーします.<BR>
     * また、その時のフィールド名に対するGetterメソッドは
     * protected,privateの場合、エラーとなります.
     * <BR>
     * @param src コピー元のBeanオブジェクトを設定します.
     * @param dest コピー先のBeanオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void copy( BaseBean src,BaseBean dest )
        throws InputException,AccessException
    {
        GetterSetter.copy( src,dest ) ;
    }
    
    
    
    /**
     * CB32名を生成.
     */
    private static final String createCB32( String name )
    {
        String ret = null ;
        StringBuffer buf = null ;
        
        if( CB32_PLUS == null ){
            buf = new StringBuffer() ;
            buf.append( CB32_PLUS_FIRST ) ;
            buf.append( CommonsBean.createCB32_PLUS() ) ;
            buf.append( CB32_PLUS_END ) ;
            CB32_PLUS = buf.toString() ;
            buf = null ;
        }
        
        buf = new StringBuffer() ;
        buf.append( CB32_HEADER ) ;
        
        if( name == null || name.length() <= 0 ){
            buf.append( DEF_LOAD_CB32 ) ;
        }
        else{
            buf.append( name ) ;
        }
        
        buf.append( CB32_PLUS ) ;
        ret = buf.toString() ;
        buf = null ;
        
        return ret ;
    }
    
    /**
     * CB32_PLUSを生成.
     */
    private static final String createCB32_PLUS()
    {
        StringBuffer buf = null ;
        String ret = null ;
        
        buf = new StringBuffer() ;
        
        buf.append( BaseDef.FILE_SEPARATOR ) ;
        buf.append( BaseDef.PATH_SEPARATOR ) ;
        buf.append( BaseDef.ENTER_SEPARATOR ) ;
        buf.append( BaseDef.THIS_CHARSET ) ;
        buf.append( "$" ) ;
        buf.append( MaachangCommonDef.getInstance().getName() ) ;
        
        try{
            ret = MD5.convert( buf.toString().getBytes( BaseDef.UTF8 ) ) ;
        }catch( Exception e ){
            ret = CB32DEF_PLUS ;
        }finally{
            buf = null ;
        }
        
        return ret ;
    }
}

