/*
 * @(#)NetAgentElement.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.net;

import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.thread.Synchronized;


/**
 * ネットワークエージェント要素.
 * <BR><BR>
 * 対象ネットワークのオープン/クローズ処理をサポートする
 * エージェント要素です.
 *
 * @version 1.00, 2004/10/12
 * @author  Masahito Suzuki
 * @since   JRcCommons 1.00
 */
public abstract class NetAgentElement
{
    
    /**
     * 要素有効フラグ.
     */
    protected boolean m_useFlg = false ;
    
    /**
     * 要素ダウンフラグ.
     */
    protected boolean m_downFlg = true ;
    
    /**
     * 媒体名.
     */
    private String m_adapter = null ;
    
    /**
     * ローカルアドレス.
     */
    protected String m_localAddress = null ;
    
    /**
     * 同期オブジェクト.
     */
    protected final Synchronized m_sync = new Synchronized() ;
    
    /**
     * コンストラクタ.
     */
    public NetAgentElement()
    {
        
    }
    
    /**
     * ネットワーク開始処理.
     * <BR><BR>
     * ネットワーク処理を開始します.
     */
    protected void startAgent()
    {
        this.exitAgent() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync.get() ){
                m_useFlg = true ;
            }
        }catch( Exception t ){
            this.exitAgent() ;
        }
    }
    
    /**
     * ネットワーク処理を終了.
     * <BR><BR>
     * ネットワーク処理を終了します.
     */
    protected void exitAgent()
    {
        try{
            this.down() ;
        }catch( Exception t ){
        }
        
        m_sync.clear() ;
        m_useFlg = false ;
        m_downFlg = true ;
        m_adapter = null ;
        m_localAddress = null ;
        
    }
    
    /**
     * 対象アダプタ名を設定.
     * <BR><BR>
     * 対象アダプタ名を設定します.
     * <BR>
     * @param name 対象アダプタ名を設定します.
     * @exception InputException 入力例外.
     */
    protected void setAdapter( String name )
        throws InputException
    {
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            synchronized( m_sync.get() ){
                m_adapter = name ;
            }
        }catch( Exception t ){
            m_adapter = null ;
        }
    }
    
    /**
     * ネットワーク復帰時の処理.
     * <BR><BR>
     * コンピュータのネットワークが復帰した場合の呼び出し処理です.<BR>
     * また、ネットワークオープン処理はこのメソッドに実装する必要があります.
     * <BR>
     * @param addr 対象のローカルアドレスを設定します.
     * @exception ExecutionException 実行例外.
     */
    protected void restore( String addr ) throws ExecutionException
    {
        try{
            synchronized( m_sync.get() ){
                
                m_downFlg = false ;
                m_localAddress = addr ;
                
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * ネットワークダウン時の処理.
     * <BR><BR>
     * コンピュータのネットワークがダウンした場合の呼び出し処理です.<BR>
     * また、ネットワーククローズ処理はこのメソッドに実装する必要があります.
     * <BR>
     * @exception ExecutionException 実行例外.
     */
    protected void down() throws ExecutionException
    {
        try{
            synchronized( m_sync.get() ){
                
                m_downFlg = true ;
                
                // ネットワークインターフェイスがサポートされている場合.
                if( NetConfig.getInstance().useObject() == true ){
                    
                    // 対象IPアドレスをクリア.
                    m_localAddress = null ;
                    
                }
                
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * オープンローカルアドレス取得.
     * <BR><BR>
     * 対象のオープンローカルアドレスを取得します.
     * <BR>
     * @return String 格納されているローカルアドレスが返されます.
     */
    protected String getLocalAddress()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_localAddress ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象アダプタ名を取得.
     * <BR><BR>
     * 設定されている対象アダプタ名を取得します.
     * <BR>
     * @return String 設定されている対象アダプタ名が返されます.
     */
    public String getAdapter()
    {
        String ret = null ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_adapter ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象要素有効チェック.
     * <BR><BR>
     * 対象要素が有効であるかチェックします.<BR>
     * この有効チェックは管理対象のネットワークエージェントでの
     * 要素削除条件として利用されます.<BR>
     * また、この条件を操作するためのメソッドは[NetAgentElement#startNetwork()]と
     * [NetAgentElement#exitNetwork()]です.
     * <BR>
     * @return boolean 有効チェックが返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    protected boolean isUse()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_useFlg ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * ダウンフラグを取得.
     * <BR><BR>
     * 現在ネットワークがダウンしているかチェックします.
     * <BR>
     * @return boolean ダウンチェック結果が返されます.<BR>
     *                 [true]が返された場合、ダウンしています.<BR>
     *                 [false]が返された場合、ダウンしていません.
     */
    protected boolean isDown()
    {
        boolean ret ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_downFlg ;
            }
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
}

