/*
 * @(#)SectorIO.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.resource.cache;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.IdManager;


/**
 * セクターI/O.
 *  
 * @version 2006/08/04
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class SectorIO {
    
    /**
     * １セクタに対するバイト数.
     */
    public static final long SECTOR_LENGTH = ( long )CacheDef.SECTOR_LENGTH ;
    
    /**
     * セクタシーケンスID管理.
     */
    private IdManager sequenceIDManage = null ;
    
    /**
     * セクタID管理.
     */
    private SectorIDManage idManage = null ;
    
    /**
     * セクタデータ数管理.
     */
    private SectorSizeManage sizeManage = null ;
    
    /**
     * セクタデータ管理.
     */
    private SectorDataFile dataFileManage = null ;
    
    /**
     * セクタI/O固有ID.
     */
    private long uniqueID = -1L ;
    
    
    
    /**
     * コンストラクタ.
     */
    private SectorIO() {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * オープンファイル名とセクタ数を設定して、セクタI/Oを生成します.
     * <BR>
     * @param name セクタデータを保持するファイル名を設定します.
     * @param size セクタサイズを設定します.<BR>
     *             設定可能な最小値は[CacheDef.MIN_SECTOR]です.<BR>
     *             設定可能な最大値は[CacheDef.MAX_SECTOR]です.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public SectorIO( String name,int size )
        throws InputException,AccessException {
        
        if (
            name == null || name.length() <= 0 || 
            size <= 0
        ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        try {
            
            idManage = new SectorIDManage( size ) ;
            size = idManage.getMaxSector() ;
            sizeManage = new SectorSizeManage( size ) ;
            dataFileManage = new SectorDataFile( name,size ) ;
            sequenceIDManage = new IdManager( 0,size ) ;
            
            uniqueID = SectorIO.createUniqueID(
                dataFileManage.getFileName(),
                idManage.getMaxSector()
            ) ;
            
        } catch( AccessException ae ) {
            this.destroy() ;
            throw ae ;
        } catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception {
        
        try {
            this.destroy() ;
        } catch ( Exception t ) {
            
        }
    }
    
    /**
    /**
     * オブジェクト破棄.
     * <BR><BR>
     * オブジェクトを破棄します.
     */
    public synchronized void destroy() {
        
        if ( idManage != null ) {
            try {
                idManage.destroy() ;
            } catch( Exception e ) {
            }
        }
        
        if( sizeManage != null ) {
            try {
                sizeManage.destroy() ;
            } catch( Exception e ) {
            }
        }
        
        if ( dataFileManage != null ) {
            try {
                dataFileManage.destroy() ;
            } catch( Exception e ) {
            }
        }
        
        if ( sequenceIDManage != null ) {
            try {
                sequenceIDManage.clear() ;
            } catch( Exception e ) {
            }
        }
        
        idManage = null ;
        sizeManage = null ;
        dataFileManage = null ;
        sequenceIDManage = null ;
        uniqueID = -1L ;
        
    }
    
    /**
     * 利用IDを予約.
     * <BR><BR>
     * 新しい利用IDを予約します.<BR>
     * また利用IDを破棄したい場合、メソッド[releaseID()]を利用します.
     * <BR>
     * @return int 新しい予約IDが返されます.<BR>
     *             [-1]が返された場合、利用できる利用IDは存在しません.
     */
    public synchronized int reservationID() {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            ret = sequenceIDManage.getID() ;
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対する予約セクタを設定.
     * <BR><BR>
     * 利用IDに対する予約セクタを設定します.
     * <BR>
     * @param id 予約対象の利用IDを設定します.
     * @param size 予約セクタ数を設定します.
     * @return int 予約されたセクタ数が返されます.<BR>
     *             [0]が返された場合、空きセクタは存在しません.
     * @exception InputException 入力例外.
     */
    public synchronized int reservationSector( int id,int size )
        throws InputException {
        
        int ret = 0 ;
        
        if ( id < 0 || size <= 0 ) {
            throw new InputException( "引数は不正です" ) ;
        }
        
        if ( sequenceIDManage != null ) {
            
            if ( sequenceIDManage.isUseID( id ) == true ) {
                
                if( idManage.getReservationSectorSize( id ) <= 0 ) {
                    sizeManage.set( id,0 ) ;
                }
                
                try {
                    ret = idManage.searchByReservationSector( id,size ) ;
                } catch( Exception e ) {
                    ret = 0 ;
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDを解放.
     * <BR><BR>
     * 利用IDを解放します.<BR>
     * メソッド[reservationID()]で予約した利用IDは、使い終わった場合
     * このメソッドで解放する必要があります.
     * <BR>
     * @param id 解放対象の利用IDを設定します.
     */
    public synchronized void releaseID( int id ) {
        
        if ( id >= 0 || sequenceIDManage != null ) {
            
            try {
                idManage.releaseUseIDByAllSector( id ) ;
            } catch( Exception e ) {
            }
            
            try {
                sizeManage.set( id,0 ) ;
            } catch( Exception e ) {
            }
            
            try {
                sequenceIDManage.removeID( id ) ;
            } catch( Exception e ) {
            }
            
        }
        
    }
    
    /**
     * 利用IDに対する予約セクタ内容に情報読み込み.
     * <BR><BR>
     * 利用IDに対する予約セクタ内容に対して情報を読み込みます.
     * <BR>
     * @param out 読み込み対象のバイナリを設定します.
     * @param id 利用IDを設定します.
     * @param no 利用IDに対する予約セクタ項番を設定します.
     * @param sectorOffset セクタ位置からの開始位置を設定します.
     * @param offset 予約セクタ項番から読み込むオフセット値を設定します.
     * @param length 読み込みデータ長が返されます.
     * @return int 読み込まれたデータ長が返されます.<BR>
     *             [-1]が返された場合、情報は存在しません.
     * @exception AccessException アクセス例外.
     */
    public synchronized int readData( byte[] out,int id,int no,int sectorOffset,int offset,int length )
        throws AccessException {
        
        int sectorPoint ;
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            sectorPoint = idManage.getReservationOneSector( id,no ) ;
            
            if ( sectorPoint != -1 ) {
                
                if ( SectorIDManage.SECTOR_LENGTH < sectorOffset + length ) {
                    length = SectorIDManage.SECTOR_LENGTH - sectorOffset ;
                }
                
                try {
                    
                    ret = dataFileManage.read(
                        out,sectorPoint,sectorOffset,offset,length ) ;
                    
                } catch( AccessException ae ) {
                    throw ae ;
                } catch( Exception e ) {
                    throw new AccessException( e ) ;
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対する予約セクタ内容に情報書き込み.
     * <BR><BR>
     * 利用IDに対する予約セクタ内容に対して情報を書き込みます.
     * <BR>
     * @param in 書き込み対象のバイナリを設定します.
     * @param id 利用IDを設定します.
     * @param no 利用IDに対する予約セクタ項番を設定します.
     * @param sectorOffset セクタ位置からの開始位置を設定します.
     * @param offset 予約セクタ項番から書き込むオフセット値を設定します.
     * @param length 書き込みデータ長が返されます.
     * @return int 書き込まれた情報長が返されます.
     * @exception AccessException アクセス例外.
     */
    public synchronized int writeData( byte[] in,int id,int no,int sectorOffset,int offset,int length )
        throws AccessException {
        
        int sectorPoint ;
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            sectorPoint = idManage.getReservationOneSector( id,no ) ;
            
            if ( sectorPoint != -1 ) {
                
                if ( SectorIDManage.SECTOR_LENGTH < sectorOffset + length ) {
                    length = SectorIDManage.SECTOR_LENGTH - sectorOffset ;
                }
                
                try {
                    
                    dataFileManage.write(
                        in,sectorPoint,sectorOffset,offset,length ) ;
                    ret = length ;
                    
                } catch( AccessException ae ) {
                    throw ae ;
                } catch( Exception e ) {
                    throw new AccessException( e ) ;
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対する指定予約セクタ位置を取得.
     * <BR><BR>
     * 利用IDに対する指定予約セクタ位置を取得します.
     * <BR>
     * @param id 利用IDを設定します.
     * @param no 利用IDに対する予約セクタ項番を設定します.
     * @return int 予約セクタ位置が返されます.<BR>
     *             [-1]が返された場合、情報は無効です.
     */
    public synchronized int getReservationSectorPoint( int id,int no ) {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            ret = idManage.getReservationOneSector( id,no ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対する予約セクタ数を取得.
     * <BR><BR>
     * 利用IDに対する予約セクタ数を取得します.
     * <BR>
     * @param id 対象の利用IDを設定します.
     * @return int 予約セクタ数が返されます.<BR>
     *             [-1]が返された場合、利用IDでは予約されていません.
     */
    public synchronized int getReservationSectorSize( int id ) {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            ret = idManage.getReservationSectorSize( id ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用IDに対してデータ長を設定.
     * <BR><BR>
     * 利用IDに対してデータ長を設定します.
     * <BR>
     * @param id 対象の利用IDを設定します.
     * @param size 対象のデータ長を設定します.
     */
    public synchronized void setUseIDByDataSize( int id,int size ) {
        
        if( sizeManage != null ) {
            
            sizeManage.set( id,size ) ;
            
        }
        
    }
    
    /**
     * 利用IDに対するデータ長を取得.
     * <BR><BR>
     * 利用IDに対するデータ長を取得します.
     * <BR>
     * @param useID 対象の利用IDを設定します.
     * @return int データ長が返されます.
     */
    public synchronized int getUseIDByDataSize( int id ) {
        
        int ret = 0 ;
        
        if( sizeManage != null ) {
            
            ret = sizeManage.get( id ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 現在利用されている利用ID数を取得.
     * <BR><BR>
     * 現在利用されている利用ID数を取得します.
     * <BR>
     * @return int 現在利用されている利用IDが返されます.
     */
    public synchronized int getIDLength() {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            ret = sequenceIDManage.getUseIDSize() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * ファイル名を取得.
     * <BR><BR>
     * 現在オープンされているファイル名を取得します.
     * <BR>
     * @return String オープンされているファイル名が返されます.
     */
    public synchronized String getFileName() {
        
        String ret = null ;
        
        if ( sequenceIDManage != null ) {
            
            ret = dataFileManage.getFileName() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 最大セクタ数を取得.
     * <BR><BR>
     * 最大セクタ数を取得します.
     * <BR>
     * @return int 最大セクタ数が返されます.<BR>
     *             [-1]が返された場合、オブジェクトは無効です.
     */
    public synchronized int getMaxSector() {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            ret = idManage.getMaxSector() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 利用セクタ数を取得.
     * <BR><BR>
     * 現在利用されているセクタ数を取得します.
     * <BR>
     * @return int 現在利用されているセクタ数が返されます.
     */
    public synchronized int getUseSector() {
        
        int ret = -1 ;
        
        if ( sequenceIDManage != null ) {
            
            ret = idManage.getUseSector() ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * ユニークIDを設定.
     * <BR><BR>
     * ユニークIDを再設定します.
     * <BR>
     * @param id 再設定対象のユニークIDを設定します.
     */
    public synchronized void setUniqueID( long id ) {
        
        uniqueID = id ;
        
    }
    
    /**
     * ユニークIDを取得.
     * <BR><BR>
     * 設定されているユニークIDを取得します.
     * <BR>
     * @return int ユニークIDが返されます.
     */
    public synchronized long getUniqueID() {
        
        long ret = -1L ;
        
        if ( sequenceIDManage != null ) {
            
            ret = uniqueID ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * ユニークIDを生成.
     * <BR><BR>
     * ユニークIDを生成します.
     * <BR>
     * @param fileName 対象のファイル名を設定します.
     * @param size 対象のキャッシュサイズを設定します.
     * @return long ユニークIDが返されます.
     */
    public static final long createUniqueID( String fileName,int size ) {
        
        return SectorIO.getHash( fileName ) + ( long )( ( long )size * 31L ) ;
        
    }
    
    /**
     * 利用IDが有効であるかチェック.
     * <BR><BR>
     * 指定された利用IDが有効であるかチェックします.
     * <BR>
     * @param id チェック対象の利用IDを設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、指定した利用IDは存在します.<BR>
     *                 [false]が返された場合、指定した利用IDは存在しません.
     */
    public synchronized boolean isUseId( int id ) {
        
        boolean ret = false ;
        
        if ( sequenceIDManage != null ) {
            
            ret = sequenceIDManage.isUseID( id ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * このオブジェクトが有効であるかチェック.
     * <BR><BR>
     * このオブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、オブジェクトは有効です.<BR>
     *                 [false]が返された場合、オブジェクトは無効です.
     */
    public synchronized boolean isUseObject() {
        
        return ( sequenceIDManage != null ) ? true : false ;
        
    }
    
    
    /**
     * 対象名に対するHash計算.
     */
    private static final long getHash( String name ) {
        
        int i ;
        int len ;
        long ret = 0 ;
        
        len = name.length() ;
        for( i = 0 ; i < len ; i ++ ) {
            
            ret = ( ( 31L * ret ) +
                ( long )( name.charAt( i ) & 0x0000ffff ) ) &
                    0x3fffffffffffffffL ;
            
        }
        
        return ret ;
    }
        
}

