/*
 * @(#)Cache.java
 *
 * Copyright (c) 2006 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.sys ;

import org.maachang.commons.beans.BaseBean;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.util.CharTableSync;


/**
 * ネーミングマネージャ.
 *  
 * @version 2006/08/29
 * @author  masahito suzuki
 * @since   JRcCommons 1.00
 */
public class NamingManager implements Initializer {
    
    /**
     * キャッシュオブジェクト管理名.
     */
    public static final String MANAGER_NAME = "naming@manager" ;
    
    /**
     * 同期オブジェクト.
     */
    private static final Object SYNC = new Object() ;
    
    /**
     * コンストラクタ.
     */
    public NamingManager() {
        
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.<BR>
     * また、このメソッドの場合、デフォルト値での初期化処理になります.
     * <BR>
     * @exception AccessException アクセス例外.
     */
    public synchronized void init() throws AccessException {
        this.init( null ) ;
    }
    
    /**
     * 初期処理.
     * <BR><BR>
     * 初期処理を実施します.<BR>
     * この処理はこのオブジェクトを利用する場合、１度呼び出す必要があります.<BR>
     * また、１度呼び出した場合２度目に呼び出しても効果がありません.<BR>
     * しかし、一度オブジェクトを破棄 Initializer.destroy() した場合、
     * 再び呼び出す事が可能となります.
     * <BR>
     * @param bean 設定値が設定されているBeanオブジェクトを設定します.
     * @exception AccessException アクセス例外.
     */
    public synchronized void init( BaseBean bean ) throws AccessException {
        
        CharTableSync table = null ;
        
        if( Manager.getInstance().isData( MANAGER_NAME ) == true ) {
            throw new AccessException( NamingManager.class.getName() + " は既に存在します" ) ;
        }
        
        try {
            table = new CharTableSync() ;
            Manager.getInstance().put( MANAGER_NAME,table ) ;
        }catch( Exception e ) {
            this.destroy() ;
            throw new AccessException( e ) ;
        }
        
    }
    
    /**
     * 終了処理.
     * <BR><BR>
     * 終了処理を実施します.<BR>
     * また、この処理を実施した場合、再び Initializer.init() を
     * 呼び出さないとオブジェクトの利用が出来なくなります.
     */
    public synchronized void destroy() {
        
        CharTableSync table = null ;
        
        if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
            return ;
        }
        
        table.clear() ;
        table = null ;
        
    }
    
    /**
     * ネーミングマネージャに登録.
     * <BR><BR>
     * ネーミングマネージャに登録します.
     * <BR>
     * @param name 対象のネーミング名を設定します.
     * @param value 対象のオブジェクトを設定します.
     */
    public static final void add( String name,Object value ) {
        
        CharTableSync table = null ;
        
        if( name == null || name.length() <= 0 || value == null ) {
            return ;
        }
        
        synchronized( SYNC ) {
            if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return ;
            }
            
            try {
                table.add( name,value ) ;
            } catch( Exception e ) {
            }
        }
        
    }
    
    /**
     * ネーミングマネージャから削除.
     * <BR><BR>
     * ネーミングマネージャから削除します.
     * <BR>
     * @param name 削除対象の名前を設定します.
     * @return Object 削除された情報が返されます.
     */
    public static final Object remove( String name ) {
        
        CharTableSync table = null ;
        
        if( name == null || name.length() <= 0 ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return null ;
            }
            
            try {
                return table.remove( name ) ;
            } catch( Exception e ) {
            }
        }
        
        return null ;
        
    }
    
    /**
     * キャッシュオブジェクトを取得.
     * <BR><BR>
     * キャッシュオブジェクトを取得します.
     * <BR>
     * @param uniqueID 対象のユニークIDを設定します.
     * @return CacheIO 対象のキャッシュI/Oオブジェクトが返されます.
     */
    public static final Object get( String name ) {
        
        CharTableSync table = null ;
        
        if( name == null || name.length() <= 0 ) {
            return null ;
        }
        
        synchronized( SYNC ) {
            if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return null ;
            }
            
            return table.get( name ) ;
        }
        
    }
    
    /**
     * ネーミングマネージャに管理されているオブジェクト群を取得.
     * <BR><BR>
     * ネーミングマネージャに管理されているオブジェクト群を取得します.
     * <BR>
     * @return int ネーミングマネージャに管理されているオブジェクト群が返されます.
     */
    public static final int size() {
        
        CharTableSync table = null ;
        
        synchronized( SYNC ) {
            if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return 0 ;
            }
            
            return table.size() ;
        }
        
    }
    
    /**
     * ネーミングマネージャに管理されている名前一覧を取得.
     * <BR><BR>
     * ネーミングマネージャに管理されている名前一覧を取得します.
     * <BR>
     * @return String[] ネーミングマネージャに管理されている名前一覧が返されます.
     */
    public static final String[] getNames() {
        
        CharTableSync table = null ;
        
        synchronized( SYNC ) {
            if( ( table = ( CharTableSync )Manager.getInstance().get( MANAGER_NAME ) ) == null ) {
                return null ;
            }
            
            return table.getNames() ;
        }
        
    }
    
    /**
     * 管理オブジェクトが有効であるかチェック.
     * <BR><BR>
     * 管理オブジェクトが有効であるかチェックします.
     * <BR>
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、有効です.<BR>
     *                 [false]が返された場合、無効です.
     */
    public static final boolean isManager() {
        
        synchronized( SYNC ) {
            return Manager.getInstance().isData( MANAGER_NAME ) ;
        }
        
    }
    
}

