/*
 * @(#)Synchronized.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.thread;

import org.maachang.commons.serialize.InitSerialize;
import org.maachang.commons.serialize.SerializeUtil;

/**
 * 同期用オブジェクト.
 * <BR><BR>
 * 同期用オブジェクトを実施します.
 *
 * @version 1.0.0 2003/04/15
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class Synchronized implements InitSerialize
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            Synchronized.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * デフォルトタイムアウト値.
     */
    private static final int DEF_TIMEOUT = 60000 ;
    
    
    
    /**
     * 同期用管理オブジェクト.
     */
    private Boolean m_sync = null ;
    
    /**
     * ロックタイムアウト値.
     */
    private int m_lockTimeout = 0 ;
    
    /**
     * ロックカウント.
     */
    private transient volatile int m_count = 0 ;
    
    /**
     * ロック開始時間.
     */
    private transient volatile long m_lockStartTime = -1L ;
    
    /**
     * ロック対象スレッド名.
     */
    private transient volatile String m_lockThreadName = null ;
    
    /**
     * コンストラクタ.
     */
    public Synchronized()
    {
        m_lockTimeout = Synchronized.DEF_TIMEOUT ;
        this.create() ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * ロックタイムアウトを指定して生成します.
     * <BR>
     * @param timeout 対象のロックタイムアウト値を設定します.
     */
    public Synchronized( int timeout )
    {
        m_lockTimeout = ( timeout <= 0 ) ? Synchronized.DEF_TIMEOUT : timeout ;
        this.create() ;
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行うメソッドです.<BR>
     * 基本的には、このインターフェイスを継承したオブジェクトは、
     * 初期化に必要な処理を実装することで、
     * [com.JRcServer.serialize.SerializeCom.getSerialize()]からの、
     * オブジェクトロードの際に、このメソッドを呼び出してくれるので、
     * ロード後のオブジェクトを、円滑に利用する事が出来ます.
     */
    public void initSerializable()
    {
        this.create() ;
    }
    
    /**
     * 同期オブジェクトの実施.
     * <BR><BR>
     * 同期用に利用するオブジェクトを利用可能にします.
     */
    public synchronized final void create()
    {
        if( m_sync != null ){
            this.clear() ;
        }
        
        m_sync = new Boolean( true ) ;
        m_count = 0 ;
        m_lockThreadName = null ;
        m_lockStartTime = -1 ;
    }
    
    /**
     * 同期用オブジェクトをクリア.
     * <BR><BR>
     * 同期用オブジェクトをクリアします.
     */
    public synchronized final void clear()
    {
        m_sync = null ;
        m_count = 0 ;
        m_lockThreadName = null ;
        m_lockStartTime = -1 ;
    }
    
    /**
     * 同期用オブジェクトを取得.
     * <BR><BR>
     * 管理されている同期用オブジェクトを取得します.
     * <BR>
     * @return Object 同期用オブジェクトが返されます.<BR>
     *                既にクリアされている場合[NULL]が返されます.
     */
    public synchronized final Object get()
    {
        return m_sync ;
    }
    
    /**
     * ロックタイムアウト値を取得.
     * <BR><BR>
     * ロックタイムアウト値を取得します.
     * <BR>
     * @return int ロックタイムアウト値が返されます.
     */
    public int getTimeout()
    {
        return m_lockTimeout ;
    }
    
    /**
     * ロック開始時間を取得.
     * <BR><BR>
     * ロックが開始された時間が返されます.
     * <BR>
     * @return long ロックタイムアウト検知値が返されます.<BR>
     *              [-1L]が返された場合ロックされていません.
     */
    public synchronized final long getStartLogTime()
    {
        return m_lockStartTime ;
    }
    
    /**
     * ロック対象スレッド名を取得.
     * <BR><BR>
     * ロック対象スレッド名が返されます.
     * <BR>
     * @return String ロック対象スレッド名が返されます.<BR>
     *                [null]が返された場合ロックされていません.
     */
    public synchronized final String getLockByThreadName()
    {
        return m_lockThreadName ;
    }
    
    /**
     * 同期用オブジェクト利用有無.
     * <BR><BR>
     * 同期用オブジェクトの利用可能／不可能を判別します.
     * <BR>
     * @return boolean 同期オブジェクトの利用可能／不可能判別内容が返されます.<BR>
     *                 [true]が返された場合、利用可能です.
     *                 [false]が返された場合、利用不可能です.
     */
    public synchronized final boolean isUse()
    {
        return ( m_sync == null ) ? false : true ;
    }
    
    
    
    /**
     * ロックカウントインクリメント.
     * <BR><BR>
     * ロックカウントを１インクリメントします.
     * <BR>
     * @return boolean ロックカウントが正常に１インクリメントされたか
     *                 返されます.<BR>
     *                 [true]が返された場合、正常にロックカウントはあがりました.<BR>
     *                 [false]が返された場合、ロックカウントのインクリメントに失敗しました.
     */
    protected synchronized final boolean increment()
    {
        boolean ret ;
        String tname = null ;
        
        tname = Thread.currentThread().getName() ;
        
        if( m_lockThreadName == null || m_lockThreadName.equals( tname ) == true ){
            
            m_lockStartTime = System.currentTimeMillis() ;
            
            m_lockThreadName = tname ;
            m_count ++ ;
            
            ret = true ;
            
        }
        else{
            ret = false ;
        }
        
        tname = null ;
        
        return ret ;
    }
    
    /**
     * ロックカウントデクリメント.
     * <BR><BR>
     * ロックカウントを１デクリメントします.
     * <BR>
     * @return boolean ロック状態が返されます.
     *                 [true]が返された場合、ロック状態はアンロックされました.<BR>
     *                 [false]が返された場合、ロック状態はロック中です.
     */
    protected synchronized final boolean decrement()
    {
        boolean ret = false ;
        
        if(
            m_lockThreadName != null &&
            m_lockThreadName.equals( Thread.currentThread().getName() ) == true
        )
        {
            
            m_count -- ;
            
            if( m_count <= 0 ){
                
                m_lockStartTime = -1L ;
                m_lockThreadName = null ;
                m_count = 0 ;
                
                ret = true ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * ロックカウントを取得.
     * <BR><BR>
     * 現在のロックカウントを取得します.
     * <BR>
     * @return int 現在のロックカウントが返されます.
     */
    protected synchronized final int lockCount()
    {
        return m_count ;
    }
    
}

