/*
 * @(#)BigSearchArray.java
 *
 * Copyright (c) 2003 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.array.LongArray;


/**
 * BigSearchArrayオブジェクト.
 * <BR><BR>
 * 設定順番を保持した検索対応の配列オブジェクトです.
 *
 * @version 1.00, 2004/12/25
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class BigSearchArray implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            BigSearchArray.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * データ管理オブジェクト.
     */
    private final LongArray m_array = new LongArray() ;
    
    /**
     * 検索対象用.
     */
    private final SearchLong m_search = new SearchLong() ;
    
    /**
     * コンストラクタ.
     */
    public BigSearchArray()
    {
        try{
            m_search.create() ;
        }catch( Exception e ){
            this.clear() ;
        }
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            m_array.clear() ;
            m_search.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_array.clear() ;
        m_search.clear() ;
        
        try{
            m_search.create() ;
        }catch( Exception e ){
            m_search.clear() ;
        }
        
    }
    
    /**
     * 情報の追加.
     * <BR><BR>
     * 管理オブジェクトに数値列を追加します.
     * <BR>
     * @param num 追加対象の数値情報を設定します.<BR>
     *            また既に同一の値が存在する場合エラーとなります.
     * @exception InputException 入力例外.
     */
    public final void add( long num )
        throws InputException
    {
        if( m_search.isData( num ) == true ){
            throw new InputException(
                "対象の数値(" + num +
                ")は既に存在します"
            ) ;
        }
        m_array.add( num ) ;
        m_search.add( num ) ;
    }
    
    /**
     * 指定情報の削除.
     * <BR><BR>
     * 指定された情報を削除します.
     * <BR>
     * @param num 削除対象の数値情報を指定します.
     */
    public final void removeByNumber( long num )
    {
        int i ;
        int len ;
        
        LongArray ary = null ;
        
        if( m_search.isData( num ) == true ){
            
            ary = m_array ;
            len = ary.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                if( ary.get( i ) == num ){
                    ary.remove( i ) ;
                    break ;
                }
            }
            
            m_search.remove( num ) ;
        }
    }
    
    /**
     * 指定情報の削除.
     * <BR><BR>
     * 指定された情報を削除します.
     * <BR>
     * @param no 対象の情報取得位置を設定します.
     */
    public final void removeByArray( int no )
    {
        long num ;
        
        if( no >= 0 || no < m_array.size() ){
            num = m_array.get( no ) ;
            m_search.remove( num ) ;
        }
    }
    
    /**
     * 情報取得.
     * <BR><BR>
     * 対象の取得位置を指定して情報を取得します.
     * <BR>
     * @param cnt 対象の情報取得位置を設定します.
     */
    public final long get( int cnt )
    {
        return m_array.get( cnt ) ;
    }
    
    /**
     * 数値に対する存在チェック.
     * <BR><BR>
     * 数値に対する存在チェックを行います.
     * <BR>
     * @param num チェック対象の数値を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、情報は存在します.<BR>
     *                 [false]が返された場合、情報は存在しません.
     */
    public final boolean isData( long num )
    {
        return m_search.isData( num ) ;
    }
    
    /**
     * 格納情報数の取得.
     * <BR><BR>
     * 格納情報数が返されます.
     * <BR>
     * @return int 格納情報数が返されます.
     */
    public final int size()
    {
        return m_array.size() ;
    }
    
    /**
     * 格納数値配列の取得.
     * <BR><BR>
     * 格納されているキーとなる数値をすべて取得します.
     * <BR>
     * @return long[] 格納数値配列が返されます.<BR>
     *               情報が存在しない場合[null]が返されます.
     */
    public final long[] getNumbers()
    {
        int len ;
        long[] ret = null ;
        
        if( ( len = m_array.size() ) > 0 ){
            ret = new long[ len ] ;
            System.arraycopy( m_array.getObjects(),0,ret,0,len ) ;
        }
        
        return ret ;
    }
    
}
