/*
 * @(#)IdManager.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.serialize.InitSerialize;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.thread.Synchronized;


/**
 * ID発番管理オブジェクト.
 * <BR><BR>
 * ID発番を管理するためのオブジェクトです.<BR>
 * このオブジェクトは、[org.maachang.commons.util.IdManagerEx]より、小さい値のIDを管理
 * する場合に利用することを推奨します.<BR>
 * また、IdManagerオブジェクトとIdManagerExオブジェクトの単位の幅は以下の
 * ようになっています.<BR>
 * <PRE>
 *   IdManager      :   0   -   2147483647
 *   IdManagerEx    :   0   -   9223372036854775807
 * </PRE>
 * また、速度を優先する場合は IdManagerオブジェクトを利用します.<BR>
 * また、情報桁幅を多く取りたい場合などはIdManagerExオブジェクトを利用します.
 *
 * @version     1.00, 2004/04/12
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class IdManager implements InitSerialize
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            IdManager.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 取得IDなし.
     */
    public static final int NOT_ID = -1 ;
    
    /**
     * 検索対象数.
     */
    public static final int SEARCH_LENGTH = 100000 ;
    
    
    /**
     * 利用ID管理項番.
     */
    private final SearchInt m_num = new SearchInt() ;
    
    /**
     * MAX-ID.
     */
    private int m_maxID = 0 ;
    
    /**
     * 開始値.
     */
    private int m_startID = 0 ;
    
    /**
     * 管理ID.
     */
    private int m_id = 0 ;
    
    /**
     * テーブル生成フラグ.
     */
    private volatile boolean m_tblFlg = false ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 任意の最大値を用いて情報を生成します.
     */
    public IdManager()
    {
        this.create( 0,Integer.MAX_VALUE ) ;
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     * @exception InputException 入力例外.
     */
    public IdManager( int start,int max )
    {
        this.create( start,max ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        try{
            this.clear() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行うメソッドです.<BR>
     * 基本的には、このインターフェイスを継承したオブジェクトは、
     * 初期化に必要な処理を実装することで、
     * [org.maachang.commons.serialize.SerializeCom.getSerialize()]からの、
     * オブジェクトロードの際に、このメソッドを呼び出してくれるので、
     * ロード後のオブジェクトを、円滑に利用する事が出来ます.
     */
    public void initSerializable()
    {
        m_sync.create() ;
    }
    
    /**
     * 情報生成.
     * <BR><BR>
     * 情報を生成します.
     * <BR>
     * @param start 対象の開始値を設定します.
     * @param max 管理を行うIDの幅(MAX値)を設定します.
     */
    public final void create( int start,int max )
    {
        
        if(
            start < 0 || max < 1 ||
            start >= max || max > Integer.MAX_VALUE
        )
        {
            start = 0 ;
            max = Integer.MAX_VALUE ;
        }
        
        this.clear() ;
        m_sync.create() ;
        
        try{
            synchronized( m_sync ){
                m_num.create() ;
                m_maxID = max ;
                m_startID = start ;
                m_id = start ;
                this.setTblOn() ;
            }
        }catch( NullPointerException nul ){
            this.clear() ;
        }
        
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public final void clear()
    {
        m_sync.clear() ;
        m_sync.create() ;
        
        this.setTblOff() ;
        try{
            synchronized( m_sync.get() ){
                m_num.clear() ;
                m_id = m_startID ;
            }
        }catch( NullPointerException nul ){
            m_num.clear() ;
            m_id = m_startID ;
        }
        
    }
    
    /**
     * 利用中のID情報を追加.
     * <BR><BR>
     * 利用中のID情報を追加します.
     * <BR>
     * @param id 利用中のID情報として追加します.
     */
    public final void addUseID( int id )
    {
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                if(
                    m_startID <= id &&
                    m_maxID >= id &&
                    m_num.isData( id ) == false
                )
                {
                    m_num.add( id ) ;
                }
            }
        }catch( Exception t ){
        }
    }
    
    /**
     * 一意なID項番を取得.
     * <BR><BR>
     * 一意なID項番を取得します.
     * <BR>
     * @return int 一意なID項番が返されます.
     */
    public final int getID()
    {
        int i ;
        int len ;
        int startID ;
        int maxID ;
        
        int ret ;
        boolean okFlg ;
        SearchInt tbl = null ;
        
        this.renewTable() ;
        ret = IdManager.NOT_ID ;
        len = IdManager.SEARCH_LENGTH ;
        okFlg = false ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                startID = m_startID ;
                maxID = m_maxID ;
                tbl = m_num ;
                
                if( maxID - startID <= tbl.size() ){
                    return IdManager.NOT_ID ;
                }
                
            }
            
            for( ;; ){
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    synchronized( m_sync.get() ){
                        
                        ret = m_id ;
                        m_id = ( ret >= maxID ) ? startID : ret + 1 ;
                        
                        if( tbl.isData( ret ) == false ){
                            
                            tbl.add( ret ) ;
                            okFlg = true ;
                            break ;
                            
                        }
                        
                    }
                    
                }
                
                if( okFlg == true ){
                    break ;
                }
                
                UtilCom.cpuCreate() ;
                
            }
            
            
        }catch( NullPointerException nul ){
            ret = IdManager.NOT_ID ;
        }finally{
            tbl = null ;
        }
        
        return ret ;
    }
    
    /**
     * 利用終了IDの削除.
     * <BR><BR>
     * 利用終了IDを削除します.
     * <BR>
     * @param id 利用終了のIDを設定します.
     */
    public final void removeID( int id )
    {
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                m_num.remove( id ) ;
            }
        }catch( NullPointerException nul ){
        }
        
    }
    
    /**
     * 設定ID開始番号の取得.
     * <BR><BR>
     * 設定されているID開始番号を取得します.
     * <BR>
     * @return int 設定されているID開始番号が返されます.
     */
    public final int getStartID()
    {
        int ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_startID ;
            }
        }catch( NullPointerException nul ){
            ret = IdManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 設定ID幅(MAX値)を取得します.
     * <BR><BR>
     * 設定されているID幅(MAX値)を取得します.
     * <BR>
     * @return int 設定されているID幅(MAX値)が返されます.
     */
    public final int getMaxID()
    {
        int ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_maxID ;
            }
        }catch( NullPointerException nul ){
            ret = IdManager.NOT_ID ;
        }
        
        return ret ;
    }
    
    /**
     * 現在IDを利用している情報数を取得.
     * <BR><BR>
     * 現在IDを利用している情報数を取得します.
     * <BR>
     * @return int 現在IDを利用している情報数が返されます.
     */
    public final int getUseIDSize()
    {
        int ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.size() ;
            }
        }catch( NullPointerException nul ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 現在利用されているID群を取得.
     * <BR><BR>
     * 現在利用されているID群を取得します.
     * <BR>
     * @return int[] 現在利用されているID群が返されます.
     */
    public final int[] getUseIDs()
    {
        int[] ret = null ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.getAll() ;
            }
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定ID情報が既に利用しているかチェック.
     * <BR><BR>
     * 指定されたID情報が既に利用しているかチェックします.
     * <BR>
     * @param id チェック対象のID情報を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、指定ID情報は現在利用されています.
     *                 [false]が返された場合、指定ID情報は現在利用されていません.
     */
    public final boolean isUseID( int id )
    {
        boolean ret ;
        
        this.renewTable() ;
        try{
            synchronized( m_sync.get() ){
                ret = m_num.isData( id ) ;
            }
        }catch( NullPointerException nul ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 重複管理テーブル未生成時の生成処理.
     */
    private synchronized final void renewTable()
    {
        if( m_tblFlg == false ){
            m_num.create() ;
            m_tblFlg = true ;
        }
    }
    
    /**
     * テーブル生成フラグをONに設定.
     */
    private synchronized final void setTblOn()
    {
        m_tblFlg = true ;
    }
    
    /**
     * テーブル生成フラグをOFFに設定.
     */
    private synchronized final void setTblOff()
    {
        m_tblFlg = false ;
    }
}

