/*
 * @(#)ListDirectoryImpl.java
 *
 * Copyright (c) 2004 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.io.Serializable;
import java.util.ArrayList;

import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.ExistException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.NotExistException;
import org.maachang.commons.serialize.InitSerialize;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.thread.Synchronized;
import org.maachang.commons.util.array.ObjectArray;


/**
 * ディレクトリ構成管理.
 * <BR><BR>
 * ディレクトリ構成を管理するオブジェクトを提供します.
 *
 * @version     1.00, 2004/05/01
 * @author      Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ListDirectory implements InitSerialize
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ListDirectory.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * パス区切り文字.
     */
    public static final String PATH  = "/" ;
    
    /**
     * ルートディレクトリ名.
     */
    public static final String ROOT = "" ;
    
    
    
    /**
     * ディレクトリ名セット.
     */
    private static final String PLUS_DIRNAME = "*" ;
    
    /**
     * データ格納オブジェクト.
     */
    private final DirPause m_table = new DirPause() ;
    
    /**
     * 現在位置ディレクトリオブジェクト.
     */
    private DirPause m_nowTable = null ;
    
    /**
     * 現在位置ディレクトリ名.
     */
    private final ObjectArray m_nowDir = new ObjectArray() ;
    
    
    /**
     * 同期オブジェクト.
     */
    private final Synchronized m_sync = new Synchronized() ;
    
    
    
    /**
     * コンストラクタ.
     */
    public ListDirectory()
    {
        this.thisNowDirectory() ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.destroy() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 初期化処理.
     * <BR><BR>
     * 初期化処理を行うメソッドです.<BR>
     * 基本的には、このインターフェイスを継承したオブジェクトは、
     * 初期化に必要な処理を実装することで、
     * [org.maachang.commons.serialize.SerializeCom.getSerialize()]からの、
     * オブジェクトロードの際に、このメソッドを呼び出してくれるので、
     * ロード後のオブジェクトを、円滑に利用する事が出来ます.
     */
    public void initSerializable()
    {
        m_sync.create() ;
    }
    
    /**
     * ディレクトリ構成破棄.
     * <BR><BR>
     * 全てのディレクトリ構成を破棄します.
     */
    public final void destroy()
    {
        
        m_table.m_now.clear() ;
        m_table.m_before = null ;
        
        m_nowTable = null ;
        m_nowDir.clear() ;
        
        m_sync.clear() ;
        
    }
    
    
    /**
     * 現在のディレクトリ位置に新しいディレクトリを生成.
     * <BR><BR>
     * 現在のディレクトリ位置に新しいディレクトリを生成します.
     * <BR>
     * @param directory 生成対象のディレクトリ名を設定します.
     * @param state 対象のディレクトリステータスを設定します.
     * @exception InputException 入力例外.
     * @exception ExistException 存在例外.
     */
    public final void createDirectory( String directory,Object state )
        throws InputException,ExistException
    {
        int i ;
        int len ;
        
        String name = null ;
        DirPause dir = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( directory.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + directory + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isDirName( name ) == true ){
                    
                    throw new ExistException(
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        ")に対してディレクトリ:[" + directory +
                        "]は既に存在します"
                    ) ;
                    
                }
                
                dir = new DirPause() ;
                dir.m_before = m_nowTable ;
                dir.m_state = state ;
                
                m_nowTable.m_now.add( name,dir ) ;
                
            }
            
        }catch( InputException ie ){
            throw ie ;
        }catch( ExistException ee ){
            throw ee ;
        }catch( Exception t ){
        }finally{
            
            name = null ;
            dir = null ;
            
        }
        
    }
    
    /**
     * 現在のディレクトリ位置に新しい要素情報を生成.
     * <BR><BR>
     * 現在のディレクトリ位置に新しい要素情報を生成します.
     * <BR>
     * @param name 生成対象の要素名を指定します.
     * @param value 生成対象の要素情報を指定します.
     * @exception InputException 入力例外.
     */
    public final void createPause( String name,Object value )
        throws InputException
    {
        Object vl = null ;
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        vl = ( value == null ) ? new NullObject() : value ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                m_nowTable.m_now.add( name,value ) ;
                
            }
            
        }catch( InputException ie ){
            throw ie ;
        }catch( Exception t ){
        }finally{
            
            vl = null ;
            
        }
    }
    
    /**
     * 現在のディレクトリ位置の対象要素情報を置き換え.
     * <BR><BR>
     * 現在のディレクトリ位置の対象要素情報を置き換えます.
     * <BR>
     * @param name 置き換え対象の要素名を設定します.
     * @param no 置き換え対象の要素番号を設定します.
     * @param value 置き換え対象の要素情報を設定します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final void setPause( String name,int no,Object value )
        throws InputException,NotExistException
    {
        if( name == null || value == null || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isPause( name ) == false ){
                    
                    throw new NotExistException(
                        "対象位置(" + this.getPwd() +
                        ")の設定元要素名(" + name +
                        ")は存在しません"
                    ) ;
                    
                }
                
                m_nowTable.m_now.set( value,name,no ) ;
                
            }
        }catch( NotExistException ne ){
            throw ne ;
        }
    }
    
    /**
     * 現在のディレクトリ位置の対象のディレクトリ情報を削除.
     * <BR><BR>
     * 現在のディレクトリ位置の対象のディレクトリ情報を削除します.
     * <BR>
     * @param directory 削除対象のディレクトリ名を設定します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final void removeDirectory( String directory )
        throws InputException,NotExistException
    {
        String name = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( directory.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + directory + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isDirName( name ) == false ){
                    
                    throw new NotExistException(
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        ")に対してディレクトリ:[" + directory +
                        "]は存在しません"
                    ) ;
                    
                }
                
                m_nowTable.m_now.remove( name ) ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
        }finally{
            name = null ;
        }
        
    }
    
    /**
     * 現在のディレクトリ位置の対象の要素情報を全て削除.
     * <BR><BR>
     * 現在のディレクトリ位置の対象の要素情報を全て削除します.
     * <BR>
     * @param name 削除対象の要素名を設定します.
     * @return Object[] 削除された要素情報が全て返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final Object[] removeAllPause( String name )
        throws InputException,NotExistException
    {
        Object[] ret = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isPause( name ) == true ){
                    
                    ret = m_nowTable.m_now.remove( name ) ;
                    
                }
                else{
                    throw new NotExistException(
                        "対象のディレクトリ上の(" +
                        ListDirectory.getFullPath( m_nowDir ) +
                        ")要素名(" + name +
                        ")は存在しません"
                    ) ;
                }
                
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在のディレクトリ位置の対象の要素情報を１つ削除.
     * <BR><BR>
     * 現在のディレクトリ位置の対象の要素情報を１つ削除します.
     * <BR>
     * @param name 削除対象の要素名を設定します.
     * @param no 削除対象の要素項番を設定します.
     * @return Object 削除された要素情報が返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final Object removePause( String name,int no )
        throws InputException,NotExistException
    {
        Object ret = null ;
        
        if( name == null || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isPause( name ) == true ){
                    
                    ret = m_nowTable.m_now.remove( name,no ) ;
                    
                }
                else{
                    throw new NotExistException(
                        "対象のディレクトリ上の(" +
                        ListDirectory.getFullPath( m_nowDir ) +
                        ")要素名(" + name +
                        ")は存在しません"
                    ) ;
                }
                
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在位置のディレクトリ情報を取得.
     * <BR><BR>
     * 現在位置のディレクトリ情報を取得します.
     * <BR>
     * @param directory 対象のディレクトリ名を設定します.
     * @return ArrayTable 対象のディレクトリを示すテーブルオブジェクトが返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final ArrayTable getDirectoryTable( String directory )
        throws InputException,NotExistException
    {
        String name = null ;
        Object pause = null ;
        ArrayTable ret = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( directory.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + directory + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = ( DirPause )m_nowTable.m_now.get( name,0 ) ;
                
                if( pause == null || ( pause instanceof DirPause ) == false ){
                    throw new NotExistException(
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        ")に対してディレクトリ:[" + directory +
                        "]は存在しません"
                    ) ;
                }
                
                ret = ( ( DirPause )pause ).m_now ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            name = null ;
            pause = null ;
        }
        
        return ret ;
        
    }
    
    
    /**
     * 現在位置のディレクトリステータスを設定.
     * <BR><BR>
     * 現在位置のディレクトリステータスを設定します.
     * <BR>
     * @param state 対象のステータス情報を設定します.
     */
    public final void setThisDirectoryState( Object state )
    {
        this.thisNowDirectory() ;
        
        try{
            synchronized( m_sync.get() ){
                m_nowTable.m_state = state ;
            }
        }catch( Exception e ){
        }
    }
    
    /**
     * 指定位置のディレクトリステータスを設定.
     * <BR><BR>
     * 指定位置のディレクトリステータスを設定します.
     * <BR>
     * @param directory 対象のディレクトリ名を設定します.
     * @param state 対象のステータス情報を設定します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final void setDirectoryState( String directory,Object state )
        throws InputException,NotExistException
    {
        String name = null ;
        Object pause = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( directory.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + directory + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = ( DirPause )m_nowTable.m_now.get( name,0 ) ;
                
                if( pause == null || ( pause instanceof DirPause ) == false ){
                    throw new NotExistException(
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        ")に対してディレクトリ:[" + directory +
                        "]は存在しません"
                    ) ;
                }
                
                ( ( DirPause )pause ).m_state = state ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
        }finally{
            name = null ;
            pause = null ;
        }
    }
    
    /**
     * 現在位置のディレクトリステータスを取得.
     * <BR><BR>
     * 現在位置のディレクトリステータスを取得します.
     * <BR>
     * @return Object 現在のディレクトリステータスが返されます.
     */
    public final Object getThisDirectoryState()
    {
        Object ret = null ;
        
        this.thisNowDirectory() ;
        
        try{
            synchronized( m_sync.get() ){
                ret = m_nowTable.m_state ;
            }
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 指定位置のディレクトリステータスを取得.
     * <BR><BR>
     * 指定位置のディレクトリステータスを取得します.
     * <BR>
     * @param directory 対象のディレクトリ名を設定します.
     * @return Object 対象のディレクトリステータスが返されます.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final Object getDirectoryState( String directory )
        throws InputException,NotExistException
    {
        String name = null ;
        Object pause = null ;
        Object ret = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( directory.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + directory + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = ( DirPause )m_nowTable.m_now.get( name,0 ) ;
                
                if( pause == null || ( pause instanceof DirPause ) == false ){
                    throw new NotExistException(
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        ")に対してディレクトリ:[" + directory +
                        "]は存在しません"
                    ) ;
                }
                
                ret = ( ( DirPause )pause ).m_state ;
                
            }
            
        }catch( InputException in ){
            throw in ;
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
            ret = null ;
        }finally{
            name = null ;
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * １つ下のディレクトリに遷移.
     * <BR><BR>
     * １つ下のディレクトリに遷移します.
     * <BR>
     * @exception AccessException アクセス例外.<BR>
     *                            終端(ROOT)が検知された場合発生します.
     */
    public final void lower() throws AccessException
    {
        
        this.thisNowDirectory() ;
        
        try{
            synchronized( m_sync.get() ){
                
                if( m_nowTable.m_before == null ){
                    throw new AccessException(
                        "移動に対して、終端(ROOT)位置の為移動できません"
                    ) ;
                }
                
                m_nowTable = ( DirPause )m_nowTable.m_before ;
                m_nowDir.remove( ( m_nowDir.size() - 1 ) ) ;
                
            }
            
        }catch( AccessException ac ){
            throw ac ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * １つ上のディレクトリに遷移.
     * <BR><BR>
     * 指定された１つ上のディレクトリに遷移します.
     * <BR>
     * @param directory 対象の１つ上のディレクトリに遷移します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final void upper( String directory )
        throws InputException,NotExistException
    {
        String name = null ;
        DirPause next = null ;
        
        if( directory == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.thisNowDirectory() ;
        
        name = ListDirectory.cnvDirName( directory ) ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_nowTable.m_now.getElements( name )!= 1 ){
                    
                    throw new NotExistException(
                        "対象のディレクトリ位置(" + 
                        "(" + ListDirectory.getFullPath( m_nowDir ) +
                        "に対するディレクトリ名(" +
                        directory +
                        ")は存在しません"
                    ) ;
                    
                }
                
                next = ( DirPause )m_nowTable.m_now.get( name,0 ) ;
                m_nowTable = next ;
                m_nowDir.add( directory ) ;
                
            }
        }catch( NotExistException ne ){
            throw ne ;
        }finally{
            name = null ;
            next = null ;
        }
        
    }
    
    /**
     * ディレクトリチェンジ.
     * <BR><BR>
     * ディレクトリ条件をチェンジします.
     * <BR>
     * @param pwd 指定されたディレクトリ位置に遷移します.
     * @exception InputException 入力例外.
     * @exception NotExistException 非存在例外.
     */
    public final void change( String pwd )
        throws InputException,NotExistException
    {
        int i ;
        int len ;
        
        boolean full ;
        
        String fullpath = null ;
        String pathName = null ;
        String name = null ;
        ArrayList array = null ;
        DirPause pause = null ;
        
        if( pwd == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            full = true ;
            fullpath = UtilCom.trimPlus( pwd ) ;
            
            if( 
                fullpath.charAt( 0 ) == '\0' ||
                fullpath.charAt( 0 ) != ListDirectory.PATH.charAt( 0 )
            )
            {
                if( fullpath.charAt( 0 ) == '\0' ){
                    throw new InputException(
                        "指定された引数情報は空であるか、情報が存在しません."
                    ) ;
                }
                
                full = false ;
            }
            
            array = UtilCom.pauseString( fullpath,ListDirectory.PATH ) ;
            len = array.size() ;
            
            synchronized( m_sync.get() ){
                
                if( full == true && len == 0 ){
                    
                    m_nowTable = null ;
                    m_nowTable = m_table ;
                    m_nowDir.clear() ;
                    m_nowDir.add( ListDirectory.ROOT ) ;
                    
                }
                else{
                    
                    if( full == true ){
                        pause = m_table ;
                        m_nowDir.clear() ;
                        m_nowDir.add( ListDirectory.ROOT ) ;
                        
                    }
                    else{
                        pause = m_nowTable ;
                    }
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        pathName = ( String )array.get( i ) ;
                        
                        name = ListDirectory.cnvDirName( pathName ) ;
                        
                        if( pause.m_now.getElements( name ) != 1 ){
                            
                            m_nowTable = null ;
                            m_nowTable = pause ;
                            
                            throw new NotExistException(
                                "移動対象のディレクトリ(" +
                                pwd + ")の位置(" + i + ")のディレクトリ名(" +
                                ( String )array.get( i ) +
                                ")は存在しません"
                            ) ;
                            
                        }
                        
                        pause = ( DirPause )pause.m_now.get( name,0 ) ;
                        m_nowDir.add( pathName ) ;
                        
                    }
                    
                    m_nowTable = null ;
                    m_nowTable = pause ;
                    
                }
            }
            
        }catch( NotExistException ne ){
            throw ne ;
        }catch( Exception t ){
        }finally{
            
            pathName = null ;
            name = null ;
            array = null ;
            pause = null ;
            
        }
        
    }
    
    /**
     * 現在のパス状態を取得します.
     * <BR><BR>
     * @return String 現在のパス情報が返されます.
     */
    public final String getPwd()
    {
        this.thisNowDirectory() ;
        
        return ListDirectory.getFullPath( m_nowDir ) ;
    }
    
    /**
     * 現在位置のディレクトリ数を取得.
     * <BR><BR>
     * 現在位置のディレクトリ数を取得します.
     * <BR>
     * @return int 現在位置のディレクトリ数が返されます.
     */
    public final int getDirectoryCount()
    {
        this.thisNowDirectory() ;
        
        return this.getCount( true ) ;
    }
    
    /**
     * 現在位置のディレクトリ名群を取得.
     * <BR><BR>
     * 現在位置のディレクトリ名群を取得します.
     * <BR>
     * @return String[] 現在位置のディレクトリ名群を取得します.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getDirectorys()
    {
        this.thisNowDirectory() ;
        
        return this.getNames( true ) ;
    }
    
    /**
     * 現在位置の要素数を取得.
     * <BR><BR>
     * 現在位置の要素数を取得します.
     * <BR>
     * @return int 現在位置の要素数が返されます.
     */
    public final int getPauseCount()
    {
        this.thisNowDirectory() ;
        
        return this.getCount( false ) ;
    }
    
    /**
     * 現在位置の要素名群を取得.
     * <BR><BR>
     * 現在位置の要素名群を取得します.
     * <BR>
     * @return String[] 現在位置の要素名群を取得します.<BR>
     *                  情報が存在しない場合[null]が返されます.
     */
    public final String[] getPauses()
    {
        this.thisNowDirectory() ;
        
        return this.getNames( false ) ;
    }
    
    /**
     * 指定要素名に対する管理要素情報数を取得.
     * <BR><BR>
     * 指定要素名に対する管理要素情報数を取得します.
     * <BR>
     * @param name 要素数取得対象の要素名を設定します.
     * @return int 要素名に対する管理要素数が返されます.<BR>
     *             対象の要素名が存在しない場合[0]が返されます.
     */
    public final int getPauseElements( String name )
    {
        int ret ;
        
        if( name == null ){
            return 0 ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            return 0 ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                ret = m_nowTable.m_now.getElements( name ) ;
            }
            
        }catch( Exception t ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の要素名に対する全ての要素情報を取得.
     * <BR><BR>
     * 対象の要素名に対する全ての要素情報を取得します.
     * <BR>
     * @param name 取得対象の要素名を設定します.
     * @return Object[] 要素名に対する全ての要素情報が返されます.<BR>
     *                  対象の要素名が存在しない場合[null]が返されます.
     * @exception InputException 入力例外.
     */
    public final Object[] getAllPause( String name )
        throws InputException
    {
        Object[] ret ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        
        ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isPause( name ) == true ){
                    
                    ret = m_nowTable.m_now.get( name ) ;
                    
                }
                
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象の要素名に対する１つの要素情報を取得.
     * <BR><BR>
     * 対象の要素名に対する１つの要素情報を取得します.
     * <BR>
     * @param name 取得対象の要素名を設定します.
     * @param no 取得対象の要素項番を設定します.
     * @return Object 要素名に対する要素情報の指定要素項番に対する
     *                １つの情報が返されます.<BR>
     *                  対象の要素名が存在しない場合[null]が返されます.
     * @exception InputException 入力例外.
     */
    public final Object getPause( String name,int no )
        throws InputException
    {
        boolean isData ;
        
        Object ret ;
        
        if( name == null || no < 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            throw new InputException(
                "名前(" + name + ")の中に禁止文字(" +
                ListDirectory.PLUS_DIRNAME +
                ")が含まれています"
            ) ;
        }
        
        this.thisNowDirectory() ;
        
        ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                if( this.isPause( name ) == true ){
                    
                    ret = m_nowTable.m_now.get( name,no ) ;
                    
                }
                
            }
            
        }catch( Exception t ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象のパス情報が存在するかチェック.
     * <BR><BR>
     * 対象のパス情報が存在するかチェックします.
     * <BR>
     * @param pwd チェックを行うパス情報を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、対象パス情報は存在します.<BR>
     *                 [false]が返された場合、対象パス情報は存在しません.
     */
    public final boolean isPath( String pwd )
    {
        boolean ret ;
        String bkPwd ;
        
        ret = false ;
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                bkPwd = this.getPwd() ;
                
                try{
                    this.change( pwd ) ;
                    ret = true ;
                }catch( Exception t ){
                    ret = false ;
                }
                
                this.change( bkPwd ) ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }finally{
            bkPwd = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のディレクトリに対して、対象ディレクトリ名が存在するかチェック.
     * <BR><BR>
     * 現在のディレクトリに対して、対象ディレクトリ名が存在するかチェックします.
     * <BR>
     * @param directory 存在チェック対象のディレクトリを設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、対象ディレクトリ名は存在します.<BR>
     *                 [false]が返された場合、対象ディレクトリ名は存在しません.
     */
    public final boolean isDirectory( String directory )
    {
        boolean ret ;
        
        String name = null ;
        
        if( directory == null ){
            return false ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            name = ListDirectory.cnvDirName( directory ) ;
            ret = this.isDirName( name ) ;
        }catch( InputException in ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * 現在のディレクトリに対して、対象要素名が存在するかチェック.
     * <BR><BR>
     * 現在のディレクトリに対して、対象要素名が存在するかチェックします.
     * <BR>
     * @param name 存在チェック対象の要素名を設定します.
     * @return boolean 存在結果が返されます.<BR>
     *                 [true]が返された場合、対象要素名は存在します.<BR>
     *                 [false]が返された場合、対象要素名は存在しません.
     */
    public final boolean isPause( String name )
    {
        boolean ret ;
        
        if( name == null ){
            return false ;
        }
        else if( name.indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
            return false ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = ( m_nowTable.m_now.getElements( name ) != 0 ) ?
                    true : false ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    
    /**
     * 自ディレクトリ生成.
     */
    private final void thisNowDirectory()
    {
        try{
            
            synchronized( m_sync.get() ){
                
                if( m_nowTable == null ){
                    
                    m_table.m_before = null ;
                    m_table.m_now.clear() ;
                    
                    m_nowTable = m_table ;
                    
                    m_nowDir.clear() ;
                    m_nowDir.add( ListDirectory.ROOT ) ;
                    
                }
                
            }
            
        }catch( Exception t ){
        }
        
    }
    
    /**
     * 現在ディレクトリの存在情報数を取得.
     */
    private final int getCount( boolean mode )
    {
        int i ;
        int len ;
        int dirLen ;
        int pauseLen ;
        
        int ret ;
        
        String[] names = null ;
        DirPause pause = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = m_nowTable ;
                names = pause.m_now.getNames() ;
                len = names.length ;
                
                dirLen = 0 ;
                pauseLen = 0 ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    if( names[ i ].indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
                        dirLen ++ ;
                    }else{
                        pauseLen ++ ;
                    }
                    
                }
            }
            
            ret = ( mode == true ) ? dirLen : pauseLen ;
            
        }catch( Exception t ){
            ret = 0 ;
        }finally{
            names = null ;
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在ディレクトリに存在する内容を取得.
     */
    private final String[] getNames( boolean mode )
    {
        int i ;
        int len ;
        int dirLen ;
        int pauseLen ;
        
        ObjectArray lst = null ;
        String[] names = null ;
        DirPause pause = null ;
        
        String[] ret = null ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                pause = m_nowTable ;
                names = pause.m_now.getNames() ;
                len = names.length ;
                
                dirLen = 0 ;
                pauseLen = 0 ;
                
                lst = new ObjectArray() ;
                
                if( mode == true ){
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        if( names[ i ].indexOf( ListDirectory.PLUS_DIRNAME ) != -1 ){
                            lst.add( ListDirectory.reCnvDirName( names[ i ] ) ) ;
                        }
                        
                    }
                    
                }
                else{
                    
                    for( i = 0 ; i < len ; i ++ ){
                        
                        if( names[ i ].indexOf( ListDirectory.PLUS_DIRNAME ) == -1 ){
                            lst.add( names[ i ] ) ;
                        }
                        
                    }
                    
                }
            }
            
            if( (len = lst.size() ) <= 0 ){
                
                ret = null ;
                
            }
            else{
                
                ret = new String[ len ] ;
                
                for( i = 0 ; i < len ; i ++ ){
                    ret[ i ] = ( String )lst.get( i ) ;
                }
                
            }
            
        }catch( Exception t ){
            ret = null ;
        }finally{
            lst = null ;
            names = null ;
            pause = null ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のディレクトリに対して、対象ディレクトリ名が存在するかチェック.
     */
    private final boolean isDirName( String dir )
    {
        boolean ret ;
        
        if( dir == null ){
            return false ;
        }
        
        this.thisNowDirectory() ;
        
        try{
            
            synchronized( m_sync.get() ){
                
                ret = ( m_nowTable.m_now.getElements( dir ) != 0 ) ?
                    true : false ;
                
            }
            
        }catch( Exception t ){
            ret = false ;
        }
        
        return ret ;
        
    }
    
    /**
     * ディレクトリ名生成.
     */
    private static final String cnvDirName( String name )
        throws InputException
    {
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer( name.length()+PLUS_DIRNAME.length() ) ;
        buf.append( PLUS_DIRNAME ) ;
        buf.append( name ) ;
        
        ret = UtilCom.trimPlus( buf.toString() ) ;
        
        buf = null ;
        
        return ret ;
    }
    
    /**
     * 対象ディレクトリ名を本来の名前に変換.
     */
    private static final String reCnvDirName( String name )
        throws InputException
    {
        
        String ret = null ;
        
        if( name == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        if( name.indexOf( PLUS_DIRNAME ) != -1 ){
            ret = name.substring( PLUS_DIRNAME.length() ) ;
        }
        else{
            ret = name ;
        }
        
        return ret ;
    }
    
    /**
     * 現在のディレクトリ位置のフルパス名を生成.
     */
    private static final String getFullPath( ObjectArray nowDir )
    {
        int i ;
        int len ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( nowDir == null ){
            return null ;
        }
        
        buf = new StringBuffer() ;
        len = nowDir.size() ;
        
        buf.append( ListDirectory.PATH ) ;
        
        for( i = 1 ; i < len ; i ++ ){
            
            buf.append( ( String )nowDir.get( i ) ) ;
            buf.append( ListDirectory.PATH ) ;
            
        }
        
        ret = buf.toString() ;
        buf = null ;
        
        return ret ;
        
    }
    
    
    
    /**
     * ディレクトリ要素.
     */
    class DirPause implements Serializable{
        
        /**
         * 前回のディレクトリ情報.
         */
        DirPause m_before = null ;
        
        /**
         * ディレクトリステータス.
         */
        Object m_state = null ;
        
        /**
         * 現在のディレクトリ管理領域.
         */
        final ArrayTable m_now = new ArrayTable() ;
        
        protected final void finalize() throws Exception
        {
            m_before = null ;
            m_state = null ;
            m_now.clear() ;
        }
        
    } ;
    
}

