/*
 * @(#)UtilCom.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.AccessException;
import org.maachang.commons.exception.ExecutionException;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadException;
import org.maachang.commons.exception.UnsupportException;
import org.maachang.commons.resource.BinResource;
import org.maachang.commons.resource.Resource;
import org.maachang.commons.thread.ThreadManager;


/**
 * 共通メソッド群.
 * <BR><BR>
 * 共通メソッド群を定義します.
 *  
 * @version 1.0.0 2003/10/24
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class UtilCom
{
    
    /**
     * タイムスタンプ情報取得配列数.
     */
    public static final int TIMESTAMP_STRING_LENGTH = 9 ;
    
    /**
     * タイムスタンプ情報並び順 : 年.
     */
    public static final int YEAR = 0 ;
    
    /**
     * タイムスタンプ情報並び順 : 月.
     */
    public static final int MONTH = 1 ;
    
    /**
     * タイムスタンプ情報並び順 : 日.
     */
    public static final int DAY = 2 ;
    
    /**
     * タイムスタンプ情報並び順 : 時.
     */
    public static final int HOUR = 3 ;
    
    /**
     * タイムスタンプ情報並び順 : 分.
     */
    public static final int MINUTE = 4 ;
    
    /**
     * タイムスタンプ情報並び順 : 秒.
     */
    public static final int SECOND = 5 ;
    
    /**
     * タイムスタンプ情報並び順 : ミリ秒.
     */
    public static final int MILLISECOND = 6 ;
    
    /**
     * タイムスタンプ情報並び順 : 週(日本語).
     */
    public static final int WEEK_JP = 7 ;
    
    /**
     * タイムスタンプ情報並び順 : 週(英語).
     */
    public static final int WEEK_ENG = 8 ;
    
    
    /**
     * ランダム文字列出力係数.
     */
    private static final int RAND_STRING_TO_NUMBER = 0 ;
    private static final int RAND_STRING_TO_HALF_ASC = 1 ;
    private static final int RAND_STRING_TO_FULL_ASC = 2 ;

    /**
     * ランダム文字列出力データ幅.
     */
    private static final int RAND_STRING_TO_NUBER_BY_LENGTH = (int)('9' - '0') ;
    private static final int RAND_STRING_TO_HALF_BY_LENGTH = (int)('z' - 'a') ;
    private static final int RAND_STRING_TO_FULL_BY_LENGTH = (int)('Z' - 'A') ;
    
    /**
     * 4ビット値情報から16進文字変換テーブル.
     */
    protected static final char[] convert10To16 =
    {
        '0','1','2','3','4','5','6','7',
        '8','9','A','B','C','D','E','F'
    } ;
    
    /**
     * 半角、全角文字コードチェック用エンコード.
     */
    public static final String CHECK_SJIS_CHARSET = "Shift_JIS" ;
    
    /**
     * データ付加余り値.
     * スペースをセット.
     */
    private static final String ADD_REM_CODE = " " ;
    
    /**
     * 乱数オブジェクト.
     */
    private static final Random RAND = new Random( System.currentTimeMillis() ) ;
    
    /**
     * シングルコーテーション.
     */
    private static final char SNGLCOTE = BaseDef.SINGLE_COAT.charAt( 0 ) ;
    
    /**
     * ダブルコーテーション.
     */
    private static final char DBLLCOTE = BaseDef.DOUBLE_COAT.charAt( 0 ) ;
    
    
    
    /**
     * CPUタイムクロック割り当て.
     * <BR><BR>
     * この情報は、メソッド[UtilCom#cpuCreate()],[UtilCom#idleTime()]で
     * スリープを行うミリ秒値です.
     * 
     */
    public static final long CPU_CREATE = 15 ;
    
    
    
    /**
     * コンストラクタ.
     */
    private UtilCom()
    {}
    
    /**
     * ランダムシートの設定.
     * <BR><BR>
     * 新たにランダム用のシート情報を設定します.
     */
    public static final void randomize()
    {
        synchronized( UtilCom.RAND ){
            
            UtilCom.RAND.setSeed( System.currentTimeMillis() ) ;
            
        }
    }
    
    /**
     * ランダムな値を取得.
     * <BR><BR>
     * ランダムな値を取得します.
     * <BR>
     * @param max  ランダム値の最大番号を指定します。
     * @return int 上記指定した情報のランダム値を取得します。
     */
    public static final int random( int max )
    {
        int ret ;
        
        synchronized( UtilCom.RAND ){
            
            ret = ( int )UtilCom.RAND.nextInt( max ) ;
            
        }
        
        return ret ;
    }
    
    /**
     * ランダムな値をバイナリで取得.
     * <BR><BR>
     * ランダムな値をバイナリで取得します.
     * <BR>
     * @param out  ランダム値を格納するバイナリ情報を設定します.
     * @exception InputException 入力例外.
     */
    public static final void randomBinary( byte[] out )
        throws InputException
    {
        
        if( out == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        synchronized( UtilCom.RAND ){
            
            UtilCom.RAND.nextBytes( out ) ;
            
        }
    }
    
    /**
     * ランダムな値をバイナリで取得.
     * <BR><BR>
     * ランダムな値をバイナリで取得します.
     * <BR>
     * @param length 生成対象のバイナリ情報長を設定します.
     * @return byte[]  ランダム値が格納されたバイナリ情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] randomBinary( int length )
        throws InputException
    {
        byte[] ret = null ;
        
        if( length <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = new byte[ length ] ;
        
        synchronized( UtilCom.RAND ){
            
            UtilCom.RAND.nextBytes( ret ) ;
            
        }
        
        return ret ;
    }
    
    /**
     * ランダムな文字列を取得.
     * <BR><BR>
     * ランダムな文字列を取得します.
     * <BR>
     * @param size  ランダムに設定する文字数を指定します.
     * @param number  数字を許可するか指定します.
     * @param half  半角文字を許可するか指定します.
     * @param full  全角文字を許可するか指定します.
     * @return String 上記指定された内容のランダムな文字列が返されます.
     */
    public static final String randomString(int size,boolean number,boolean half,boolean full)
    {
        int i ;
        int rnd = 0 ;
        int randSize = 0 ;
        
        int[] check = null ;
        String ret = null ;
        StringBuffer buf = null ;
        
        if(
            size <= 0 ||
            (number == false && half == false && full == false)
        )
        {
            if(number == false && half == false && full == false){
                number = true ;
            }else{
                return null ;
            }
        }
        
        check = new int[ 3 ] ;
        if(number == true){
            check[randSize] = RAND_STRING_TO_NUMBER ;
            randSize += 1 ;
        }
        if(half == true){
            check[randSize] = RAND_STRING_TO_HALF_ASC ;
            randSize += 1 ;
        }
        if(full == true){
            check[randSize] = RAND_STRING_TO_HALF_ASC ;
            randSize += 1 ;
        }
        
        buf = new StringBuffer( size ) ;
        
        for(i = 0 ; i < size ; i ++){
            
            // 乱数を発生
            rnd = UtilCom.random(randSize) ;
            
            // 取得した対象情報から情報を生成する
            switch(check[ rnd ]){
                
                // 数字
                case RAND_STRING_TO_NUMBER :
                    buf.append( (char)('0' + (byte)(UtilCom.random(RAND_STRING_TO_NUBER_BY_LENGTH))) ) ;
                    break ;
                
                // 半角
                case RAND_STRING_TO_HALF_ASC :
                    buf.append( (char)('a' + (byte)(UtilCom.random(RAND_STRING_TO_HALF_BY_LENGTH))) ) ;
                    break ;
                
                case RAND_STRING_TO_FULL_ASC :
                    buf.append( (char)('A' + (byte)(UtilCom.random(RAND_STRING_TO_FULL_BY_LENGTH))) ) ;
                    break ;
                
                default :
                    
                    // 数字として出力.
                    buf.append( (char)('0' + (byte)(UtilCom.random(RAND_STRING_TO_NUBER_BY_LENGTH))) ) ;
                    break ;
                
            }
        }
        
        ret = buf.toString() ;
        
        buf = null ;
        check = null ;
        
        return ret ;
    }
    
    /**
     * 指定された引数内の情報を取得して結果を切り出します。
     * <BR><BR>
     * @param argument 複数のシンボルを含む引数情報。
     * @param check 切り出したい情報のシンボル
     * @return String シンボル以降の条件が返されます.
     */
    public static final String getArgument(String argument[],String check)
    {
        int i,j ;
        int len ;
        boolean flg = true ;
        String tmp = null ;
        String ret = null ;
        
        if( argument == null || check == null || 
            argument.length == 0 || check.length() == 0
        )
        {
            return null ;
        }
        
        len = argument.length ;
        check = check.toLowerCase() ;
        
        for(i = 0 ; i < len ; i ++){
            
            tmp = argument[i].trim().toLowerCase() ;
            
            // チェック
            if( tmp.indexOf( check ) != -1 ){
                
                ret = new String( argument[i].substring( check.length() ) ) ;
                
                if( ret == null || ret.length() <= 0 ){
                    ret = "" ;
                }
                break ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * CPU待ち時間を設定します。
     * <BR><BR>
     * CPU待ち時間を設定します。
     */
    public static final void cpuCreate()
    {
        try{
            
            //try{ThreadManager.isExit() ;}catch( Exception tt ){}
            //Thread.yield() ;
            Thread.sleep( CPU_CREATE ) ;
            //Thread.yield() ;
            
        }catch( InterruptedException ie ){
            Thread.currentThread().interrupt() ;
        }catch( Exception t ){
        }
    }
    
    /**
     * スレッドアイドルタイム設定.
     * <BR><BR>
     * スレッドアイドルタイムを設定します.
     * スレッド処理での繰り返し処理部分に対して、アイドルタイムを
     * 設定する場合に呼び出します.
     * <BR>
     * @exception ExecutionException スレッド終了例外.
     */
    public static final void idleTime() throws ExecutionException
    {
        try{
            
            ThreadManager.isExit() ;
            //Thread.yield() ;
            Thread.sleep( CPU_CREATE ) ;
            //Thread.yield() ;
            
        }catch( InterruptedException ie ){
            throw new ExecutionException( ie,ExecutionException.LEVEL_STOP ) ;
        }catch( ExecutionException ex ){
            throw ex ;
        }
    }
    
    /**
     * 指定したミリ秒に対して処理を一時停止させます。
     * <BR><BR>
     * @param min 一時停止したいミリ秒の数値を指定します。
     */
    public static final void sleep( long min )
    {
        long delay ;
        
        delay = System.currentTimeMillis() + min ;
        
        for ( ;; ){
            UtilCom.cpuCreate() ;
            if( delay <= System.currentTimeMillis() ){
                break ;
            }
        }
    }
    
    /**
     * 指定したミリ秒に対して処理を一時停止させます。
     * <BR><BR>
     * また、この処理は[UtilCom#sleep()]とは違い、スレッドストップを
     * 検知した場合、[ExecutionException]を返します.
     * <BR>
     * @param min 一時停止したいミリ秒の数値を指定します。
     * @exception ExecutionException 実行例外.
     */
    public static final void idleSleep( long min )
        throws ExecutionException
    {
        long delay ;
        
        delay = System.currentTimeMillis() + min ;
        
        for ( ;; ){
            UtilCom.idleTime() ;
            if( delay <= System.currentTimeMillis() ){
                break ;
            }
        }
    }
    
    /**
     * クラスパス情報取得.
     * <BR><BR>
     * クラスパス情報を取得します。
     * <BR>
     * @return String[] 取得されたクラスパス情報が格納されています。
     */
    public static final String[] getClassPathByStrings()
    {
        int i ;
        int len ;
        
        ArrayList lst = null ;
        String[] ret = null ;
        
        try{
            
            lst = UtilCom.getClassPath() ;
            if( lst != null ){
                len = lst.size() ;
                if( len > 0 ){
                    ret = new String[ len ] ;
                    for( i = 0 ; i < len ; i ++ ){
                        ret[ i ] = ( String )lst.get( i ) ;
                    }
                }
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            if( lst != null ){
                lst.clear() ;
            }
            lst = null ;
        }
        
        return ret ;
    }
    
    /**
     * クラスパス情報取得.
     * <BR><BR>
     * クラスパス情報を取得します。
     * <BR>
     * @return ArrayList 取得されたクラスパス情報が格納されています。
     */
    public static final ArrayList getClassPath()
    {
        String check = null ;
        String pathCode = null ;
        
        pathCode = System.getProperty( "java.class.path" ) ;
        
        try{
            
            check = System.getProperty( "path.separator" ) ;
            return UtilCom.pauseString( pathCode,check ) ;
            
        }catch(InputException in){
            
            return null ;
            
        }finally{
            
            check = null ;
            pathCode = null ;
            
        }
    }
    
    /**
     * 対象の文字列に対する全角/半角情報取得.
     * <BR><BR>
     * 対象の文字列に対する全角/半角情報を取得します.
     * <BR>
     * @param code 対象の文字列を設定します.
     * @return int[] 全角/半角条件が返されます.
     * @exception InputException 入力例外.
     */
    public static final int[] getZenToHanString( String code )
        throws InputException
    {
        int i ;
        int len ;
        char one ;
        
        int[] ret = null ;
        
        if( code == null || ( len = code.length() ) <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        ret = new int[ len ] ;
        for( i = 0 ; i < len ; i ++ ){
            one = code.charAt( i ) ;
            ret[ i ] = ( one <= 0x000000ff ) ? 1 : 2 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象名の拡張子を取得.
     * <BR><BR>
     * 対象名の拡張子を取得します.
     * <BR>
     * @param name 対象のファイル名を設定します.
     * @return String 拡張子情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final String getPlusName( String name )
        throws InputException
    {
        int pnt ;
        int bef ;
        
        String ret = null ;
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        for( pnt = 0,bef = -1 ;; ){
            if( ( pnt = name.indexOf( ".",pnt ) ) == -1 ){
                break ;
            }
            bef = pnt ;
            pnt += 1 ;
        }
        
        if( bef != -1 ){
            ret = name.substring( bef+1 ) ;
        }
        
        
        return ( ret != null && ret.length() > 0 ) ? ret.trim().toLowerCase() : null ;
    }
    
    /**
     * 指定した文字情報が等しいかチェックします。
     * <BR><BR>
     * @param org 比較対象元の文字情報を指定します。
     * @param src 比較対象先の文字情報を指定します。
     * @return boolean [true]が返された場合、情報は一致します.
     */
    public static final boolean equalString(String org,String src)
    {
        if(org != null && src != null){
            return org.equalsIgnoreCase(src) ;
        }
        return false ;
    }
    
    /**
     * 指定文字情報を変換対象文字情報に置き換え.
     * <BR><BR>
     * 指定文字情報を変換対象文字情報に置き換えます.
     * また、この情報は１バイト系の変換しか行いません.
     * <BR>
     * @param code 指定文字情報（この情報が変換対象文字情報に置き換わります。)
     * @param src 変換対象の１文字情報
     * @param dest 変換対象の１文字列情報
     * @return String 変換結果の情報
     * @exception InputException 入力例外.
     */
    public static final String convertChar(String code,String src,String dest)
        throws InputException
    {
        int i ;
        int len ;
        
        char info ;
        char srcChr ;
        char destChr ;
        
        StringBuffer buf = null ;
        String ret ;
        
        if( code == null || src == null || dest == null){
            throw new InputException( "引数が不正です" ) ;
        }
        
        srcChr = src.charAt( 0 ) ;
        destChr = dest.charAt( 0 ) ;
        
        len = code.length() ;
        
        buf = new StringBuffer() ;
        
        for(i = 0 ; i < len ; i ++){
            
            info = code.charAt( i ) ;
            info = ( info == srcChr ) ? destChr : info ;
            buf.append( info ) ;
        }
        
        ret = buf.toString() ;
        buf = null ;
        
        return ret ;
    }
    
    /**
     * 先頭の文字チェックを行う。
     * <BR><BR>
     * 指定文字に対する先頭文字情報をチェックします.
     * <BR>
     * @param info チェックしたいデータ.
     * @param check チェック対象データ.
     * @return boolean [true]が返された場合、情報は一致します.
     * @exception InputException 入力例外.
     */
    public static final boolean checkFront( String info,String check)
        throws InputException
    {
        int i,j ;
        int len ;
        int chkLen ;
        boolean ret ;
        
        char spaceChr ;
        char tabChr ;
        char space2Chr ;
        char infoChr ;
        
        if(
            info == null || check == null ||
            ( len = info.length() ) <= 0 ||
            ( chkLen = check.length() ) <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        spaceChr = BaseDef.SPACE.charAt( 0 ) ;
        space2Chr = BaseDef.SPACE2.charAt( 0 ) ;
        tabChr = BaseDef.TAB.charAt( 0 ) ;
        
        chkLen = chkLen - 1 ;
        
        for(i = 0,j = 0,ret = false ;i < len ; i ++){
            
            infoChr = info.charAt( i ) ;
            
            if( infoChr == check.charAt( j ) ){
                
                if( j >= chkLen ){
                    
                    ret = true ;
                    break ;
                    
                }
                j ++ ;
                
            }else if(
                (
                    infoChr == spaceChr ||
                    infoChr == space2Chr ||
                    infoChr == tabChr
                ) && j == 0
            )
            {
                
                continue ;
                
            }else{
                
                ret = false ;
                break ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 指定した文字列の個数を取得.
     * <BR><BR>
     * 指定した文字列の個数を取得します.
     * <BR>
     * @param str チェック元の文字列を設定します.
     * @param check チェック対象の文字列を設定します.
     * @return int チェック対象の文字列個数が返されます.
     */
    public static final int checkCountByStirng( String str,String check ) {
        
        int len ;
        int pnt ;
        int ret ;
        
        if( str == null || str.length() <= 0 ||
            check == null || ( len = check.length() ) <= 0 ) {
            return 0 ;
        }
        
        for( ret = 0,pnt = 0 ;; ) {
            
            if( ( pnt = str.indexOf( check,pnt ) ) <= -1 ) {
                break ;
            }
            
            pnt += len ;
            ret ++ ;
            
        }
        
        return ret ;
        
    }
    
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @return ArrayList 区切られた情報が格納されています.
     * @exception InputException 入力例外.
     */
    public static final ArrayList pauseString(String str,String check)
        throws InputException
    {
        ArrayList ret = null ;
        
        ret = new ArrayList() ;
        
        UtilCom.pauseString( ret,str,check ) ;
        
        return ret ;
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param out 区切られた情報が格納されます.
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @exception InputException 入力例外.
     */
    public static final void pauseString(ArrayList out,String str,String check)
        throws InputException
    {
        int i,j ;
        int len ;
        int lenJ ;
        int s = -1 ;
        char strCode ;
        
        char[] checkCode = null ;
        String tmp = null ;
        
        if(
            out == null ||
            str == null || ( len = str.length() ) <= 0 ||
            check == null || check.length() <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        out.clear() ;
        
        lenJ = check.length() ;
        checkCode = new char[ lenJ ] ;
        check.getChars( 0,lenJ,checkCode,0 ) ;
        
        if( lenJ == 1 ){
            
            for( i = 0,s = -1 ; i < len ; i ++ ){
                
                strCode = str.charAt( i ) ;
                s = ( s == -1 ) ? i : s ;
                
                if( strCode == checkCode[ 0 ] ){
                    
                    if( s < i ){
                        tmp = str.substring( s,i ) ;
                        out.add(tmp) ;
                        tmp = null ;
                        s = -1 ;
                    }
                    else{
                        s = -1 ;
                    }
                    
                }
                
            }
            
        }
        else{
            
            for( i = 0,s = -1 ; i < len ; i ++ ){
                
                strCode = str.charAt( i ) ;
                s = ( s == -1 ) ? i : s ;
                
                for( j = 0 ; j < lenJ ; j ++ ){
                    if( strCode == checkCode[ j ] ){
                        
                        if( s < i ){
                            tmp = str.substring( s,i ) ;
                            out.add(tmp) ;
                            tmp = null ;
                            s = -1 ;
                        }
                        else{
                            s = -1 ;
                        }
                        
                        break ;
                        
                    }
                }
                
            }
            
        }
        
        if(s != -1){
            
            tmp = str.substring( s,len ) ;
            out.add(tmp) ;
            tmp = null ;
        }
        
        checkCode = null ;
        tmp = null ;
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param cote コーテーション対応であるか設定します.<BR>
     *             [true]を設定した場合、各コーテーション ( ",' ) で囲った情報内は
     *             区切り文字と判別しません.<BR>
     *             [false]を設定した場合、コーテーション対応を行いません.
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @return ArrayList 区切られた情報が格納されています.
     * @exception InputException 入力例外.
     */
    public static final ArrayList pauseString(boolean cote,String str,String check)
        throws InputException
    {
        ArrayList ret = null ;
        
        ret = new ArrayList() ;
        
        UtilCom.pauseString( ret,cote,false,str,check ) ;
        
        return ret ;
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param out 区切られた情報が格納されます.
     * @param cote コーテーション対応であるか設定します.<BR>
     *             [true]を設定した場合、各コーテーション ( ",' ) で囲った情報内は
     *             区切り文字と判別しません.<BR>
     *             [false]を設定した場合、コーテーション対応を行いません.
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @exception InputException 入力例外.
     */
    public static final void pauseString(ArrayList out,boolean cote,String str,String check)
        throws InputException
    {
        UtilCom.pauseString( out,cote,false,str,check ) ;
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param cote コーテーション対応であるか設定します.<BR>
     *             [true]を設定した場合、各コーテーション ( ",' ) で囲った情報内は
     *             区切り文字と判別しません.<BR>
     *             [false]を設定した場合、コーテーション対応を行いません.
     * @param mode コーテーション区切りでコーテーションが存在する場合、その情報も含むか否かを
     *             設定します.<BR>
     *             [true]を設定した場合、情報を含みます.<BR>
     *             [false]を設定した場合、情報を含みません.
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @return ArrayList 区切られた情報が格納されています.
     * @exception InputException 入力例外.
     */
    public static final ArrayList pauseString(boolean cote,boolean mode,String str,String check)
        throws InputException
    {
        ArrayList ret = null ;
        
        ret = new ArrayList() ;
        
        UtilCom.pauseString( ret,cote,mode,str,check ) ;
        
        return ret ;
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param out 区切られた情報が格納されます.
     * @param cote コーテーション対応であるか設定します.<BR>
     *             [true]を設定した場合、各コーテーション ( ",' ) で囲った情報内は
     *             区切り文字と判別しません.<BR>
     *             [false]を設定した場合、コーテーション対応を行いません.
     * @param coteFlg コーテーションが入っている場合に、コーテーションを範囲に含むか否かを
     *                設定します.<BR>
     *                [true]を設定した場合、コーテーション情報も範囲に含みます.<BR>
     *                [false]を設定した場合、コーテーション情報を範囲としません.
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.<BR>
     *              区切り対象文字を複数設定する事により、それらに対応した区切りとなります.
     * @exception InputException 入力例外.
     */
    public static final void pauseString(ArrayList out,boolean cote,boolean coteFlg,String str,String check)
        throws InputException
    {
        int i,j ;
        int len ;
        int lenJ ;
        int s = -1 ;
        
        char coteChr ;
        char nowChr ;
        char strCode ;
        
        char[] checkCode = null ;
        String tmp = null ;
        
        if( cote == false ){
            UtilCom.pauseString( out,str,check ) ;
        }
        else{
            
            if(
                out == null ||
                str == null || ( len = str.length() ) <= 0 ||
                check == null || check.length() <= 0
            )
            {
                throw new InputException( "引数が不正です" ) ;
            }
            
            out.clear() ;
            
            lenJ = check.length() ;
            checkCode = new char[ lenJ ] ;
            check.getChars( 0,lenJ,checkCode,0 ) ;
            
            if( lenJ == 1 ){
                
                for( i = 0,s = -1,coteChr = 0 ; i < len ; i ++ ){
                    
                    strCode = str.charAt( i ) ;
                    nowChr = strCode ;
                    s = ( s == -1 ) ? i : s ;
                    
                    if( coteChr == 0 ){
                        
                        if( nowChr == UtilCom.SNGLCOTE || nowChr == UtilCom.DBLLCOTE ){
                            
                            coteChr = nowChr ;
                            
                            if( s < i ){
                                tmp = str.substring( s,i ) ;
                                out.add(tmp) ;
                                tmp = null ;
                                s = -1 ;
                            }
                            else{
                                s = -1 ;
                            }
                            
                        }
                        else if( strCode == checkCode[ 0 ] ){
                            
                            if( s < i ){
                                tmp = str.substring( s,i ) ;
                                out.add(tmp) ;
                                tmp = null ;
                                s = -1 ;
                            }
                            else{
                                s = -1 ;
                            }
                            
                        }
                    }
                    else if( coteChr == nowChr ){
                        
                        coteChr = 0 ;
                        
                        if( s == i && coteFlg == true ){
                            out.add( new StringBuffer().append( strCode ).append( strCode ).toString() ) ;
                            s = -1 ;
                        }
                        else if( s < i ){
                            
                            if( coteFlg == true ){
                                tmp = str.substring( s-1,i+1 ) ;
                            }
                            else{
                                tmp = str.substring( s,i ) ;
                            }
                            
                            out.add(tmp) ;
                            tmp = null ;
                            s = -1 ;
                        }
                        else{
                            s = -1 ;
                        }
                        
                    }
                    
                }
            }
            else{
                
                for( i = 0,s = -1,coteChr = 0 ; i < len ; i ++ ){
                    
                    strCode = str.charAt( i ) ;
                    nowChr = strCode ;
                    s = ( s == -1 ) ? i : s ;
                    
                    if( coteChr == 0 ){
                        
                        if( nowChr == UtilCom.SNGLCOTE || nowChr == UtilCom.DBLLCOTE ){
                            
                            coteChr = nowChr ;
                            
                            if( s < i ){
                                tmp = str.substring( s,i ) ;
                                out.add(tmp) ;
                                tmp = null ;
                                s = -1 ;
                            }
                            else{
                                s = -1 ;
                            }
                            
                        }
                        else{
                            
                            for( j = 0 ; j < lenJ ; j ++ ){
                                
                                if( strCode == checkCode[ j ] ){
                                    
                                    if( s < i ){
                                        tmp = str.substring( s,i ) ;
                                        out.add(tmp) ;
                                        tmp = null ;
                                        s = -1 ;
                                    }
                                    else{
                                        s = -1 ;
                                    }
                                    
                                    break ;
                                    
                                }
                                
                            }
                            
                        }
                    }
                    else if( coteChr == nowChr ){
                        
                        coteChr = 0 ;
                        
                        if( s == i && coteFlg == true ){
                            out.add( new StringBuffer().append( strCode ).append( strCode ).toString() ) ;
                            s = -1 ;
                        }
                        else if( s < i ){
                            
                            if( coteFlg == true ){
                                tmp = str.substring( s-1,i+1 ) ;
                            }
                            else{
                                tmp = str.substring( s,i ) ;
                            }
                            
                            out.add(tmp) ;
                            tmp = null ;
                            s = -1 ;
                        }
                        else{
                            s = -1 ;
                        }
                        
                    }
                    
                }
            }
            
            if(s != -1){
                
                if( coteChr != 0 && coteFlg == true ){
                    tmp = str.substring( s-1,len ) + ( char )coteChr ;
                }
                else{
                    tmp = str.substring( s,len ) ;
                }
                
                out.add(tmp) ;
                tmp = null ;
                
            }
            
            checkCode = null ;
            tmp = null ;
            
        }
    }
    
    /**
     * チェック情報単位で情報を区切ります。
     * <BR><BR>
     * @param str 区切り対象の情報を設置します.
     * @param check 区切り対象の文字情報をセットします.
     * @return ArrayList 区切られた情報が格納されています.
     * @exception InputException 入力例外.
     */
    public static final ArrayList pauseBinary( byte[] str,byte[] check )
        throws InputException
    {
        int i,j ;
        int s ;
        int len ;
        int chkLen ;
        int tmpLen ;
        
        byte[] tmp = null ;
        ArrayList ret = null ;
        
        if(
            str == null || ( len = str.length ) <= 0 ||
            check == null || ( chkLen = check.length ) <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        
        if( len <= chkLen ){
            
            tmp = new byte[ len ] ;
            System.arraycopy( str,0,tmp,0,len ) ;
            ret.add( tmp ) ;
            
        }
        else{
            
            for( i = 0,j = 0,s = -1 ; i < len ; i ++ ){
                
                if( s == -1 ){
                    s = i ;
                }
                
                if( str[ i ] == check[ j ] ){
                    
                    if( j >= chkLen ){
                        
                        tmpLen = ( i - s ) - chkLen ;
                        tmp = new byte[ tmpLen ] ;
                        System.arraycopy( str,s,tmp,0,tmpLen ) ;
                        ret.add( tmp ) ;
                        tmp = null ;
                        
                        s = -1 ;
                        j = 0 ;
                        
                    }
                    else{
                        j ++ ;
                    }
                    
                }
                else{
                    j = 0 ;
                }
                
            }
            
            if( s != -1 ){
                
                tmpLen = ( len - 1 ) - s + 1 ;
                tmp = new byte[ tmpLen ] ;
                System.arraycopy( str,s,tmp,0,tmpLen ) ;
                ret.add( tmp ) ;
                tmp = null ;
                
            }
            
        }
        
        return ret ;
    }
    
    /**
     * バイナリ内の指定条件を検索.
     * <BR><BR>
     * @param src 対象のバイナリを設定します.
     * @param code チェック対象のコードを設定します.
     * @param offset チェック開始のオフセット値が返されます.
     * @return int チェック結果が返されます.<BR>
     *             [-1]が返された場合、指定条件に当てはまる内容は存在しません.
     */
    public static final int indexOfBinary( byte[] src,byte[] code,int offset )
    {
        int i,j ;
        int len ;
        int lenJ ;
        int ret ;
        
        if(
            src == null ||
            ( len = src.length ) <= 0 ||
            code == null ||
            ( lenJ = code.length ) <= 0 ||
            offset < 0 || offset > len
        )
        {
            return -1 ;
        }
        
        for( i = offset,ret = -1,j = 0 ; i < len ; i ++ ){
            
            if( src[ i ] == code[ j ] ){
                j ++ ;
                if( j >= lenJ ){
                    ret = i - lenJ ;
                    break ;
                }
            }
            else{
                j = 0 ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 改行コード区切り.
     * <BR><BR>
     * 指定文字列を改行単位で区切ります。
     * <BR>
     * @param str 改行コード区切り対象のバイナリコードを設定します。
     * @return ArrayList 区切られた改行コード単位でバイナリ情報が格納されています。
     * @exception InputException 入力例外
     */
    public static final ArrayList pauseEnter( String str )
        throws InputException
    {
        int i,s ;
        int len ;
        int enterLen ;
        String enter = null ;
        String tmp = null ;
        
        ArrayList ret = null ;
        
        if( str == null || ( len = str.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        enter = BaseDef.ENTER_SEPARATOR ;
        
        enterLen = enter.length() ;
        
        // 改行コードが1byteの場合
        // for unix or macintosh.
        if( enterLen == 1 ){
            
            char enterCode0 ;
            enterCode0 = enter.charAt( 0 ) ;
            
            for(i = 0,s = -1 ; i < len ; i ++){
                
                s = ( s == -1 ) ? i : s ;
                
                if( str.charAt( i ) == enterCode0 ){
                    
                    tmp = str.substring( s,i ) ;
                    ret.add( tmp ) ;
                    tmp = null ;
                    s = -1 ;
                    
                }
                
            }
            
            if(s != -1){
                
                tmp = str.substring( s,len ) ;
                ret.add(tmp) ;
                tmp = null ;
                
            }
            
            enter = null ;
            
            return ret ;
            
        }
        
        
        // 改行コードが2byteの場合
        // for windows.
        else if( enterLen == 2 ){
            
            char enterCode0 ;
            char enterCode1 ;
            enterCode0 = enter.charAt( 0 ) ;
            enterCode1 = enter.charAt( 1 ) ;
            
            for(i = 0,s = -1 ; i < len ; i ++){
                
                s = ( s == -1 ) ? i : s ;
                
                if( str.charAt( i ) == enterCode0 && i < len ){
                    
                    if( str.charAt( i + 1 )== enterCode1 ){
                        
                        tmp = str.substring( s,i ) ;
                        ret.add(tmp) ;
                        tmp = null ;
                        s = -1 ;
                        i ++ ;
                        
                    }
                    
                }
                
            }
            
            if(s != -1){
                
                tmp = str.substring( s,len ) ;
                ret.add(tmp) ;
                tmp = null ;
                
            }
            
            enter = null ;
            
            return ret ;
        }
        
        enter = null ;
        
        throw new InputException(
            "対象の改行情報は、サポート外です[" + enterLen +
            "]-enter.length"
        ) ;
    }
    
    /**
     * ￥コード単位で情報を区切ります。
     * <BR><BR>
     * @param str 区切り対象の情報を設置します。
     * @return ArrayList 区切られた情報が格納されています。
     * @exception InputException 入力例外
     */
    public static final ArrayList pauseYen( String str )
        throws InputException
    {
        int i ;
        int s = -1 ;
        int len ;
        char yenCode  ;
        
        String tmp = null ;
        ArrayList ret = null ;
        
        if( str == null || ( len = str.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        yenCode = BaseDef.YEN_CODE.charAt( 0 ) ;
        
        for(i = 0,s = -1 ; i < len ; i ++){
            
            s = ( s == -1 ) ? i : s ;
            
            if( str.charAt( i ) == yenCode ){
                
                tmp = str.substring( s,i ) ;
                ret.add(tmp) ;
                tmp = null ;
                s = -1 ;
                
            }
        }
        
        if( s != -1 ){
            
            tmp = str.substring( s,len ) ;
            ret.add(tmp) ;
            tmp = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * /(スラッシュ)コード単位で情報を区切ります。
     * <BR><BR>
     * @param str 区切り対象の情報を設置します。
     * @return ArrayList 区切られた情報が格納されています。
     * @exception InputException 入力例外
     */
    public static final ArrayList pauseSlash( String str )
        throws InputException
    {
        int i ;
        int s = -1 ;
        int len ;
        char slashCode ;
        
        String tmp = null ;
        ArrayList ret ;
        
        if( str == null || ( len = str.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        slashCode = BaseDef.SLASH_CODE.charAt( 0 ) ;
        
        for(i = 0,s = -1 ; i < len ; i ++){
            
            s = ( s == -1 ) ? i : s ;
            
            if(str.charAt( i ) == slashCode){
                
                tmp = str.substring( s,i ) ;
                ret.add(tmp) ;
                tmp = null ;
                s = -1 ;
                
            }
        }
        
        if(s != -1){
            
            tmp = str.substring( s,len ) ;
            ret.add(tmp) ;
            tmp = null ;
            
        }
        
        return ret ;
    }
    
    /**
     * シングルコーテーションおよび、ダブルコーテーションで囲まれた情報を取得します。
     * <BR><BR>
     * @param str 区切り対象の情報を設置します。
     * @return ArrayList 区切られた情報が格納されています。
     * @exception InputException 入力例外
     */
    public static final ArrayList pauseCoat( String str )
        throws InputException
    {
        int i ;
        int s = -1 ;
        int len ;
        boolean flg = false ;
        
        String tmp = null ;
        ArrayList ret = null ;
        
        char singleCoat ;
        char doubleCoat ;
        
        
        if( str == null || ( len = str.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        singleCoat = BaseDef.SINGLE_COAT.charAt( 0 ) ;
        doubleCoat = BaseDef.DOUBLE_COAT.charAt( 0 ) ;
        
        for(i = 0,s = -1 ; i < len ; i ++){
            
            if(
                (
                    str.charAt( i ) == singleCoat ||
                    str.charAt( i ) == doubleCoat
                ) && s == -1
            )
            {
                
                flg = ( str.charAt( i ) == singleCoat ) ? false : true ;
                s = i + 1 ;
                
                continue ;
                
            }else if(s == -1){
                
                continue ;
                
            }
            else if(
                ( str.charAt( i ) == singleCoat && flg == false ) ||
                ( str.charAt( i ) == doubleCoat && flg == true )
            )
            {
                
                if( (i - s) == 0){
                    
                    ret.add( new String( "" ) ) ;
                    s = -1 ;
                    
                }else{
                    
                    tmp = str.substring( s,i ) ;
                    ret.add( tmp ) ;
                    tmp = null ;
                    s = -1 ;
                    
                }
            }
        }
        
        return ret ;
    }
    
    /**
     * 情報の前後に存在するスペース、タブ関連情報を削除します。
     * <BR><BR>
     * @param str 削除対象の情報を設定します。
     * @return String 削除された情報が格納されます。
     */
    public static final String trimPlus( String str ) {
        
        int i ;
        int s,e ;
        int len ;
        
        String ret = null ;
        
        char spaceCode ;
        char tabCode ;
        char space2Code ;
        
        char now ;
        
        if( str == null || ( len = str.length() ) <= 0){
            return "" ;
        }
        
        spaceCode = BaseDef.SPACE.charAt( 0 ) ;
        tabCode = BaseDef.TAB.charAt( 0 ) ;
        space2Code = BaseDef.SPACE2.charAt( 0 ) ;
        
        for(i = 0,s = -1 ; i < len ; i ++){
            
            now = str.charAt( i ) ;
            
            if(
                now != spaceCode &&
                now != tabCode &&
                now != space2Code &&
                now != 0
            )
            {
                
                s = i ;
                break ;
                
            }
            
        }
        
        for(i = len - 1,e = -1 ; i >= 0 ; i --){
            
            now = str.charAt( i ) ;
            
            if(
                now != spaceCode &&
                now != tabCode &&
                now != space2Code &&
                now != 0
            )
            {
                
                e = i ;
                break ;
                
            }
            
        }
        
        if( ( s == -1 && e == -1 ) ){
            if( s == -1 && e == -1 ){
                return "" ;
            }else{
                return "" ;
            }
        }
        s = ( s == -1 ) ? 0 : s ;
        e = ( e == -1 ) ? len-1 : e ;
        
        ret = str.substring( s,e+1 ) ;
        return ret ;
    }
    
    /**
     * 対象文字列を指定文字数単位で区切ります.
     * <BR><BR>
     * 対象文字列を指定文字数単位で区切り、改行を入れます.
     * <BR>
     * @param message 区切り対象メッセージを設定します.
     * @param size 区切り単位を設定します.
     * @return String 区切られた文字列が返されます.
     * @exception InputException 入力例外.
     */
    public static final String catString( String message,int size )
        throws InputException
    {
        int i ;
        int st ;
        int ed ;
        int msgLen ;
        int len ;
        int etcLen ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( message == null || size <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        msgLen = message.length() ;
        len = msgLen / size ;
        etcLen = msgLen % size ;
        
        buf = new StringBuffer( msgLen + ( len * 2 ) ) ;
        
        for( i = 0,st = 0,ed = size ; i < len ; i ++ ){
            
            buf.append( message.substring( st,ed ) ) ;
            buf.append( BaseDef.ENTER_SEPARATOR ) ;
            
            st = ed ;
            ed += size ;
            
        }
        
        if( etcLen != 0 ){
            buf.append( message.substring( msgLen - etcLen,msgLen ) ) ;
        }
        
        ret = buf.toString() ;
        buf = null ;
        
        return ret ;
    }
    
    /**
     * 指定コードチェックを行います.
     * <BR><BR>
     * 指定文字列に対して、指定した文字コードが含まれているか
     * チェックします.
     * <BR>
     * @param str チェック対象の文字コードを設定します.
     * @param chk チェックを行う文字情報を指定します.
     * @return int チェック結果が返されます.
     * @exception InputException 入力例外.
     */
    public static final int checkString( String str,String chk )
        throws InputException
    {
        int i,j ;
        int s,e ;
        int len ;
        int chkLen ;
        int ret ;
        
        if(
            str == null || chk == null ||
            ( len = str.length() ) <= 0 ||
            ( chkLen = chk.length() ) <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        for( i = 0,j = 0,s = -1,ret = 0 ; i < len ; i ++ ){
            
            s = ( s == -1 ) ? i : s ;
            if( str.charAt( i ) == chk.charAt( j ) ){
                
                if( j >= ( chkLen - 1 ) ){
                    
                    e = i - chkLen ;
                    
                    if( e <= 0 ){
                        ret ++ ;
                        s = -1 ;
                        j = 0 ;
                        continue ;
                    }
                    
                    s = -1 ;
                    j = 0 ;
                    ret ++ ;
                    
                }else{
                    
                    j ++ ;
                    
                }
            }else{
                j = 0 ;
            }
            
        }
        
        return ret ;
    }
    
    /**
     * 指定コードを置き換えます.
     * <BR><BR>
     * 指定文字列に対して、変更対象の文字列を指定して
     * 変更後の文字列に置き換えます.
     * <BR>
     * @param str 変更を行う文字列を指定します.
     * @param src 変更対象の文字列を指定します.<BR>
     *            変換条件は複数してい可能です.
     * @param dest 変更後の文字列を指定します.<BR>
     *             変換結果は１文字しか指定出来ません.
     * @return String 変更された文字列が返されます.
     * @exception InputException 入力例外.
     */
    public static final String changeStringByOneDest( String str,String src,char dest )
        throws InputException
    {
        int i ;
        int len ;
        int pnt ;
        char chk ;
        char[] chrs = null ;
        ArrayList lst = null ;
        String ret = null ;
        
        if(
            str == null || str.length() <= 0 ||
            src == null || ( len = src.length() ) <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        lst = new ArrayList() ;
        
        for( i = 0,pnt = 0 ; i < len ; i ++ ){
            
            chk = src.charAt( i ) ;
            
            for( pnt = 0 ;; ){
                
                if( ( pnt = str.indexOf( chk,pnt ) ) != -1 ){
                    lst.add( new Integer( pnt ) ) ;
                }
                else{
                    break ;
                }
                
                pnt ++ ;
                
            }
            
        }
        
        len = lst.size() ;
        if( len > 0 ){
            chrs = str.toCharArray() ;
            for( i = 0 ; i < len ; i ++ ){
                chrs[ ( ( Integer )lst.get( i ) ).intValue() ] = dest ;
            }
            
            ret = new String( chrs ) ;
        }
        else{
            ret = str ;
        }
        
        if( lst != null ){
            lst.clear() ;
        }
        
        return ret ;
    }
    
    /**
     * 指定コードを置き換えます.
     * <BR><BR>
     * 指定文字列に対して、変更対象の文字列を指定して
     * 変更後の文字列に置き換えます.
     * <BR>
     * @param str 変更を行う文字列を指定します.
     * @param src 変更対象の文字列を指定します.
     * @param dest 変更後の文字列を指定します.
     * @return String 変更された文字列が返されます.
     * @exception InputException 入力例外.
     */
    public static final String changeString( String str,String src,String dest )
        throws InputException
    {
        int i ;
        int len ;
        int pnt ;
        int end ;
        int bef ;
        int srcLen ;
        int destLen ;
        
        StringBuffer buf = null ;
        
        if(
            str == null || str.length() <= 0 ||
            src == null || ( srcLen = src.length() ) <= 0 ||
            dest == null || ( destLen = dest.length() ) <= 0
        )
        {
            throw new InputException( "引数は不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        pnt = str.indexOf( src ) ;
        if( pnt == -1 ){
            return str ;
        }
        else if( pnt == 0 ){
            buf.append( dest ) ;
            end = srcLen ;
        }
        else{
            buf.append( str.substring( 0,pnt ) ) ;
            buf.append( dest ) ;
            end = pnt + srcLen ;
        }
        
        for( pnt += srcLen ;; ){
            if( ( pnt = str.indexOf( src,pnt ) ) == -1 ){
                buf.append( str.substring( end ) ) ;
                break ;
            }
            if( pnt == end ){
                buf.append( dest ) ;
            }
            else{
                buf.append( str.substring( end,pnt ) ) ;
                buf.append( dest ) ;
            }
            
            end = pnt + srcLen ;
            pnt += srcLen ;
            
        }
        
        return buf.toString() ;
    }
    
    /**
     * 指定文字情報を指定数分作成.
     * <BR><BR>
     * 指定文字情報を指定数分作成します.
     * <BR>
     * @param code 作成対象の文字情報を指定します.
     * @param size 作成数を指定します.
     * @return String 作成された情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final String getCodes( String code,int size )
        throws InputException
    {
        int i ;
        
        char addCode ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( code == null || code.length() <= 0 || size <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        buf = new StringBuffer() ;
        
        addCode = code.charAt( 0 ) ;
        for( i = 0 ; i < size ; i ++ ){
            buf.append( addCode ) ;
        }
        
        ret = buf.toString() ;
        buf = null ;
        return ret ;
    }
    
    /**
     * 指定文字情報で指定文字数分、文字内容を埋めます.
     * <BR><BR>
     * 指定された文字情報に対して、指定文字数分文字情報を埋めます.
     * これにより、対象の文字列が、指定文字数分となります.
     * また、文字数は全角文字を２文字分、半角文字を１文字として処理します.
     * <BR>
     * @param mode 文字設定モードを定義します.<BR>
     *             [true]を設定した場合、先頭の位置に対して情報を埋めます.<BR>
     *             [false]を設定した場合、最後の位置に対して情報を埋めます.
     * @param info 埋める対象の情報を設定します.
     * @param param 埋める文字情報を指定します.
     * @param size 結果の文字列情報長が返されます.
     * @return String 埋められた文字情報が返されます.
     * @exception InputException 入力例外.
     * @exception ReadException 読み込み例外.
     */
    public static final String setLengthToFilledBinary(
        boolean mode,String info,String param,int size)
        throws InputException,ReadException
    {
        int i ;
        int len ;
        int strLen ;
        int byteLen ;
        
        int[] codeLen = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        if( 
            info == null || ( strLen = info.length() ) < 0 ||
            param == null || param.length() <= 0 ||
            size <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        try{
            
            buf = new StringBuffer() ;
            byteLen = ( (byte[])info.getBytes( UtilCom.CHECK_SJIS_CHARSET ) ).length ;
            
            // バイナリ長より、結果文字列長が小さい場合.
            if( byteLen >= size ){
                
                int rem ;
                int cnt ;
                
                codeLen = UtilCom.getZenToHanString( info ) ;
                len = codeLen.length ;
                
                // 先頭の位置に対象情報を埋める場合
                if( mode == true ){
                    
                    for( i = len - 1,cnt = 0 ; i >= 0 ; i -- ){
                        
                        cnt += codeLen[ i ] ;
                        if( cnt >= size ){
                            
                            rem = cnt - size ;
                            if( rem == 1 ){
                                
                                buf.append( UtilCom.ADD_REM_CODE ) ;
                                buf.append( info.substring( i+1 ) ) ;
                                
                            }else{
                                
                                buf.append( info.substring( i ) ) ;
                                
                            }
                            break ;
                        }
                    }
                    
                // 最後の位置に対象情報を埋める場合.
                }else{
                    
                    for( i = 0 ,cnt = 0 ; i < len ; i++ ){
                        
                        cnt += codeLen[ i ] ;
                        if( cnt >= size ){
                            
                            rem = cnt - size ;
                            if( rem == 1 ){
                                
                                buf.append( info.substring( 0,i ) ) ;
                                buf.append( UtilCom.ADD_REM_CODE ) ;
                                
                            }else{
                                
                                buf.append( info.substring( 0,i+1 ) ) ;
                                
                            }
                            break ;
                        }
                    }
                    
                }
                
            // バイナリ長より、結果文字列長が大きい場合.
            }else{
                
                int addLen ;
                
                codeLen = UtilCom.getZenToHanString( param ) ;
                addLen = size - byteLen ;
                
                // 追加情報に対して余りが存在する場合.
                if( addLen % codeLen[ 0 ] != 0 ){
                    
                    addLen = addLen / codeLen[ 0 ] ;
                    
                    // 先頭の位置に対象情報を埋める場合
                    if( mode == true ){
                        
                        buf.append( UtilCom.ADD_REM_CODE ) ;
                        if( addLen != 0 ){
                            buf.append( UtilCom.getCodes( param,addLen ) ) ;
                        }
                        buf.append( info ) ;
                        
                    // 最後の位置に対象情報を埋める場合.
                    }else{
                        
                        buf.append( info ) ;
                        if( addLen != 0 ){
                            buf.append( UtilCom.getCodes( param,addLen ) ) ;
                        }
                        buf.append( UtilCom.ADD_REM_CODE ) ;
                        
                    }
                    
                // 追加情報に対して余り情報が存在しない場合.
                }else{
                    
                    addLen = addLen / codeLen[ 0 ] ;
                    
                    // 先頭の位置に対象情報を埋める場合
                    if( mode == true ){
                        
                        buf.append( UtilCom.getCodes( param,addLen ) ) ;
                        buf.append( info ) ;
                        
                    // 最後の位置に対象情報を埋める場合.
                    }else{
                        
                        buf.append( info ) ;
                        buf.append( UtilCom.getCodes( param,addLen ) ) ;
                        
                    }
                }
            }
            
            ret = buf.toString() ;
            return ret ;
            
        }catch( UnsupportedEncodingException ue ){
            throw new ReadException( ue ) ;
        }catch( InputException in ){
            throw in ;
        }finally{
            
            buf = null ;
            codeLen = null ;
            
        }
        
    }
    
    /**
     * 指定文字情報で指定文字数分、文字内容を埋めます.
     * <BR><BR>
     * 指定された文字情報に対して、指定文字数分文字情報を埋めます.
     * これにより、対象の文字列が、指定文字数分となります.
     * また、文字数は半角文字を１文字として処理します.
     * <BR>
     * @param mode 文字設定モードを定義します.<BR>
     *             [true]を設定した場合、先頭の位置に対して情報を埋めます.<BR>
     *             [false]を設定した場合、最後の位置に対して情報を埋めます.
     * @param info 埋める対象の情報を設定します.
     * @param param 埋める文字情報を指定します.
     * @param size 結果の文字列情報長が返されます.
     * @return String 埋められた文字情報が返されます.
     * @exception InputException 入力例外.
     */
    public static final String setLengthToFilledString(
        boolean mode,String info,String param,int size)
        throws InputException
    {
        int i ;
        int len ;
        int strLen ;
        
        StringBuffer buf = null ;
        String ret = null ;
        
        if( 
            info == null || ( strLen = info.length() ) < 0 ||
            param == null || param.length() <= 0 ||
            size <= 0
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
                
        try{
            
            buf = new StringBuffer() ;
            
            // 文字列長より、結果文字列長が小さい場合.
            if( strLen >= size ){
                
                // 先頭の位置に対象情報を埋める場合
                if( mode == true ){
                    
                    buf.append( info.substring( strLen - size ) ) ;
                    
                // 最後の位置に対象情報を埋める場合.
                }else{
                    
                    buf.append( info.substring( 0,size ) ) ;
                    
                }
                
            // 文字列長より、結果文字列長が大きい場合.
            }else{
                
                // 先頭の位置に対象情報を埋める場合
                if( mode == true ){
                    
                    buf.append( UtilCom.getCodes( param,size-strLen ) ) ;
                    buf.append( info ) ;
                    
                // 最後の位置に対象情報を埋める場合.
                }else{
                    
                    buf.append( info ) ;
                    buf.append( UtilCom.getCodes( param,size-strLen ) ) ;
                    
                }
                
            }
            
            ret = buf.toString() ;
            return ret ;
            
        }catch( InputException in ){
            throw in ;
        }finally{
            
            buf = null ;
            
        }
        
    }

    
    /**
     * 指定文字情報を検索して、開始位置を取得.
     * <BR><BR>
     * 指定文字情報を検索して、開始位置を取得します.
     * <BR>
     * @param info 検索対象の文字列を指定します.
     * @param search 検索を行う文字列を指定します.
     * @param offset 検索開始位置を指定します.
     * @return int 検索結果の開始位置が返されます.<BR>
     *             情報が検索できなかった場合、[-1]が返されます.
     * @exception InputException 入力例外.
     */
    public static final int searchString( String info,String search,int offset )
        throws InputException
    {
        int i,j ;
        int len ;
        int searchLen ;
        int ret ;
        
        char chkChr ;
        char searchChr ;
        
        if(
            info == null || search == null ||
            ( len = info.length() ) <= 0 ||
            ( searchLen = search.length() ) <= 0 ||
            offset < 0 || offset + searchLen >= len
        )
        {
            throw new InputException( "引数が不正です" ) ;
        }
        
        searchChr = search.charAt( 0 ) ;
        
        for( i = offset,j = -1,ret = -1 ; i < len ; i ++ ){
            
            chkChr = info.charAt( i ) ;
            
            if( j == -1 && chkChr == searchChr ){
                
                ret = i ;
                j = 0 ;
                
                i -= 1 ;
                continue ;
                
            }else if( j != -1 && chkChr == search.charAt( j ) ){
                
                j += 1 ;
                if( j >= searchLen ){
                    break ;
                }
                
            }else{
                
                j = -1 ;
                ret = -1 ;
                
            }
        }
        
        return ret ;
    }
    
    /**
     * 指定されたString情報（CSV形式）の情報をカンマ単位で区切ります。
     * <BR><BR>
     * @param info カンマ編集された情報を格納します。
     * @return ArrayList カンマ単位で区切った内容が格納されています。
     * @exception InputException 入力例外
     */
    public static final ArrayList sawCSV(String info)
        throws InputException
    {
        int i ;
        int len ;
        
        int count = 0 ;
        int getCount = 0 ;
        
        int startPoint = -1 ;
        int endPoint = -1 ;
        
        char pauseCode = 0 ;
        
        ArrayList ret = null ;
        
        pauseCode = ",".charAt( 0 ) ;
        
        // 引数チェック
        if( info == null || ( len = info.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        ret = new ArrayList() ;
        
        // カンマチェック
        for( i = 0,count = 0;i < info.length() ; i ++ ){
            
            count += ( pauseCode == info.charAt( i ) ) ? 1 : 0 ;
            
        }
        
        // 取得したカウント数が０の場合
        if(count == 0){
            
            ret.add( new String( info ) ) ;
            return ret ;
            
        }
        
        // カンマ数分リターン値の情報を生成
        getCount = count + 1 ;
        
        // カンマの区切りまでの情報を格納
        for(i = 0,count = 0,startPoint = 0;i < info.length() ; i ++){
            
            // カンマが存在する場合
            if( pauseCode == info.charAt( i ) ){
                
                // カンマを除いてコピー
                endPoint = (i + 1) - 1 ;
                
                // 情報が存在しない場合
                if( endPoint == startPoint ){
                    
                    ret.add( new String( "" ) ) ;
                    
                }else{
                    
                    // 情報のコピー
                    ret.add( new String( info.substring( startPoint,endPoint ) ) ) ;
                    
                }
                
                // 次のスタート位置を設定(カンマを飛ばした次の情報位置)
                startPoint = endPoint + 1 ;
                
                // カウント＋１
                count ++ ;
            }
            
        }
        // 一番最後の情報を処理する
        
        // カンマを除いてコピー
        endPoint = ( ( info.length() - 1 ) + 1 ) - 1 + 1 ;
        
        // 情報が存在しない場合
        if(endPoint == startPoint){
            
            ret.add( new String( "" ) ) ;
            
        }else{
            
            // 情報のコピー
            ret.add( new String( info.substring( startPoint,endPoint ) ) ) ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 現在タイムスタンプ情報の取得.
     * <BR><BR>
     * 現在タイムスタンプ情報を取得します。
     * <BR>
     * @return ArrayList 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final ArrayList getNowTimestamp()
    {
        ArrayList ret = null ;
        String[] data = null ;
        
        data = UtilCom.getNowTimestampByString() ;
        
        ret = new ArrayList() ;
        ret.add( data[ YEAR ] ) ;
        ret.add( data[ MONTH ] ) ;
        ret.add( data[ DAY ] ) ;
        ret.add( data[ HOUR ] ) ;
        ret.add( data[ MINUTE ] ) ;
        ret.add( data[ SECOND ] ) ;
        ret.add( data[ MILLISECOND ] ) ;
        ret.add( data[ WEEK_JP ] ) ;
        ret.add( data[ WEEK_ENG ] ) ;
        
        data[ YEAR ] = null ;
        data[ MONTH ] = null ;
        data[ DAY ] = null ;
        data[ HOUR ] = null ;
        data[ MINUTE ] = null ;
        data[ SECOND ] = null ;
        data[ MILLISECOND ] = null ;
        data[ WEEK_JP ] = null ;
        data[ WEEK_ENG ] = null ;
        data = null ;
        
        return ret ;
    }
    
    /**
     * 現在の時間をInteger(4byte)で表した情報として取得.
     * <BR><BR>
     * 現在の時間をInteger(4byte)で表した情報として取得します.<BR>
     * ただし、秒情報までは含まれません.
     * <BR>
     * @return int 現在の時間をInteger(4byte)で表した情報が返されます.<BR>
     *             ただし、秒情報までは含まれません.
     */
    public static final int getTimeByInteger()
    {
        int y,m,d,th,tm ;
        
        Calendar now = Calendar.getInstance() ;
        
        synchronized( now ){
            y = now.get(Calendar.YEAR) ;
            m = now.get(Calendar.MONTH) ;
            d = now.get(Calendar.DAY_OF_MONTH) ;
            th = now.get(Calendar.HOUR_OF_DAY) ;
            tm = now.get(Calendar.MINUTE) ;
            
            return (
                ( y &    0x00000fff ) |
                ( ( m &  0x0000000f ) << 12 ) |
                ( ( d &  0x0000001f ) << 16 ) |
                ( ( th & 0x0000001f ) << 21 ) |
                ( ( tm & 0x0000003f ) << 26 )
            ) ;
        }
    }
    
    /**
     * [UtilCom.getTimeByInteger()]で生成されたInt条件から、日付情報を復元.
     * <BR><BR>
     * [UtilCom.getTimeByInteger()]で生成されたInt条件から、日付情報を復元します.
     * <BR>
     * @param time [UtilCom.getTimeByInteger()]で生成されたInt情報を設定します.
     * @return Calendar 対象の日付情報が返されます.
     */
    public static final Calendar getIntByCalendar( int time )
    {
        Calendar ret = Calendar.getInstance() ;
        
        synchronized( ret ){
            ret.clear() ;
            ret.set(Calendar.YEAR,( time &             0x00000fff ) ) ;
            ret.set(Calendar.MONTH,( ( ( time &        0x0000f000 ) >> 12 ) & 0x0000000f ) ) ;
            ret.set(Calendar.DAY_OF_MONTH,( ( ( time & 0x001f0000 ) >> 16 ) & 0x0000001f ) ) ;
            ret.set(Calendar.HOUR_OF_DAY,( ( ( time &  0x03e00000 ) >> 21 ) & 0x0000001f ) ) ;
            ret.set(Calendar.MINUTE,( ( ( time &       0xfc000000 ) >> 26 ) & 0x0000003f ) ) ;
        }
        
        return ret ;
    }
    
    /**
     * [UtilCom.getTimeByInteger()]で生成されたInt条件から、日付情報を復元.
     * <BR><BR>
     * [UtilCom.getTimeByInteger()]で生成されたInt条件から、日付情報を復元します.
     * <BR>
     * @param time [UtilCom.getTimeByInteger()]で生成されたInt情報を設定します.
     * @return Date 対象の日付情報が返されます.
     */
    public static final Date getIntByDate( int time )
    {
        Calendar cl = null ;
        Date ret = null ;
        
        synchronized( ( cl = UtilCom.getIntByCalendar( time ) ) ){
            if( cl != null ){
                ret = cl.getTime() ;
                cl = null ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 現在タイムスタンプ情報の取得.
     * <BR><BR>
     * 現在タイムスタンプ情報を取得します。
     * <BR>
     * @return String[] 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final String[] getNowTimestampByString()
    {
        String[] ret = null ;
        
        ret = new String[ TIMESTAMP_STRING_LENGTH ] ;
        
        UtilCom.getNowTimestampByString( ret ) ;
        
        return ret ;
    }
    
    /**
     * 現在タイムスタンプ情報の取得.
     * <BR><BR>
     * 現在タイムスタンプ情報を取得します。
     * <BR>
     * @param out 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final void getNowTimestampByString( String[] out )
    {
        Calendar now = null ;
        
        if( out == null || out.length != TIMESTAMP_STRING_LENGTH ){
            return ;
        }
        
        now = Calendar.getInstance() ;
        synchronized( now ){
            
            out[ YEAR ] = String.valueOf(now.get(Calendar.YEAR)) ;
            out[ MONTH ] = String.valueOf(now.get(Calendar.MONTH)+1) ;
            out[ DAY ] = String.valueOf(now.get(Calendar.DAY_OF_MONTH)) ;
            out[ HOUR ] = String.valueOf(now.get(Calendar.HOUR_OF_DAY)) ;
            out[ MINUTE ] = String.valueOf(now.get(Calendar.MINUTE)) ;
            out[ SECOND ] = String.valueOf(now.get(Calendar.SECOND)) ;
            out[ MILLISECOND ] = String.valueOf(now.get(Calendar.MILLISECOND)) ;
            
            switch(now.get(Calendar.DAY_OF_WEEK)){
                case Calendar.SUNDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_SUNDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_SUNDAY ;
                    break ;
                case Calendar.MONDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_MONDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_MONDAY ;
                    break ;
                case Calendar.TUESDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_TUESDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_TUESDAY ;
                    break ;
                case Calendar.WEDNESDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_WEDNESDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_WEDNESDAY ;
                    break ;
                case Calendar.THURSDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_THURSDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_THURSDAY ;
                    break ;
                case Calendar.FRIDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_FRIDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_FRIDAY ;
                    break ;
                case Calendar.SATURDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_SATURDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_SATURDAY ;
                    break ;
                default :
                    out[ WEEK_JP ] = "?" ;
                    out[ WEEK_ENG ] = "?" ;
                    break ;
            }
        }
    }
    
    /**
     * 対象タイムスタンプ情報の取得.
     * <BR><BR>
     * 対象タイムスタンプ情報を取得します。
     * <BR>
     * @param code 変換対象のカレンダーオブジェクトを設定します.
     * @return ArrayList 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final ArrayList getTimestamp( Calendar code )
    {
        ArrayList ret = null ;
        String[] data = null ;
        
        data = UtilCom.getTimestampByString( code ) ;
        
        ret = new ArrayList() ;
        ret.add( data[ YEAR ] ) ;
        ret.add( data[ MONTH ] ) ;
        ret.add( data[ DAY ] ) ;
        ret.add( data[ HOUR ] ) ;
        ret.add( data[ MINUTE ] ) ;
        ret.add( data[ SECOND ] ) ;
        ret.add( data[ MILLISECOND ] ) ;
        ret.add( data[ WEEK_JP ] ) ;
        ret.add( data[ WEEK_ENG ] ) ;
        
        data[ YEAR ] = null ;
        data[ MONTH ] = null ;
        data[ DAY ] = null ;
        data[ HOUR ] = null ;
        data[ MINUTE ] = null ;
        data[ SECOND ] = null ;
        data[ MILLISECOND ] = null ;
        data[ WEEK_JP ] = null ;
        data[ WEEK_ENG ] = null ;
        data = null ;
        
        return ret ;
    }
    
    /**
     * 対象タイムスタンプ情報の取得.
     * <BR><BR>
     * 対象タイムスタンプ情報を取得します。
     * <BR>
     * @param code 変換対象のカレンダーオブジェクトを設定します.
     * @return String[] 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final String[] getTimestampByString( Calendar code )
    {
        String[] ret = null ;
        
        ret = new String[ TIMESTAMP_STRING_LENGTH ] ;
        
        UtilCom.getTimestampByString( code,ret ) ;
        
        return ret ;
    }
    
    /**
     * 対象タイムスタンプ情報の取得.
     * <BR><BR>
     * 対象タイムスタンプ情報を取得します。
     * <BR>
     * @param code 変換対象のカレンダーオブジェクトを設定します.
     * @param out 現在のタイムスタンプ情報が格納されています。<BR>
     *   要素１から順番に、1=年　2=月　3=日　4=時　5=分
     *   6=秒　7=ミリ秒　8=曜日(日本語) 9=曜日(英語)
     */
    public static final void getTimestampByString( Calendar code,String[] out )
    {
        
        if( out == null || out.length != TIMESTAMP_STRING_LENGTH ){
            return ;
        }
        
        synchronized( code ){
            out[ YEAR ] = String.valueOf(code.get(Calendar.YEAR)) ;
            out[ MONTH ] = String.valueOf(code.get(Calendar.MONTH)+1) ;
            out[ DAY ] = String.valueOf(code.get(Calendar.DAY_OF_MONTH)) ;
            out[ HOUR ] = String.valueOf(code.get(Calendar.HOUR_OF_DAY)) ;
            out[ MINUTE ] = String.valueOf(code.get(Calendar.MINUTE)) ;
            out[ SECOND ] = String.valueOf(code.get(Calendar.SECOND)) ;
            out[ MILLISECOND ] = String.valueOf(code.get(Calendar.MILLISECOND)) ;
            
            switch(code.get(Calendar.DAY_OF_WEEK)){
                case Calendar.SUNDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_SUNDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_SUNDAY ;
                    break ;
                case Calendar.MONDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_MONDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_MONDAY ;
                    break ;
                case Calendar.TUESDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_TUESDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_TUESDAY ;
                    break ;
                case Calendar.WEDNESDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_WEDNESDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_WEDNESDAY ;
                    break ;
                case Calendar.THURSDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_THURSDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_THURSDAY ;
                    break ;
                case Calendar.FRIDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_FRIDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_FRIDAY ;
                    break ;
                case Calendar.SATURDAY :
                    out[ WEEK_JP ] = DateTime.WEEK_JP_SATURDAY ;
                    out[ WEEK_ENG ] = DateTime.WEEK_ENG_SATURDAY ;
                    break ;
                default :
                    out[ WEEK_JP ] = "?" ;
                    out[ WEEK_ENG ] = "?" ;
                    break ;
            }
        }
    }
    
    /**
     * 現在の日付情報を取得.
     * <BR><BR>
     * 現在の日付情報を取得します.
     * <BR>
     * @return java.sql.Date 現在の日付情報が返されます.
     */
    public static final java.sql.Date getNowDateToSql()
    {
        java.sql.Date ret = null ;
        
        ret = new java.sql.Date( System.currentTimeMillis() ) ;
        
        return ret ;
    }
    
    /**
     * 現在の時間情報を取得.
     * <BR><BR>
     * 現在の時間情報を取得します.
     * <BR>
     * @return java.sql.Time 現在の時間情報が返されます.
     */
    public static final java.sql.Time getNowTimeToSql()
    {
        java.sql.Time ret = null ;
        
        ret = new java.sql.Time( System.currentTimeMillis() ) ;
        
        return ret ;
    }
    
    /**
     * 現在の日付・時間情報を取得.
     * <BR><BR>
     * 現在の日付・時間情報を取得します.
     * <BR>
     * @return java.sql.Time 現在の日付・時間情報が返されます.
     */
    public static final java.sql.Timestamp getNowTimestampToSql()
    {
        java.sql.Timestamp ret = null ;
        
        ret = new java.sql.Timestamp( System.currentTimeMillis() ) ;
        
        return ret ;
    }
    
    /**
     * 数値チェック.
     * <BR><BR>
     * 指定情報に対して、数値情報であるかチェックします.
     * <BR>
     * @param info チェック対象の情報を指定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が帰ってきた場合、数値です.<BR>
     *                 [false]が帰ってきた場合、数値情報ではありません.
     * @exception InputException 入力例外.
     */
    public static final boolean checkStringByNumber( String info )
        throws InputException 
    {
        int i ;
        int len ;
        
        char chk ;
        boolean ret ;
        
        if( info == null || ( len = info.length() ) <= 0 ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        for( i = 0,ret = true ; i < len ; i ++ ){
            
            chk = info.charAt( i ) ;
            if(
                ( chk >= '0' && chk <= '9' ) ||
                chk == '.'
            )
            {
                continue ;
            }else{
                ret = false ;
                break ;
            }
        }
        
        return ret ;
    }
    
    /**
     * 指定バイナリコードを16進情報として、文字列変換.
     * <BR><BR>
     * 指定バイナリ情報を16進数文字列で変換します.
     * <BR>
     * @param binary 変換対象の文字列を指定します.
     * @return String 変換された16進情報が返されます.
     */
    public static final String convertBinaryTo16String( byte[] binary )
    {
        if( binary == null || binary.length <= 0 ) {
            return " " ;
        }
        return UtilCom.convertBinaryTo16String( binary,0,binary.length ) ;
    }
    
    /**
     * 対象のバイナリ情報を16進数に変換.
     * <BR><BR>
     * 対象のバイナリ情報を16進数情報に変換します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @param offset 対象のオフセット情報値を設定します.
     * @param length 対象の情報長を設定します.
     * @return String 変換された情報が返されます.
     */
    public static final String convertBinaryTo16String(
        byte[] binary,int offset,int length 
        )
    {
        if( binary == null || binary.length <= 0 ) {
            return " " ;
        }
        return UtilCom.convertBinaryTo16String(
            Resource.createBinResource( null,binary ),offset,length ) ;
    }
    
    /**
     * 対象のバイナリ情報を16進数に変換.
     * <BR><BR>
     * 対象のバイナリ情報を16進数情報に変換します.
     * <BR>
     * @param resource 対象のバイナリ情報を設定します.
     * @param offset 対象のオフセット情報値を設定します.
     * @param length 対象の情報長を設定します.
     * @return String 変換された情報が返されます.
     */
    public static final String convertBinaryTo16String(
       BinResource resource,int offset,int length 
        )
    {
        
        int i ;
        int len ;
        
        int tmp ;
        
        char[] cnv = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            
            cnv = UtilCom.convert10To16 ;
            buf = new StringBuffer() ;
            
            len = offset + length ;
            for( i = offset ; i < len ; i ++ ){
                
                tmp = ( int )( resource.get( i ) & 0x000000ff ) ;
                buf.append( cnv[ ( ( tmp & 0x000000f0 ) >> 4 ) ] ) ;
                buf.append( cnv[ ( tmp & 0x0000000f ) ] ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception e ) {
            ret = " " ;
        }finally{
            
            buf = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象のバイナリ情報を１６進数に変換.
     * <BR><BR>
     * 対象のバイナリ情報を１６進数情報に変換します.
     * <BR>
     * @param binary 対象のバイナリ情報を設定します.
     * @param rem 区切り情報データを設定します.
     * @param offset 対象のオフセット情報値を設定します.
     * @param length 対象の情報長を設定します.
     * @return String 変換された情報が返されます.
     */
    public static final String convertBinaryTo16String(
        byte[] binary,char rem,int offset,int length 
        )
    {
        if( binary == null || binary.length <= 0 ) {
            return " " ;
        }
        
        return UtilCom.convertBinaryTo16String(
            Resource.createBinResource( null,binary ),rem,0,binary.length ) ;
    }
    
    /**
     * 対象のバイナリ情報を１６進数に変換.
     * <BR><BR>
     * 対象のバイナリ情報を１６進数情報に変換します.
     * <BR>
     * @param resource 対象のバイナリ情報を設定します.
     * @param rem 区切り情報データを設定します.
     * @param offset 対象のオフセット情報値を設定します.
     * @param length 対象の情報長を設定します.
     * @return String 変換された情報が返されます.
     */
    public static final String convertBinaryTo16String(
        BinResource resource,char rem,int offset,int length 
        )
    {
        
        int i ;
        int len ;
        
        int tmp ;
        
        char[] cnv = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            
            cnv = UtilCom.convert10To16 ;
            buf = new StringBuffer() ;
            
            len = offset + length ;
            for( i = offset ; i < len ; i ++ ){
                
                if( i != offset ) {
                    buf.append( rem ) ;
                }
                
                tmp = ( int )( resource.get( i )  & 0x000000ff ) ;
                buf.append( cnv[ ( ( tmp & 0x000000f0 ) >> 4 ) ] ) ;
                buf.append( cnv[ ( tmp & 0x0000000f ) ] ) ;
                
            }
            
            ret = buf.toString() ;
            
        }catch( Exception e ) {
            ret = " " ;
        }finally{
            
            buf = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * [convertBinaryTo16String]で変換した内容をバイナリに戻す.
     * <BR><BR>
     * [convertBinaryTo16String]で変換した内容をバイナリに戻します.
     * <BR>
     * @param remFlg 区切り情報が設定されている場合[true]、
     *               されていない場合[false]を設定します.
     * @param string 変換対象のストリングコードを設定します.
     * @return byte[] 再変換された内容が返されます.
     * @exception InputException 入力例外.
     */
    public static final byte[] convert16StringToBinary(
        boolean remFlg,String string
    )
        throws InputException
    {
        
        int i ;
        int len ;
        int j ;
        int lenJ ;
        
        char code ;
        int codeA ;
        int codeB ;
        
        String tmp = null ;
        byte[] ret = null ;
        StringBuffer buf = null ;
        
        try{
        	
        	string = string.trim().toLowerCase() ;
            
            len = string.length() ;
            
            if( remFlg == true ){
                
                buf = new StringBuffer( len ) ;
                
                for( i = 0 ; i < len ; i ++ ){
                    
                    if( ( i % 3 )== 2 ){
                        continue ;
                    }
                    
                    code = string.charAt( i ) ;
                    buf.append( code ) ;
                    
                }
                
                tmp = buf.toString() ;
                
            }else{
                
                tmp = string ;
                
            }
            
            len = tmp.length() ;
            lenJ = len / 2 ;
            ret = new byte[ lenJ ] ;
            
            for( i = 0,j = 0 ; j < lenJ ; i +=2,j ++ ){
                
            	code = tmp.charAt( i ) ;
            	if( code >= '0' && code <= '9' ) {
            		codeA = code - '0' ;
            	}
            	else if( code >= 'a' && code <= 'f' ) {
            		codeA = ( code - 'a' ) + 10 ;
            	}
            	else {
            		throw new InputException(
            			"引数には、16進コード以外が含まれています(no:" +
            			i + " code:" + code + ")" ) ;
            	}
            	
            	code = tmp.charAt( i+1 ) ;
            	if( code >= '0' && code <= '9' ) {
            		codeB = code - '0' ;
            	}
            	else if( code >= 'a' && code <= 'f' ) {
            		codeB = ( code - 'a' ) + 10 ;
            	}
            	else {
            		throw new InputException(
            			"引数には、16進コード以外が含まれています(no:" +
            			(i+1) + " code:" + code + ")" ) ;
            	}            	
            	
                ret[ j ] =
	                ( byte )(
	                    ( ( ( codeA ) & 0x0000000f ) << 4 ) |
	                    ( ( codeB ) & 0x0000000f )
	                ) ;
                
            }
            
            return ret ;
            
        }catch( InputException in ) {
            throw in ;
        }catch( Exception t ){
            throw new InputException( t ) ;
        }finally{
            tmp = null ;
            buf = null ;
        }
    }
    
    /**
     * 変換された16進データを対象のバイナリに変換.
     * <BR><BR>
     * 変換された16進データを対象のバイナリ情報に変換します.
     * <BR>
     * @param code 16進数で構成された文字列を指定します.
     * @return byte[] 変換されたバイナリが返されます.
     * @exception InputException 入力例外.
     * @exception UnsupportException サポート外例外.
     */
    public static final byte[] convertString16ToBinary( String code )
        throws InputException,UnsupportException
    {
        
        int i,j ;
        int len ;
        
        char[] cnv = null ;
        byte[] ret = null ;
        int[] tmp = null ;
        
        try{
            
            len = code.length() ;
            
            if( len % 2 == 1 ){
                throw new InputException( "指定文字長は、奇数では対応されていません" ) ;
            }
            
            code.toUpperCase() ;
            
            len /= 2 ;
            ret = new byte[ len ] ;
            cnv = new char[ 2 ] ;
            tmp = new int[ 2 ] ;
            
            for( i = 0,j = 0 ; i < len ; i ++,j += 2 ){
                
                cnv[ 0 ] = code.charAt( j ) ;
                cnv[ 1 ] = code.charAt( j+1 ) ;
                
                tmp[ 0 ] = 
                    (
                        ( cnv[ 0 ] >= '0' && cnv[ 0 ] <= '9' ) ?
                            cnv[ 0 ] - '0' :
                            ( 
                                (
                                    cnv[ 0 ] >= 'A' && cnv[ 0 ] <= 'F'
                                ) ?
                                cnv[ 0 ] - 'A' + 10 :
                                (
                                    (
                                        ( cnv[ 0 ] >= 'a' && cnv[ 0 ] <= 'f' ) ?
                                            cnv[ 0 ] - 'a' + 10 : -1
                                    )
                                )
                            )
                    ) ;
                
                
                tmp[ 1 ] = 
                    (
                        ( cnv[ 1 ] >= '0' && cnv[ 1 ] <= '9' ) ?
                            cnv[ 1 ] - '0' :
                            ( 
                                ( cnv[ 1 ] >= 'A' && cnv[ 1 ] <= 'F' ) ?
                                cnv[ 1 ] - 'A' + 10 :
                                (
                                    (
                                        ( cnv[ 1 ] >= 'a' && cnv[ 1 ] <= 'f' ) ?
                                            cnv[ 1 ] - 'a' + 10 : -1
                                    )
                                )
                            )
                    ) ;
                
                if( tmp[ 0 ] == -1 || tmp[ 1 ] == -1 ){
                    
                    throw new UnsupportException(
                        "指定引数は、数値(16進数)以外の情報が含まれています"
                    ) ;
                    
                }
                
                ret[ i ] = ( byte )(
                    ( tmp[ 1 ] & 0x0000000f ) |
                    ( ( tmp[ 0 ] & 0x0000000f ) << 4 )
                ) ;
                
            }
            
        }catch( NullPointerException nul ){
            throw new InputException( nul ) ;
        }catch( UnsupportException ue ){
            throw ue ;
        }finally{
            
            cnv = null ;
            tmp = null ;
            
        }
        
        return ret ;
        
    }
    
    /**
     * 対象オブジェクト生成.
     * <BR><BR>
     * 指定したクラス名(パッケージ名+クラス名)に対するオブジェクトを生成します.
     * <BR>
     * @param name 生成対象のクラス名(パッケージ名+クラス名)を設定します.
     * @param loader 対象のクラスローダを設定します.<BR>
     *               デフォルトのローダを利用する場合[null]を設定します.
     * @return Object 生成されたオブジェクト情報が返されます.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final Object createObject( String name,ClassLoader loader )
        throws InputException,AccessException
    {
        
        Object ret = null ;
        String clsName = null ;
        
        if( name == null || name.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            
            clsName = UtilCom.trimPlus( name ) ;
            
            if( loader != null ){
                ret = Class.forName( clsName,true,loader ).newInstance() ;
            }
            else{
                ret = Class.forName( clsName ).newInstance() ;
            }
            
        }catch( NullPointerException nul ){
            throw new InputException(
                "対象のクラス名(" + name +
                ")は存在しません"
            ) ;
        }catch( IllegalAccessException il ){
            throw new AccessException( il ) ;
        }catch( InstantiationException  ie ){
            throw new AccessException( ie ) ;
        }catch( ExceptionInInitializerError ei ){
            throw new AccessException( ei ) ;
        }catch( LinkageError le ){
            throw new AccessException( le ) ;
        }catch( ClassNotFoundException cn ){
            throw new AccessException( cn ) ;
        }finally{
            clsName = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象ファイルにオブジェクトをシリアライズ.
     * <BR><BR>
     * 対象ファイルに対してオブジェクトをシリアライズ.
     * <BR>
     * @param name シリアライズ先のファイル名を設定します.
     * @param obj シリアライズ対象のオブジェクトを設定します.
     * @exception InputException 入力例外.
     * @exception AccessException アクセス例外.
     */
    public static final void putSerialize( String name,Serializable obj )
        throws InputException,AccessException
    {
        ObjectOutputStream o = null ;
        
        if( name == null || name.length() <= 0 || obj == null ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        try{
            o = new ObjectOutputStream(
                new FileOutputStream( name )
            ) ;
            o.writeObject( obj ) ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }finally{
            try{
                o.close() ;
            }catch( Exception ee ){
            }
            o = null ;
        }
    }
    
    /**
     * 対象ファイルから、オブジェクトを復元.
     * <BR><BR>
     * 対象のファイル名からオブジェクトを復元します.
     * <BR>
     * @param name 復元対象のファイル名を設定します.
     * @return Serializable 復元されたオブジェクトが返されます.<BR>
     *                      復元に失敗した場合[null]が返されます.
     */
    public static final Serializable getSerialize( String name )
    {
        ObjectInputStream in = null;
        Serializable ret = null ;
        
        if( name == null || name.length() <= 0 ){
            return ret ;
        }
        
        try{
            in = new ObjectInputStream(
                new FileInputStream( name )
            ) ;
            ret = ( Serializable )in.readObject() ;
        }catch( Exception e ){
            ret = null ;
        }finally{
            try{
                in.close() ;
            }catch( Exception e ){
            }
            in = null ;
        }
        
        return ret ;
    }
    
    
    /**
     * シリアライズオブジェクトをコピー.
     * <BR><BR>
     * 対象のシリアライズオブジェクトをコピーします.
     * <BR>
     * @param obj コピー元のオブジェクトを設定します.
     * @return Serializable コピーされたオブジェクト情報が返されます.<BR>
     *                      [null]が返されてた場合、オブジェクトのコピーに失敗しました.
     * @exception IOException IO例外.
     * @exception ClassNotFoundException クラス非存在例外.
     */
    public static Serializable copyObject( Serializable obj )
        throws IOException,ClassNotFoundException
    {
        byte[] tmp = null ;
        ByteArrayOutputStream bin = null ;
        ObjectInputStream in = null ;
        ObjectOutputStream out = null ;
        
        Object cp = null ;
        Serializable ret = null ;
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            out = new ObjectOutputStream( bin ) ;
            
            out.writeObject( obj ) ;
            out.close() ;
            out = null ;
            
            tmp = bin.toByteArray() ;
            
            in = new ObjectInputStream( new ByteArrayInputStream( tmp ) ) ;
            cp = in.readObject() ;
            
            in.close() ;
            in = null ;
            
            if( cp != null ){
                ret = ( Serializable )cp ;
            }
            else{
                ret = null ;
            }
            
        }catch( ClassNotFoundException cf ){
            ret = null ;
        }catch( IOException ie ){
            ret = null ;
        }finally{
            
            if( out != null ){
                try{
                    out.close() ;
                }catch( Exception e ){
                }
            }
            
            if( in != null ){
                try{
                    in.close() ;
                }catch( Exception e ){
                }
            }
            
            tmp = null ;
            bin = null ;
            in = null ;
            out = null ;
            cp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象シリアライズオブジェクトをバイナリに変換.
     * <BR><BR>
     * 対象シリアライズオブジェクトをバイナリに変換します.
     * <BR>
     * @param obj 対象のシリアライズオブジェクトを設定します.
     * @return byte[] 変換されたバイナリオブジェクトが返されます.<BR>
     *                [null]が返された場合バイナリ変換に失敗しました.
     */
    public static final byte[] convertObjectByBinary( Serializable obj )
    {
        ByteArrayOutputStream bin = null ;
        ObjectOutputStream out = null ;
        
        byte[] ret = null ;
        
        try{
            
            bin = new ByteArrayOutputStream() ;
            out = new ObjectOutputStream( bin ) ;
            
            out.writeObject( obj ) ;
            out.close() ;
            out = null ;
            
            ret = bin.toByteArray() ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            
            if( out != null ){
                try{
                    out.close() ;
                }catch( Exception e ){
                }
            }
            
            bin = null ;
            out = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象バイナリをシリアライズオブジェクトに変換.
     * <BR><BR>
     * 対象バイナリをシリアライズオブジェクトに変換します.
     * <BR>
     * @param bin 対象のバイナリオブジェクトを設定します.
     * @return Serializable 変換されたオブジェクト情報が返されます.<BR>
     *                      [null]が返された場合オブジェクト変換に失敗しました.
     */
    public static final Serializable convertBinaryByObject( byte[] bin )
    {
        ObjectInputStream in = null ;
        Object cp = null ;
        Serializable ret = null ;
        
        try{
            
            in = new ObjectInputStream( new ByteArrayInputStream( bin ) ) ;
            cp = in.readObject() ;
            
            in.close() ;
            in = null ;
            
            if( cp != null ){
                ret = ( Serializable )cp ;
            }
            else{
                ret = null ;
            }
            
        }catch( ClassNotFoundException cf ){
            ret = null ;
        }catch( IOException ie ){
            ret = null ;
        }finally{
            
            if( in != null ){
                try{
                    in.close() ;
                }catch( Exception e ){
                }
            }
            
            in = null ;
            cp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 対象クラス内から、ファイル情報を取得.
     * <BR><BR>
     * 指定した、クラス内から、ファイル情報を取得します.
     * <BR>
     * @param name 取得内容の、条件を設定します.
     * @return InputStream ファイル情報が返されます.
     * @exception AccessException アクセス例外.
     */
    public static final InputStream getClassToFile( String name )
        throws AccessException
    {
        InputStream ret = null ;
        
        try{
            ret = new UtilCom().getClasspathToFileObject( name ) ;
        }catch( Exception e ){
            throw new AccessException( e ) ;
        }
        
        return ret ;
    }
    
    /**
     * 変換処理.
     * <BR><BR>
     * 対象文字情報を条件の近いオブジェクトに変換します.<BR>
     * 対象文字情報が数値系の場合は[Long]に、小数点を含む場合は[Double]に
     * フラグ系の場合は[Boolean]に変換します.<BR>
     * また、上記以外の情報の場合[String]とします.
     * <BR>
     * @param value 変換対象の処理を設定します.
     * @return Object 変換された条件が返されます.
     */
    public static final Object convertObject( String value )
    {
        Object ret = null ;
        
        ret = UtilCom.convertNumber( value ) ;
        if( ret == null ){
            ret = UtilCom.convertFloat( value ) ;
            if( ret == null ){
                ret = UtilCom.convertBoolean( value ) ;
                if( ret == null ){
                    ret = value ;
                }
            }
        }
        
        return ret ;
    }
    
    /**
     * スペース情報を指定条件数分生成.
     * <BR><BR>
     * スペース情報を指定条件数分生成します.
     * <BR>
     * @param len スペースを生成する数を設定します.
     * @return String 指定数分のスペース文字列が返されます.
     */
    public static final String createSpace( int len )
    {
        int i ;
        StringBuffer buf = null ;
        
        if( len <= 0 ){
            return "" ;
        }
        
        buf = new StringBuffer() ;
        for( i = 0 ; i < len ; i ++ ){
            buf.append( " " ) ;
        }
        
        return buf.toString() ;
    }
    
    /**
     * 文字列の前後コーテーションを削除.
     * <BR><BR>
     * 文字列の前後コーテーションを削除します.
     * <BR>
     * @param str 変換対象の文字列を設定します.
     * @return String 変換された結果が返されます.
     */
    public static final String catCote( String str )
    {
        if( str == null || str.length() <= 0 ){
            if( str == null ){
                return null ;
            }
            else{
                return "" ;
            }
        }
        
        str = str.trim() ;
        
        return (
            ( str.startsWith( "\'" ) == true && str.endsWith( "\'" ) == true ) ||
            ( str.startsWith( "\"" ) == true && str.endsWith( "\"" ) == true )
        ) ? str.substring( 1,str.length()-1 ) : str ;
    }
    
    /**
     * バイナリを比較.
     * <BR><BR>
     * バイナリを比較します.
     * <BR>
     * @param src チェック元のバイナリを設定します.
     * @param dest チェック先のバイナリを設定します.
     * @exception AccessException アクセス例外.
     */
    public static final void checkBinary( byte[] src,byte[] dest )
        throws AccessException {
        
        int i ;
        int len ;
        
        if( src == null || dest == null ) {
            if( src == null ) {
                throw new AccessException( "比較元[src]のバイナリは設定されていません" ) ;
            }
            else {
                throw new AccessException( "比較先[dest]のバイナリは設定されていません" ) ;
            }
        }
        
        if( ( len = src.length ) != dest.length ) {
            throw new AccessException(
                "比較元[src:length(" + len +
                    ")]と比較先[dest:length(" + dest.length +
                    ")]は長さが違います" ) ;
        }
        
        for( i = 0 ; i < len ; i ++ ) {
            if( src[ i ] != dest[ i ] ) {
                throw new AccessException(
                "比較元[src:data(" + ( src[ i ] & 0x000000ff ) +
                    ")]と比較先[dest:data(" + ( dest[ i ] & 0x000000ff ) +
                    ")]は内容が違います(point:" + i + ")" ) ;
            }
        }
        
    }
    
    /**
     * 数値変換.
     */
    private static final Long convertNumber( String value )
    {
        int i ;
        long j ;
        int len ;
        long tmp ;
        char tmpObj ;
        
        Long ret = null ;
        String vTmp = null ;
        
        try{
            
            vTmp = UtilCom.trimPlus( value.toLowerCase() ) ;
            
            if( vTmp.indexOf( "0x" ) == 0 ){
                
                tmp = 0 ;
                len = vTmp.length() ;
                if( len > 0 && len <= 18 ){
                    
                    for( i = len-1,j = 0L ; i >= 2 ; i -- ){
                        
                        tmpObj = vTmp.charAt( i ) ;
                        
                        switch( tmpObj ){
                            case '0' : break ;
                            case '1' : tmp |= ( 1L << j ) ; break ;
                            case '2' : tmp |= ( 2L << j ) ; break ;
                            case '3' : tmp |= ( 3L << j ) ; break ;
                            case '4' : tmp |= ( 4L << j ) ; break ;
                            case '5' : tmp |= ( 5L << j ) ; break ;
                            case '6' : tmp |= ( 6L << j ) ; break ;
                            case '7' : tmp |= ( 7L << j ) ; break ;
                            case '8' : tmp |= ( 8L << j ) ; break ;
                            case '9' : tmp |= ( 9L << j ) ; break ;
                            case 'a' : tmp |= ( 10L << j ) ; break ;
                            case 'b' : tmp |= ( 11L << j ) ; break ;
                            case 'c' : tmp |= ( 12L << j ) ; break ;
                            case 'd' : tmp |= ( 13L << j ) ; break ;
                            case 'e' : tmp |= ( 14L << j ) ; break ;
                            case 'f' : tmp |= ( 15L << j ) ; break ;
                            default : throw new Exception() ;
                        }
                        
                        j = ( j == 0L ) ? 4L : ( j + 4L ) ;
                        
                    }
                    
                    ret = new Long( tmp ) ;
                    
                }
                else{
                    ret = null ;
                }
                
            }
            else{
                ret = new Long( value ) ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            vTmp = null ;
        }
        
        return ret ;
    }
    
    /**
     * 小数点変換.
     */
    private static final Double convertFloat( String value )
    {
        Double ret = null ;
        
        try{
            ret = new Double( value ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * BOOLEAN変換.
     */
    private static final Boolean convertBoolean( String value )
    {
        String vTmp = null ;
        Boolean ret = null ;
        
        try{
            
            vTmp = UtilCom.trimPlus( value.toLowerCase() ) ;
            
            if(
                vTmp.equals( "on" ) == true ||
                vTmp.equals( "true" ) == true
            )
            {
                ret = new Boolean( true ) ;
                
            }
            else if(
                vTmp.equals( "off" ) == true ||
                vTmp.equals( "false" ) == true
            )
            {
                ret = new Boolean( false ) ;
            }
            else{
                ret = null ;
            }
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            vTmp = null ;
        }
        
        return ret ;
        
    }
    
    
    
    /**
     * 指定されたクラス内のファイル名に対するファイルオブジェクトを取得.
     */
    private final InputStream getClasspathToFileObject( String name )
        throws Exception
    {
        return this.getClass().getClassLoader().getResourceAsStream( name ) ;
    }
    
}
