/*
 * @(#)ReadFormat.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.read;

import java.io.Serializable;

import org.maachang.commons.exception.InputException;
import org.maachang.commons.exception.ReadSyntaxException;
import org.maachang.commons.serialize.SerializeUtil;
import org.maachang.commons.util.CharTable;
import org.maachang.commons.util.SearchString;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;



/**
 * フォーマットオブジェクト.
 * <BR><BR>
 * 定義フォーマットを解析し、再構成します.
 *
 * @version 1.00, 2004/05/23
 * @author  Masahito Suzuki
 * @since  JRcCommons 1.00
 */
public class ReadFormat implements Serializable
{
    
    static {
        serialVersionUID = SerializeUtil.serialVersionUID(
            ReadFormat.class.getName()
        ) ;
    }
    
    /**
     * シリアライズUID.
     */
    private static final long serialVersionUID ;
    
    /**
     * 定義位置 - 開始.
     */
    private static final String START_CODE = "<$" ;
    
    /**
     * 定義位置 - 終了.
     */
    private static final String END_CODE = "$>" ;
    
    
    
    /**
     * フォーマット管理オブジェクト.
     */
    private ObjectArray m_man = null ;
    
    /**
     * 定義内容管理オブジェクト.
     */
    private CharTable m_table = null ;
    
    /**
     * 定義名一覧.
     */
    private ObjectArray m_defineNames = null ;
    
    
    /**
     * コンストラクタ.
     */
    public ReadFormat()
    {
        
    }
    
    /**
     * コンストラクタ.
     * <BR><BR>
     * 対象の情報を生成します.
     * <BR>
     * @param format 対象のフォーマット情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 解析失敗例外.
     */
    public ReadFormat( String format )
        throws InputException,ReadSyntaxException
    {
        this.create( format ) ;
    }
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    
    /**
     * 情報生成.
     * <BR><BR>
     * 対象の情報を生成します.
     * <BR>
     * @param format 対象のフォーマット情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 解析失敗例外.
     */
    public void create( String format )
        throws InputException,ReadSyntaxException
    {
        if( format == null || format.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            
            m_man = new ObjectArray() ;
            m_table = new CharTable() ;
            
            m_defineNames = ReadFormat.analysisFormat( format,m_man ) ;
            
        }catch( ReadSyntaxException rs ){
            this.clear() ;
            throw rs ;
        }catch( Exception e ){
            this.clear() ;
            throw new ReadSyntaxException( e ) ;
        }
    }
    
    /**
     * 情報クリア.
     * <BR><BR>
     * 情報をクリアします.
     */
    public void clear()
    {
        if( m_man != null ){
            m_man.clear() ;
        }
        if( m_table != null ){
            m_table.clear() ;
        }
        if( m_defineNames != null ){
            m_defineNames.clear() ;
        }
        
        m_man = null ;
        m_table = null ;
        m_defineNames = null ;
    }
    
    /**
     * 定義内容をクリア.
     * <BR><BR>
     * 設定されている定義内容をクリアします.
     */
    public void clearDefine()
    {
        if( m_table != null ){
            m_table.clear() ;
        }
    }
    
    /**
     * フォーマット情報を再セット.
     * <BR><BR>
     * フォーマット情報を再セットします.
     * <BR>
     * @param format 再設定対象のフォーマット情報を設定します.
     * @exception InputException 入力例外.
     * @exception ReadSyntaxException 解析失敗例外.
     */
    public void putFormat( String format )
        throws InputException,ReadSyntaxException
    {
        ObjectArray oldMan = null ;
        if( format == null || format.length() <= 0 ){
            throw new InputException( "引数は不正です" ) ;
        }
        else if( m_man == null ){
            this.create( format ) ;
        }
        else{
            
            try{
                
                oldMan = m_man ;
                m_man = new ObjectArray() ;
                m_defineNames = ReadFormat.analysisFormat( format,m_man ) ;
                
            }catch( ReadSyntaxException rs ){
                m_man = oldMan ;
                throw rs ;
            }catch( Exception e ){
                m_man = oldMan ;
                throw new ReadSyntaxException( e ) ;
            }finally{
                oldMan = null ;
            }
            
        }
    }
    
    /**
     * フォーマットと定義内容から、データ内容を取得.
     * <BR><BR>
     * フォーマットと定義内容から、データ内容を取得します.
     * <BR>
     * @return String 変換されたデータが返されます.
     */
    public String getData()
    {
        
        int i ;
        int len ;
        
        Object obj = null ;
        String val = null ;
        StringBuffer buf = null ;
        String ret = null ;
        
        try{
            
            buf = new StringBuffer() ;
            len = m_man.size() ;
            
            for( i = 0 ; i < len ; i ++ ){
                
                obj = m_man.get( i ) ;
                if( obj instanceof String ){
                    
                    buf.append( ( String )obj ) ;
                    
                }
                else{
                    
                    val = ( String )m_table.get( ( ( DefineName )obj ).getName() ) ;
                    
                    if( val != null && val.length() > 0 ){
                        
                        buf.append( val ) ;
                        
                    }
                    
                }
                
                obj = null ;
                val = null ;
                
            }
            
            ret = buf.toString() ;
            buf = null ;
            
        }catch( Exception e ){
            ret = null ;
        }finally{
            obj = null ;
            val = null ;
            buf = null ;
        }
        
        return ret ;
        
    }
    
    /**
     * 対象定義名に情報を追加.
     * <BR><BR>
     * 対象定義名に情報を追加します.
     * <BR>
     * @parma name 対象の定義名を設定します.
     * @param value 対象の定義内容を設定します.
     * @exception InputException 入力例外.
     */
    public void put( String name,String value )
        throws InputException
    {
        try{
            m_table.add( name,value ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * 対象定義名に設定されている情報を取得.
     * <BR><BR>
     * 対象定義名に設定されている情報を取得します.
     * <BR>
     * @param name 取得対象の定義名を取得します.
     * @return String 定義内容が返されます.
     */
    public String get( String name )
    {
        String ret = null ;
        
        try{
            ret = ( String )m_table.get( name ) ;
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 定義名一覧を取得.
     * <BR><BR>
     * 定義されている定義名一覧が返されます.
     * <BR>
     * @return String[] 定義名一覧が返されます.
     */
    public String[] getNames()
    {
        int len ;
        String[] ret = null ;
        
        try{
            
            len = m_defineNames.size() ;
            ret = new String[ len ] ;
            System.arraycopy( m_defineNames.getObjects(),0,ret,0,len ) ;
            
        }catch( Exception e ){
            ret = null ;
        }
        
        return ret ;
    }
    
    /**
     * 定義名数を取得.
     * <BR><BR>
     * 定義されている定義名数が返されます.
     * <BR>
     * @return int 定義されている定義名数が返されます.
     */
    public int size()
    {
        int ret ;
        
        try{
            ret = m_defineNames.size() ;
        }catch( Exception e ){
            ret = 0 ;
        }
        
        return ret ;
    }
    
    /**
     * 対象定義名が存在するかチェック.
     * <BR><BR>
     * 対象の定義名が存在するかチェックします.
     * <BR>
     * @param name チェック対象の定義名を設定します.
     * @return boolean チェック結果が返されます.<BR>
     *                 [true]が返された場合、存在します.<BR>
     *                 [false]が返された場合、存在しません.
     */
    public boolean isData( String name )
    {
        boolean ret ;
        
        try{
            ret = m_table.isData( name ) ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    /**
     * 生成チェック.
     * <BR><BR>
     * このオブジェクトが生成されているかチェックします.
     * <BR>
     * @return boolean 生成結果が返されます.<BR>
     *                 [true]が返された場合、生成されています.<BR>
     *                 [false]が返された場合、生成されていません.
     */
    public boolean isCreate()
    {
        boolean ret ;
        
        try{
            ret = ( m_man != null ) ? true : false ;
        }catch( Exception e ){
            ret = false ;
        }
        
        return ret ;
    }
    
    
    
    /**
     * フォーマット解析.
     */
    private static final ObjectArray analysisFormat( String format,ObjectArray outFormat )
        throws ReadSyntaxException,Exception
    {
        
        int before,now ;
        int flg ;
        
        String name = null ;
        ObjectArray ret = null ;
        SearchString ss = null ;
        
        ss = new SearchString() ;
        
        for( before = 0,now = 0,flg = 0 ;; ){
            
            if( flg == 0 ){
                
                before = now ;
                now = format.indexOf( START_CODE,before ) ;
                if( now <= -1 ){
                    outFormat.add( format.substring( before ) ) ;
                    break ;
                }
                else{
                    flg = 1 ;
                    if( before != now ){
                        outFormat.add( format.substring( before,now ) ) ;
                    }
                }
                
            }
            else{
                
                before = now ;
                now = format.indexOf( END_CODE,before ) ;
                if( now <= -1 ){
                    throw new ReadSyntaxException(
                        "フォーマットの終端[" + END_CODE +
                        "]が見つかりません"
                    ) ;
                }
                else{
                    flg = 0 ;
                    name = UtilCom.trimPlus( format.substring( before+START_CODE.length(),now ) ) ;
                    if( name != null && name.length() > 0 ){
                        outFormat.add( new DefineName( name ) ) ;
                        if( ret == null ){
                            ret = new ObjectArray() ;
                        }
                        if( ss.isData( name ) == false ){
                            ss.add( name ) ;
                            ret.add( name ) ;
                        }
                    }
                    name = null ;
                    now += END_CODE.length() ;
                }
                
            }
            
        }
        
        return ret ;
        
    }
    
}

/**
 * 定義名オブジェクト.
 */
class DefineName
{
    
    private String name = null ;
    
    protected DefineName(){}
    protected DefineName( String name )
    {
        this.name = name ;
    }
    
    protected final void finalize(){
        name = null ;
    }
    
    public final void setName( String name )
    {
        this.name = name ;
    }
    
    public final String getName()
    {
        return this.name ;
    }
    
}

