/*
 * @(#)ReadKey.java
 *
 * Copyright (c) 2005 masahito suzuki, Inc. All Rights Reserved
 */
package org.maachang.commons.util.read;

import org.maachang.commons.def.BaseDef;
import org.maachang.commons.exception.InputException;
import org.maachang.commons.util.ReadIndex;
import org.maachang.commons.util.UtilCom;
import org.maachang.commons.util.array.ObjectArray;


/**
 * キーワード情報の取得.
 * <BR><BR>
 * 指定された情報から、キーワード情報を取得します。
 *
 * @version 1.0.0 2003/10/28
 * @author  masahito suzuki
 * @since  JRcCommons 1.00
 */
public class ReadKey
{
    /**
     * コメント情報 : デフォルト.
     */
    private static final String DEF_COMMENT = "#" ;
    
    /**
     * 内容確保.
     */
    private final ObjectArray m_info = new ObjectArray() ;
    
    /**
     * 行数管理.
     */
    private final ObjectArray m_lineManage = new ObjectArray() ;
    
    /**
     * 列数管理.
     */
    private final ObjectArray m_sequenceManage = new ObjectArray() ;
    
    /**
     * ファイルオープンチェック.
     */
    private boolean m_isNew = false ;
    
    /**
     * コメント記述コード.
     */
    private String m_comment = ReadKey.DEF_COMMENT ;
    
    /**
     * コンストラクタ.
     */
    public ReadKey()
    {}
    
    /**
     * ファイナライズ処理定義.
     * <BR><BR>
     * ファイナライズ処理定義.
     * @exception Exception 例外処理が返されます.
     */
    protected final void finalize() throws Exception
    {
        
        try{
            this.clear() ;
        }catch( Exception t ){
        }
        
    }
    
    /**
     * キーワードを取得する情報を生成します。
     * <BR><BR>
     * キーワードを取得するための情報を生成します。
     * <BR>
     * @param info キーワードを取得するReadIndexオブジェクトを設定します。
     * @exception InputException 引数例外
     */
    public final void create( ReadIndex info )
        throws InputException
    {
        try{
            this.create( info,ReadKey.DEF_COMMENT ) ;
        }catch( InputException in ){
            throw in ;
        }
    }
    
    /**
     * キーワードを取得する情報を生成します。
     * <BR><BR>
     * キーワードを取得するための情報を生成します。
     * <BR>
     * @param info キーワードを取得するReadIndexオブジェクトを設定します。
     * @param comment コメントとなる条件の１ワード情報を設定します
     * @exception InputException 引数例外
     */
    public final void create( ReadIndex info,String comment )
        throws InputException
    {
        int i ;
        int len ;
        
        if(
            info == null ||
            ( len = info.size() ) <= 0 ||
            comment == null || 
            comment.length() <= 0 ||
            comment.length() > 1
        )
        {
            m_isNew = false ;
            throw new InputException( "引数が不正です" ) ;
        }
        
        this.clear() ;
        
        try{
            
            m_comment = comment.substring( 0,1 ) ;
            m_isNew = ReadKey.analysis(
                info,m_info,m_lineManage,
                m_sequenceManage,m_comment
            ) ;
            
        }catch( InputException in ){
            m_isNew = false ;
            throw in ;
        }
    }
    
    /**
     * キーワード取得情報をクリアーします。
     */
    public final void clear()
    {
        m_info.clear() ;
        m_lineManage.clear() ;
        m_sequenceManage.clear() ;
        m_isNew = false ;
    }
    
    /**
     * 取得したキーワードを順番で取得します。
     * <BR><BR>
     * 取得したキーワード情報を順番で取得します。
     * 取得前にファイルをオープンして情報を取得する必要があります。
     * <BR>
     * @param no 取得対象の項番を指定します。
     * @return String ファイルから取得したキーワード情報を取得します。<BR>
     *                [null]が返された場合、指定項番の情報は存在しません.
     */
    public final String get(int no)
    {
        String ret = null ;
        ObjectArray array = null ;
        
        array = m_info ;
        
        try{
            ret = (String)array.get(no) ;
        }catch(IndexOutOfBoundsException io){
            ret = null ;
        }finally{
            array = null ;
        }
        return ret ;
    }
    
    /**
     * 格納キーワードに対する行情報を取得.
     * <BR><BR>
     * 格納キーワードに対する行情報を取得します.
     * 取得方法は、取得対象キーワードと同様の指定方法を
     * 行います.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return int 対象キーワードに対する格納行位置が返されます.<BR>
     *             [-1]が返された場合、指定項番の情報は存在しません.
     */
    public final int getLine( int no )
    {
        int ret ;
        Integer info ;
        ObjectArray array = null ;
        
        array = m_lineManage ;
        
        try{
            info = ( Integer )array.get( no ) ;
        }catch(IndexOutOfBoundsException io){
            info = null ;
        }finally{
            array = null ;
        }
        
        if( info == null ){
            ret = -1 ;
        }else{
            ret = info.intValue() ;
        }
        
        info = null ;
        
        return ret ;
    }
    
    /**
     * 格納キーワードに対する列情報を取得.
     * <BR><BR>
     * 格納キーワードに対する列情報を取得します.
     * 取得方法は、取得対象キーワードと同様の指定方法を
     * 行います.
     * <BR>
     * @param no 取得対象の項番を指定します.
     * @return int 対象キーワードに対する格納列位置が返されます.<BR>
     *             [-1]が返された場合、指定項番の情報は存在しません.
     */
    public final int getSequence( int no )
    {
        int ret ;
        Integer info ;
        ObjectArray array = null ;
        
        array = m_sequenceManage ;
        
        try{
            info = ( Integer )array.get( no ) ;
        }catch(IndexOutOfBoundsException io){
            info = null ;
        }finally{
            array = null ;
        }
        
        if( info == null ){
            ret = -1 ;
        }else{
            ret = info.intValue() ;
        }
        
        info = null ;
        
        return ret ;
    }
    
    /**
     * 取得したキーワード数を取得します。
     * <BR><BR>
     * 取得したキーワード数を取得します。
     * <BR>
     * @return int 取得したキーワード数
     */
    public final int size()
    {
        return m_info.size() ;
    }
    
    /**
     * 生成チェック。
     * <BR><BR>
     * キーワード取得情報が生成されているかチェックします。
     * <BR>
     * @return boolean 生成チェック
     * 生成されている場合、引数はtrueが返されます。
     * 生成されていない場合、引数はfalseが返されます。
     */
    public final boolean isNew()
    {
        return m_isNew ;
    }
    
    /**
     * 要素群情報の取得.
     * <BR><BR>
     * 要素群情報の取得.
     * <BR>
     * @return ObjectArray 要素群情報が返されます.
     */
    public final ObjectArray getElement()
    {
        return m_info ;
    }
    
    
    
    /**
     * 情報解析.
     */
    private static final boolean analysis(
        ReadIndex info,ObjectArray manage,
        ObjectArray lineManage,
        ObjectArray sequenceManage,
        String comment
    )
        throws InputException
    {
        int i,j,line ;
        int s = -1 ;
        int len,len2 ;
        boolean scoteFlg ;
        boolean dcoteFlg ;
        
        String tmp = null ;
        String addData = null ;
        
        char chkChr ;
        
        char scoteChr ;
        char dcoteChr ;
        char commentChr ;
        char spaceChr ;
        char tabChr ;
        char equalChr ;
        
        String oneLine = null ;
        
        if(
            info == null || manage == null ||
            lineManage == null || sequenceManage == null
        ){
            throw new InputException( "引数が不正です" ) ;
        }
        
        scoteFlg = false ;
        dcoteFlg = false ;
        
        commentChr = comment.charAt( 0 ) ;
        scoteChr = BaseDef.SINGLE_COAT.charAt( 0 ) ;
        dcoteChr = BaseDef.DOUBLE_COAT.charAt( 0 ) ;
        spaceChr = BaseDef.SPACE.charAt( 0 ) ;
        tabChr = BaseDef.TAB.charAt( 0 ) ;
        equalChr = BaseDef.EQUAL.charAt( 0 ) ;
        
        
        try{
            
            len = info.size() ;
            
            for(i = 0 ; i < len ; i ++){
                
                oneLine = ( String )info.getIndex( i ) ;
                
                //  改行のみの情報の場合、読み飛ばす.
                // 全パターンの改行パターンでチェック.
                // 情報が存在しない場合も同様.
                if(
                    oneLine.length() == 0
                //  info[ i ].equalsIgnoreCase( BaseDef.ENTER_UNIX ) == true || 
                //  info[ i ].equalsIgnoreCase( BaseDef.ENTER_WINDOWS ) == true ||  
                //  info[ i ].equalsIgnoreCase( BaseDef.ENTER_MACINTOSH ) == true
                )
                {
                    continue ;
                }
                
                // コメント対応
                // 対象行に対してコメント情報が検知される場合.
                if( UtilCom.checkFront( oneLine ,comment ) == true ){
                    continue ;
                }
                
                tmp = oneLine ;
                len2 = tmp.length() ;
                
                // 指定行 i に対する列チェック.
                for( j = 0,line = 1,s = -1 ; j < len2 ; j ++,line ++ ){
                    
                    chkChr = tmp.charAt( j ) ;
                    
                    // 情報取得内でのコメント情報
                    if(
                        chkChr == commentChr &&
                        scoteFlg == false &&
                        dcoteFlg == false
                    )
                    {
                        
                        // コメント前に、チェック対象情報が存在.
                        if( s != -1 ){
                            addData = tmp.substring( s,j ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                        }
                        
                        s = -1 ;
                        break ;
                        
                    // シングルコーテーション開始対応.
                    }else if(
                        chkChr == scoteChr &&
                        scoteFlg == false &&
                        dcoteFlg == false
                    )
                    {
                        
                        // シングルコーテーション前に、チェック対象情報が存在.
                        if( s != -1 ){
                            addData = tmp.substring( s,j ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                        }
                        
                        s = j ;
                        scoteFlg = true ;
                        
                    // ダブルコーテーション開始対応.
                    }else if(
                        chkChr == dcoteChr &&
                        scoteFlg == false &&
                        dcoteFlg == false
                    )
                    {
                        
                        // ダブルコーテーション前に、チェック対象情報が存在.
                        if( s != -1 ){
                            addData = tmp.substring( s,j ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                        }
                        
                        s = j ;
                        dcoteFlg = true ;
                        
                    // シングルコーテーション終了.
                    }else if(   
                        chkChr == scoteChr &&
                        scoteFlg == true &&
                        dcoteFlg == false
                    )
                    {
                        
                        addData = tmp.substring( s+1,j ) ;
                        manage.add( addData ) ;
                        lineManage.add( new Integer( line ) ) ;
                        sequenceManage.add( new Integer( i ) ) ;
                        addData = null ;
                        s = -1 ;
                        
                        scoteFlg = false ;
                        
                    // ダブルコーテーション終了.
                    }else if(   
                        chkChr == dcoteChr &&
                        scoteFlg == false &&
                        dcoteFlg == true
                    )
                    {
                        
                        addData = tmp.substring( s+1,j ) ;
                        manage.add( addData ) ;
                        lineManage.add( new Integer( line ) ) ;
                        sequenceManage.add( new Integer( i ) ) ;
                        addData = null ;
                        s = -1 ;
                        
                        dcoteFlg = false ;
                        
                    // 他情報区切りを検知した場合.
                    }else if(   
                        (
                            chkChr == spaceChr ||
                            chkChr == tabChr ||
                            chkChr == equalChr
                        ) &&
                        scoteFlg == false &&
                        dcoteFlg == false
                    )
                    {
                        
                        // 情報が存在する場合.
                        if(s != -1 && j != s){
                            
                            addData = tmp.substring( s,j ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                            s = -1 ;
                            
                        }
                        
                        // [=]が存在した場合、セット.
                        if( chkChr == equalChr ){
                            
                            addData = "=" ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                            s = -1 ;
                            
                        }
                        
                    // 行情報ラストの場合.
                    }else if(j >= len2 - 1){
                        
                        // それまでに情報が存在する場合.
                        if(s != -1 && j != s){
                            
                            addData = tmp.substring( s,j+1 ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                            s = -1 ;
                            
                        // 最後の１文字情報が有効な場合、その情報をセット.
                        }else if(
                            chkChr != spaceChr ||
                            chkChr != tabChr ||
                            chkChr != equalChr ||
                            chkChr != commentChr
                        )
                        {
                            addData = tmp.substring( len2-1,len2 ) ;
                            manage.add( addData ) ;
                            lineManage.add( new Integer( line ) ) ;
                            sequenceManage.add( new Integer( i ) ) ;
                            addData = null ;
                                
                        }
                        
                    // 有効情報を検知.
                    }else if(s == -1){
                        
                        s = j ;
                        
                    }
                }
            }
            
            return true ;
            
        }catch( Exception t ){
            
            throw new InputException( t ) ;
            
        }finally{
            
            tmp = null ;
            addData = null ;
            oneLine = null ;
            
        }
    }
}
